<?php

use Symfony\Component\DomCrawler\Crawler;

class TestCase extends Illuminate\Foundation\Testing\TestCase
{
    /**
     * The base URL to use while testing the application.
     *
     * @var string
     */
    protected $baseUrl = 'http://reco.ma.orange.imi.local';

    /**
     * @var Faker\Generator
     */
    protected $faker;

    /**
     * Setup faker
     */
    public function setUp()
    {
        parent::setUp();
        $this->baseUrl = $this->app['config']->get('app.url');
        $this->faker = \Faker\Factory::create('de_DE');
    }

    /**
     * Creates the application.
     *
     * @return \Illuminate\Foundation\Application
     */
    public function createApplication()
    {
        $app = require __DIR__.'/../bootstrap/app.php';

        $app->make(Illuminate\Contracts\Console\Kernel::class)->bootstrap();

        return $app;
    }

    protected function assertStatus($status, $object)
    {
        $this->assertEquals($status, $object->status,
            sprintf('%s with ID %d should have status %s', get_class($object), $object->id, $status)
        );
    }

    public function registration($email, $password = 'dev-123', $payment = 'month')
    {
        $this->visit('/register');

        $this->type($this->faker->firstName, 'firstname');
        $this->type($this->faker->lastName, 'surname');
        $this->type($email, 'email');
        $this->type($password, 'password');
        $this->type($password, 'password_confirmation');

        $this->type($this->faker->company, 'company');
        $this->type('10178', 'zip');

        $this->press('Jetzt registrieren');

        $this->see('Registrierung erfolgreich');

    }

    /**
     * Create verified user with given role
     * $role = string seller/agent/consumer
     * $email = string email
     * $password = string password
     * $payment = string month/year
     *
     */
    public function registrationService($role, $email, $payment = 'month', $password = 'dev-123')
    {
        $data = [
            'salutation' => 'Herr',
            'firstname' => $this->faker->firstName,
            'surname' => $this->faker->lastName,
            'email' => $email,
            'mobile' => '0123456789',
            'phone' => '0123456789',
            'fax' => '0123456789',
            'password' => $password,
            'email_verified' => 1,

            'iban' => $this->faker->iban('DE', null, 20),
            'iban_owner' => $this->faker->firstname . ' ' . $this->faker->lastName,
            'company' => $this->faker->company,
            'street' => $this->faker->streetName,
            'housenumber' => $this->faker->randomNumber(),
            'zip' => '65343',
            'city' => $this->faker->city,
            'tax_id' => 'DE123456789',
            'payment' => $payment,
            'description' => 'Created by running the tests',
            'recruiter_user_id' => \App\User::where('email', '=', 'agent@reco.ma')->first()->id
        ];

        if($role == 'seller') {
            $data['all_connections'] = random_int(0, 1);
            $user = new \App\Services\SellerRegistrationService($data);
            $user->register(true);
            $user->getUser()->update(['status'=>\App\User::STATUS_ACTIVE]);
            $user->getAccount()->createMembershipInvoice();
            $user->applyInvitationBonus($user->getAccount());
        }
        if($role == 'agent'){
            $user = new \App\Services\AgentRegistrationService($data);
            $user->register();
        }
        if($role == 'consumer'){
            $user = new \App\Services\ConsumerRegistrationService($data);
            $user->register();
        }

    }

    //resolved in 5.3 -> https://github.com/laravel/framework/issues/10514#issuecomment-188334342
    protected function storeInput($element, $text)
    {
        $this->assertFilterProducesResults($element);
        $element = str_replace('#', '', $element);
        $element = str_replace("[]", "", $element);
        $this->inputs[$element] = $text;
        return $this;
    }

    protected function login($password, $email)
    {
        $this->visit('/login');

        $this->type($email, 'email');
        $this->type($password, 'password');
        $this->press('Anmelden');
    }

    protected function setUpMailMock()
    {
        $that = $this;
        Mail::shouldReceive('queue')
            ->andReturnUsing(function ($view, $view_params) use ($that) {
                $that->recentMailData = $view_params;
            });
    }

    protected function clearMail()
    {
        $this->recentMailData = null;
    }

    protected function seeMailSent()
    {
        $this->assertNotNull($this->recentMailData, 'Failed asserting that a mail was sent.');
    }

    /**
     *
     * Get target with $this->response->getTargetUrl()
     *
     * @param $method
     * @param $uri
     * @param array $parameters
     * @param array $cookies
     * @param array $files
     * @return $this
     *
     */
    protected function makeRequestWithoutFollowingRedirect($method, $uri, $parameters = [], $cookies = [], $files = [])
    {
        $uri = $this->prepareUrlForRequest($uri);

        $this->call($method, $uri, $parameters, $cookies, $files);

        $this->clearInputs();

        $this->currentUri = $this->app->make('request')->fullUrl();

        $this->crawler = new Crawler($this->response->getContent(), $this->currentUri);

        return $this;
    }
}
