<?php

use App\BankTransaction;
use App\Coupon;
use App\CreditNote;
use App\Invoice;
use App\Transaction;
use App\User;

class FullChainFrontendTest extends FrontendTestCase
{
    protected $password;
    protected $sellerEmail;
    protected $agentEmail;
    protected $setAgentPasswordUrl;
    protected $consumerEmail;
    protected $recentMailData;

    protected function setUpMailMock()
    {
        $that = $this;
        Mail::shouldReceive('queue')
            ->andReturnUsing(function ($view, $view_params) use ($that) {
                $that->recentMailData = $view_params;
                $that->mailHistory[] = $view_params;
            });
    }

    protected function clearMail()
    {
        $this->recentMailData = null;
    }

    protected function seeMailSent()
    {
        $this->assertNotNull($this->recentMailData, 'Failed asserting that a mail was sent.');
    }

    public function testInvalidIbanSignup()
    {
        $this->password = 'dev-123';
        $this->sellerEmail = $this->faker->email;
        $this->agentEmail = $this->faker->email;
        $this->consumerEmail = $this->faker->email;
        $this->testRegistration($this->password, $this->sellerEmail, 'DE17570928000217117501', true);
    }

    public function testChain()
    {
        $this->setUpMailMock();

        $this->password = 'dev-123';
        $this->sellerEmail = $this->faker->email;
        $this->agentEmail = $this->faker->email;
        $this->consumerEmail = $this->faker->email;

        $this->clearMail();
        $this->testRegistration($this->password, $this->sellerEmail);

        //Seller
        $this->login($this->password, $this->sellerEmail);
        $this->see('Ihre E-Mail-Adresse ist noch nicht verifiziert. Bitte prüfen Sie Ihr E-Mail Postfach.');

        // Link anklicken
	    $this->seeMailSent();
	    $this->assertArrayHasKey('url', $this->recentMailData);
	    $url = $this->recentMailData['url'];
	    $this->visit($url);

	    //seller Overlay
        $this->stepIntoSellerOverlay();

        $this->checkReceivedSellerStatus($this->sellerEmail);


//        $this->login($this->password, $this->sellerEmail); //FIXME login wegen opt-in-mail unnötig
        $this->see('Abmelden'); // check if log in worked

        $this->createOrChangeCouponTemplate();
        $this->inviteAgent($this->agentEmail);
        $this->visit('/logout');

        //Agent
        $this->setAgentPassword();
        $url = $this->recentMailData['url'];
        $this->visit($url);
        $this->skipAgentOverlay();

        $this->fillAgentProfile();
        $this->shareCoupon($this->consumerEmail);
        $this->visit('/logout');

        //Consumer
        $this->setConsumerPassword();

        foreach($this->mailHistory as $mail) {
            if ($mail['toUserArray']['email'] == $this->consumerEmail) {
                $url = $mail['url'];
            }
        }
        //Consumer must verify EmailAdress
        $this->visit($url);
        //Email verified. Consumer will automatically be logged in
        $this->fillConsumerProfile();
        $this->visit('/logout');

        //Seller
        $this->login($this->password, $this->sellerEmail);

        $this->createOrChangeCouponTemplate(true);
        $this->createOrChangeCouponTemplate(true, '/2');

        $this->redeemAndPayCoupon($this->consumerEmail);
        $this->visit('/logout');


        $this->requestPayment();

        $this->payout();

        //invite consumer as agent
        //Seller
        $this->login($this->password, $this->sellerEmail);

        $this->visit('/agent/create');
        $this->select('Herr', 'salutation');
        $this->type($this->faker->firstName, 'firstname');
        $this->type($this->faker->lastName, 'surname');
        $this->type($this->consumerEmail, 'email');
        $this->press('Einladen');

        $this->visit('/logout');

        //Consumer
        $this->login($this->password, $this->consumerEmail);

        $this->visit('/dashboard/agent');

        $this->visit('logout');

        //share coupon with agent
        //Agent - dev user
        $this->login('dev', 'agent@reco.ma');

        $this->shareCoupon($this->agentEmail);
        $this->visit('logout');

        //Agent
        $this->login($this->password, $this->agentEmail);

        $this->visit('/dashboard/agent');
        $this->visit('/logout');
    }

    public function testRegistration($password = 'dev-123', $email = null, $iban = 'DE17570928000217117500', $fail = false)
    {
        if ($email == null) {
            $email = $this->faker->email;
        }

        $this->visit('/register');

        $this->type($this->faker->firstName, 'firstname');
        $this->type($this->faker->lastName, 'surname');
        $this->type($company = $this->faker->company, 'company');

        $this->type($email, 'email');
        $this->type($password, 'password');
        $this->type($password, 'password_confirmation');

        $this->type('10178', 'zip');

        $this->press('Jetzt Registrieren');

        if (!$fail) {
            $this->see('Registrierung erfolgreich');
            $this->seeInDatabase('users',

                [
                    'email' => $email,
                ]
            );

            $this->seeInDatabase('accounts',
                [
                    'company' => $company,
                ]
            );
        }
    }

    protected function login($password, $email)
    {
        $this->visit('/login');

        $this->type($email, 'email');
        $this->type($password, 'password');
        $this->press('Anmelden');
    }

    /**
     * use true to check if mail was sent after agents were invited and/or template was edited
     * @param bool $afterAgentsInvited
     * @param string $urlSuffix
     */
    protected function createOrChangeCouponTemplate($afterAgentsInvited = false, $urlSuffix = '')
    {
        $this->visit('/coupon/edit' . $urlSuffix);

        $this->type($this->faker->sentence(3), 'title');
        $this->type($this->faker->dateTimeBetween('+31 years', '+33 years')->format('d.m.Y'), 'expiration_date');
        $this->type($this->faker->paragraph(rand(1, 10)), 'description');
        $this->type($this->faker->randomDigitNotNull, 'discount_consumer');
        $this->type($this->faker->randomDigit, 'provision_agent');
        $this->type($this->faker->randomDigitNotNull, 'quota');
        $this->type(3, 'validity');
        $this->clearMail();
        $this->press('Erstellen');

        $this->see('toastr[\'success\']');
        //mail will not be sent if no agents are present for seller
        if ($afterAgentsInvited) {
            $this->seeMailSent();
        }
        $this->visit('/coupon');
    }

    protected function inviteAgent($email)
    {
        $this->visit('/seller-invite-new-agent');
        $this->click('E-Mail');

        $this->select('Herr', 'salutation');
        $this->type($this->faker->firstName, 'firstname');
        $this->type($this->faker->lastName, 'surname');
        $this->type($email, 'email');

        $this->clearMail();
        $this->press('Empfehlen');
        $this->seeMailSent();
        $this->assertArrayHasKey('url', $this->recentMailData);
    }

    protected function setAgentPassword()
    {
        $this->visit($this->recentMailData['url']);

        $this->click('Jetzt registrieren');

        $this->type($this->faker->firstname, 'firstname');
        $this->type($this->faker->name, 'surname');
        $this->type($this->agentEmail, 'email');
        $this->type('65346', 'zip');

        $this->type($this->password, 'password');
        $this->type($this->password, 'password_confirmation');
        $this->press('Speichern');
    }

    protected function fillAgentProfile()
    {
        $this->visit('/user/profile');

        $this->type($this->faker->streetName, 'street');
        $this->type($this->faker->randomDigitNotNull, 'housenumber');
        $this->type($this->faker->city, 'city');

        $this->type('DE17570928000217117500', 'iban');
        $this->type($this->faker->name, 'iban_owner');
        $this->press('Speichern');
    }

    protected function shareCoupon($email)
    {
        $this->visit('/coupon');

        $seller = \App\User::whereEmail($this->sellerEmail)->first();
        $template = $seller->couponTemplates->first();

        $this->assertNotNull($template);

        $this->visit('share-coupon/' . $template->id);

        $this->click('E-Mail');

        $this->select('Herr', 'salutation');
        $this->type($this->faker->firstName, 'firstname');
        $this->type($this->faker->lastName, 'surname');
        $this->type($email, 'email');

        $this->clearMail();
        $this->press('Senden');
        $this->seeMailSent();
        $this->assertArrayHasKey('url', $this->recentMailData);
    }

    protected function setConsumerPassword()
    {
        $this->visit($this->recentMailData['url']);

        $this->click('Jetzt sparen');

        $this->type($this->faker->firstName, 'firstname'); // FIXME
        $this->type($this->faker->lastName, 'surname');    // FIXME Temporary until Consumer LP?
        $this->type($this->recentMailData['toUserArray']['email'], 'email');
        $this->type('10178', 'zip');
        $this->type($this->password, 'password');
        $this->type($this->password, 'password_confirmation');
        $this->press('Speichern');
    }

    protected function fillConsumerProfile()
    {
        $this->visit('/user/profile');

        $this->type($this->faker->streetName, 'street');
        $this->type($this->faker->randomDigitNotNull, 'housenumber');
        $this->type('10178', 'zip');
        $this->type('Berlin', 'city');


        $this->type('DE12500105170648489890', 'iban');
        $this->type($this->faker->name, 'iban_owner');
        $this->press('Speichern');
    }

    protected function redeemAndPayCoupon($consumerEmail)
    {
        $this->visit('coupon/redeem');

        $consumer = \App\User::where('email', '=', $consumerEmail)->first();
        $coupon = \App\Coupon::where('consumer_user_id', '=', $consumer->id)->whereStatus(Coupon::STATUS_ACTIVE)->first();

        $this->type($coupon->code, 'coupon_code');
        $value = $this->faker->randomDigitNotNull;
        $this->type($value, 'net_amount');
        $this->press('Einlösen');

        $coupon = $coupon->fresh();
        $this->assertEquals(Coupon::STATUS_REDEEMED, $coupon->status);
        /**
         * @var $transaction Transaction
         */
        foreach ($coupon->creditNoteTransactions as $transaction) {
            $this->assertNull($transaction->credit_note);
        }
        foreach ($coupon->paymentTransactions as $transaction) {
            $this->assertNull($transaction->invoice);
        }

        $this->visit(route('coupon-paid', $coupon->id));
        $this->assertResponseOk();

        $coupon = $coupon->fresh();
        $this->assertEquals(Coupon::STATUS_PAID, $coupon->status);

        foreach ($coupon->creditNoteTransactions as $transaction) {
            $transaction->fresh();
            $this->assertNull($transaction->credit_note);
        }
        foreach ($coupon->paymentTransactions as $transaction) {
            $transaction->fresh();
            $this->assertNotNull($transaction->invoice);
        }

        $this->createAndProcessInvoicePayment($coupon);
    }

    protected function createAndProcessInvoicePayment(Coupon $coupon)
    {

        $file = Invoice::generateSepaDebitXml();
        $this->assertFileExists($file);

        BankTransaction::createFakeFromReceipt($coupon->invoice);

        foreach ($coupon->paymentTransactions as $transaction) {
            $transaction = $transaction->fresh();
            $this->assertStatus(Transaction::STATUS_PAID, $transaction);
        }

	    // simulate 10 days delay
	    Transaction::releaseAllEligible(true);

        // agent can get money
	    $transaction = $coupon->getTransactionByType(Transaction::TYPE_AGENT_COUPON_PROVISION_CREDIT);
	    $transaction = $transaction->fresh();
	    $this->assertStatus(Transaction::STATUS_ALLOCATED, $transaction);


	    // consumer can't
	    $transaction = $coupon->getTransactionByType(Transaction::TYPE_CONSUMER_COUPON_DISCOUNT_CREDIT);
	    $transaction = $transaction->fresh();
	    $this->assertStatus(Transaction::STATUS_ALLOCATED, $transaction);


	    //Rate Coupon to set coupon status done
	    $this->visit('logout');
	    $this->login($this->password, $this->consumerEmail);
	    $this->visit('/transaction');
	    $this->press('Bewerten');
	    $ratedTransaction = $coupon->creditNoteTransactions()->where('type', Transaction::TYPE_CONSUMER_COUPON_DISCOUNT_CREDIT)->first();
	    $this->inputs['transaction_id'] = $ratedTransaction->id;
	    $this->type('Beschreibung', 'description');
	    $inputs = $this->inputs;
	    $this->press('Senden');

	    // simulate 10 days delay - FIXME: not really necassary here, because the 10 days might already have passed
	    Transaction::releaseAllEligible(true);

        $transaction = $coupon->getTransactionByType(Transaction::TYPE_CONSUMER_COUPON_DISCOUNT_CREDIT);
        $transaction = $transaction->fresh();
        $this->assertStatus(Transaction::STATUS_ALLOCATED, $transaction);
    }

    protected function requestPayment()
    {
        // Consumer requests
        $this->login($this->password, $this->consumerEmail);

        $creditNoteCountBefore = User::findByEmail($this->consumerEmail)->transactions()->whereNull('credit_note_id')->count();
        $this->assertGreaterThan(0, $creditNoteCountBefore, 'Failed asserting that there any are transactions without a credit_note');

        $this->visit('/transaction');
        $this->see('Bitte bestätigen Sie die Auszahlung');
        $this->visit('/transaction/request_all_payments');

        $creditNoteCountAfter = User::findByEmail($this->consumerEmail)->transactions()->whereNull('credit_note_id')->count();
        $this->assertLessThan($creditNoteCountBefore, $creditNoteCountAfter);

        $this->visit('/logout');

        // Agent requests
        $this->login($this->password, $this->agentEmail);

        $creditNoteCountBefore = User::findByEmail($this->agentEmail)->transactions()->whereNull('credit_note_id')->count();
        $this->assertGreaterThan(0, $creditNoteCountBefore, 'Failed asserting that there any are transactions without a credit_note');

        $this->visit('/transaction');
        $this->see('Bitte bestätigen Sie die Auszahlung');
        $this->visit('/transaction/request_all_payments');

        $creditNoteCountAfter = User::findByEmail($this->agentEmail)->transactions()->whereNull('credit_note_id')->count();
        $this->assertLessThan($creditNoteCountBefore, $creditNoteCountAfter);

        $this->visit('/logout');
    }

    protected function payout()
    {
        $file = CreditNote::generateSepaCreditXml();
        $this->assertFileExists($file);
    }
}