<?php

namespace App;

use Illuminate\Database\Eloquent\Model;
use App\Presenters\UsersGiftPresenter;
use App\Presenters\PresentsData;

class UsersGift extends Model
{
    use PresentsData;

    protected $presenter = UsersGiftPresenter::class;

    /**
     * Gift is just created (coupon redeem).
     */
    const STATUS_NEW      = 'new';

    /**
     * Gift is paid by seller (coupon done).
     */
    const STATUS_PAID     = 'paid';

    /**
     * Gift is ready to be sent (user confirmed address).
     */
    const STATUS_READY    = 'ready';

    /**
     * Gift is sent by operator.
     */
    const STATUS_SENT     = 'sent';

    /**
     * Gift is received by user.
     */
    const STATUS_RECEIVED = 'received';

    public function gift()
    {
        return $this->hasOne('\App\Gift', 'id', 'gift_id');
    }

    public function user()
    {
        return $this->hasOne('\App\User', 'id', 'user_id');
    }

    public function coupon()
    {
        return $this->hasOne('\App\Coupon', 'id', 'coupon_id');
    }

    /**
     * Check whether users gift has status STATUS_NEW.
     *
     * @return bool
     */
    public function isNew()
    {
        return self::STATUS_NEW === $this->status;
    }

    /**
     * Check whether users gift has status STATUS_PAID.
     *
     * @return bool
     */
    public function isPaid()
    {
        return self::STATUS_PAID === $this->status;
    }

    /**
     * Check whether users gift has status STATUS_READY.
     *
     * @return bool
     */
    public function isReady()
    {
        return self::STATUS_READY === $this->status;
    }

    /**
     * Check whether users gift has status STATUS_SENT.
     *
     * @return bool
     */
    public function isSent()
    {
        return self::STATUS_SENT === $this->status;
    }

    /**
     * Check whether users gift has status STATUS_RECEIVED.
     *
     * @return bool
     */
    public function isReceived()
    {
        return self::STATUS_RECEIVED === $this->status;
    }

    /**
     * Check whether owner can edit data.
     *
     * @return bool
     */
    public function isEditableByOwner()
    {
        return $this->isPaid() || $this->isReady();
    }
}
