<?php

namespace App;

use App\Presenters\PresentsData;
use App\Presenters\UserPresenter;
use App\Services\ReferralLinkService;
use Carbon\Carbon;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Mail\Message;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\View;
use Illuminate\Support\HtmlString;
use Mockery\CountValidator\Exception;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Traits\HasRoles;
use willvincent\Rateable\Rateable;

use App\Library\Billomat\BillomatClientHelper;

class User extends Authenticatable
{
    use HasRoles {
        assignRole as traitAssignRole;
        removeRole as traitRemoveRole;
    }

    use Rateable;

    use HasStatus;

    use HasCounters;

    use PresentsData;

    protected $presenter = UserPresenter::class;

    public function usersGifts()
    {
        return $this->hasMany('\App\UsersGift', 'user_id', 'id');
    }

    public function options()
    {
        return $this->hasMany('\App\Options', 'user_id', 'id');
    }

    public function favorites()
    {
        return $this->belongsToMany('\App\CouponTemplate', 'favorites', 'user_id', 'deal_id')->withTimeStamps();
    }

    public function places() // sellers only can have places
    {
        return $this->hasMany('\App\Place', 'seller_id', 'id')
            ->where('status', '<>', Place::STATUS_DELETED);
    }

    public function redeems()
    {
        return $this->hasMany('\App\Coupon', 'redeemer_id', 'id');
    }

    /**
     * Relation with activity log.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function activity()
    {
        return $this->hasMany('\App\Activity');
    }

    /**
     * Relation with employee's boss user.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function boss()
    {
        return $this->belongsTo('\App\User', 'boss_id', 'id');
    }

    /**
     * Relation with sessions.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function sessions()
    {
        return $this->hasMany('\App\Session');
    }

    /**
     * Relation with devices.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function devices()
    {
        return $this->hasMany('\App\TrustedDevice');
    }

    /**
     * Coupons, belongs to seller.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasManyThrough
     */
    public function publishedCoupons()
    {
        return $this->hasManyThrough('\App\Coupon', '\App\CouponTemplate', 'user_id', 'coupon_template_id');
    }

    protected static function boot()
    {
        parent::boot();
        //static::addGlobalScope(new UserStatusScope);
    }

    public function assignRole($role)
    {
        if (!$this->hasRole($role)) {
            $this->traitAssignRole($role);
            if ($this->completedProfile()) {
                $this->account->syncReceivedSellerRoleAt();
            }
        }
    }

    public function removeRole($role)
    {
        $this->traitRemoveRole($role);
        if ($this->account) {
            $this->account->syncReceivedSellerRoleAt();
        }
    }

    const STATUS_COLORS = [
        self::STATUS_INACTIVE => 'default',
        self::STATUS_DISABLED => 'warning',
        self::STATUS_PENDING => 'error',
        self::STATUS_ACTIVE => 'success',
        self::STATUS_DELETED => 'default',
    ];

    const STATUS_INACTIVE = 'inactive';
    const STATUS_DISABLED = 'disabled';
    const STATUS_PENDING = 'pending';
    const STATUS_ACTIVE = 'active';
    const STATUS_DELETED = 'removed';

    const ACCESS_DENIED = 'access_denied';
    const REGISTER_BY_EMAIL = 'register-by-email';
    const REGISTER_BY_FACEBOOK = 'register-by-facebook';

    const STATUS_GROUP_ALL = [
        self::STATUS_PENDING,
        self::STATUS_ACTIVE,
        self::STATUS_INACTIVE,
        self::STATUS_DISABLED,
        self::STATUS_DELETED,
    ];

    // referral statuses
    const REFERRAL_STATUS_NEW          = 'new'; // just registered
    const REFERRAL_STATUS_ACTIVE       = 'active'; // has been set as active
    const REFERRAL_STATUS_HAS_COUPONS  = 'has_coupons'; // has any coupons
    const REFERRAL_STATUS_HAS_REDEEMED = 'has_redeem'; // has redeemed coupons
    const REFERRAL_STATUS_PAID         = 'paid'; // got money; finished

    /**
     * Which fields are relevant for Billomat
     *
     * @see makeBillomatArray
     */
    const BILLOMAT_FIELDS = [
        "email",
        "phone",
        "salutation",
        "last_name",
        "first_name",
    ];

    /**
     * A seller offers products
     */
    const SELLER = 'seller';

    /**
     * An agent promotes products products
     */
    const AGENT = 'agent';

    /**
     * A customer gets a coupon and can redeem them
     */
    const CONSUMER = 'consumer';

    /**
     * An operator can do everything
     */
    const OPERATOR = 'operator';

    /**
     * An employee of seller.
     */
    const EMPLOYEE = 'employee';

    /**
     * An Fieldservice employee (ADM) sells reco.ma service to sellers
     */
    const FIELDSERVICE_EMPLOYEE = 'fieldservice_employee';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'account_id',
        'firstname',
        'surname',
        'email',
        'mobile',
        'phone',
        'fax',
        'status',
        'password',
        'salutation',
        'mobile_verification_token',
        'mobile_verified',
        'upload_portrait',
        'email_verification_token',
        'last_visited_consumer_dashboard',
        'email_verified',
        'alias',
        'referred_by',
        'invited_by',
        'referral_status',
        'newsletter',
        'boss_id',
        'tfa_code',
        'tfa_code_expires_at',
        'google_tfa_secret',
    ];

    protected $hidden = [
        'password',
        'remember_token',
        'google_tfa_secret',
    ];

    protected $appends = [
        'fullname',
        'userpic',
    ];

    protected $dates = [
        'updated_at',
        'created_at',
        'tfa_code_expires_at',
    ];

    /**
     * List of all available permissions.
     */
    const EMPLOYEE_PERMISSIONS = [
        'view deals',
        'create deals',
        'edit deals',
        'view coupons',
        'view agents',
        'redeem coupons',
        'share reflink',
        'invite sellers',
        'view transactions',
        'view reviews',
        'view company data',
        'edit company data',
        'view places',
        'create places',
        'edit places',
        'view widgets',
        'edit widgets',
    ];

    /**
     * Attributes to generate on array calls
     * @var array
     */
    protected $virtual_attributes = ['displayName', 'displayNameWithCompany'];

    public function toArray($withVirtual = false)
    {
        $array = parent::toArray(); // TODO: Change the autogenerated stub
        if (!$withVirtual) {
            return $array;
        }
        foreach ($this->virtual_attributes as $key) {
            $functionName = 'get' . ucfirst($key);
            $array[$key] = $this->$functionName();
        }

        $collection = $this->roles()->implode('name')->get()->toArray();
        $roles = [];
        foreach ($collection as $role) {
            array_push($roles, $role['name']);
        }
        $array['roles'] = $roles;
        return $array;
    }

    public function creditNotes()
    {
        return $this->hasMany(CreditNote::class);
    }

    public static function findByEmail($email)
    {
        return self::where('email', '=', $email)->first();
    }

    public static function findStandardTestUser($type)
    {
        return self::findByEmail($type . '@reco.ma');
    }

    public function getDisplayName()
    {
        $name = trim($this->firstname ?? '' . ' ' . $this->surname ?? '');
        if (empty($name)) {
            $name = null;
        }
        return $name;
    }

    public function getCompany()
    {
        return $this->account->company;
    }

    public function getDisplayNameWithCompany()
    {
        if (!$this->account) {
            return $this->getDisplayName();
        }
        return $this->getDisplayName() . ' - ' . $this->getCompany();
    }

    public static function getDisplayNameDBExpression($as = 'display_name', $table = 'users')
    {
        return DB::raw('CONCAT(' . $table . '.firstname, " ", ' . $table . '.surname)' . ($as ? ' as ' . $as : ''));
    }

    public function getPendingCouponCount()
    {
        return count($this->coupons()->where('status', '=', Coupon::STATUS_PENDING)->get());
    }

    public function coupons()
    {
        return $this->hasMany('\App\Coupon', 'consumer_user_id');
    }

    /**
     * Get number of interaction requests for the current user
     *
     * @return int
     */
    public function getNotificationCount()
    {
        $count = $this->account->getPendingInvitationCount() + $this->account->getPendingCouponCount();

        return $count;
    }

    public function countUnseenCoupons()
    {
        $last_visited_consumer_dashboard = $this->last_visited_consumer_dashboard;
        if ($last_visited_consumer_dashboard === NULL) {
            $count = $this->coupons()->count();
        } else {
            $count = $this->coupons()->where('updated_at', '>', $last_visited_consumer_dashboard)->count();
        }
        return $count;
    }

    public static function getPendingUserCount()
    {
        return User::where('status', '=', 'pending')->count();
    }

    public function relatedUsers()
    {
        $relatedAccountIds = $this->account->relatedAccounts()->pluck('id');
        return User::whereIn('account_id', $relatedAccountIds);
    }

    public function relatedFromUsers()
    {
        $relatedFromAccountIds = $this->account->relatedFromAccounts()->pluck('id');
        return User::whereIn('account_id', $relatedFromAccountIds);
    }

    /**
     * Connected sellers
     *
     * @return Array
     */
    public function getSellers()
    {
        $sellerAccounts = $this->account->relatedSellerAccounts;
        $sellers = new Collection();
        foreach ($sellerAccounts as $sellerAccount) {
            $sellers = $sellerAccount->users->union($sellers);
        }
        return $sellers;
    }

    /**
     * All available sellers
     *
     * Connected sellers plus those including those who allow "selling by all" (all_connections)
     *
     * @return array
     */
    public function availableSellersQuery()
    {
        $sellerAccountIds = $this->account->availableSellerAccountsQuery()->pluck('accounts.id');
        $usersQuery = User::query()->whereIn('account_id', $sellerAccountIds->toArray());
        return $usersQuery;
    }

    public function availableCouponsQuery($withSellerForAll = true, $excludeExpired = true)
    {

        $couponTemplatesFromSellersWithCompleteProfile = CouponTemplate::get()->filter(function($value){
            if($value->seller && $value->seller->completedProfile()){
                return true;
            }
            return false;
        })->pluck('id');

        $query = CouponTemplate::query()
            ->where(function ($query) use ($withSellerForAll, $excludeExpired) {
                $query->whereIn('user_id', $this->availableSellersQuery()->pluck('users.id')->toArray());
                if ($excludeExpired) {
                    $query->where('expiration_date', '>=', Carbon::today());
                }
                if ($withSellerForAll) {
                    $query->orWhere(function ($query) {
                        $query
                            ->where('accounts.status','=','active')
                            ->where('all_connections', '=', '1')
                            ->where('expiration_date', '>=', Carbon::today());
                    });
                }
            })->whereIn('coupon_templates.id',$couponTemplatesFromSellersWithCompleteProfile);
        return $query;
    }

    public static function findByToken($token)
    {
        $data = DB::table('password_resets')->where('token', $token)->first();

        if ($data == null) {
            die('Ungültiges Token'); //FIXME: Real error page
        }

        $email = $data->email;
        return self::findByEmail($email);
    }

    public function getAvailableSellers()
    {
        return $this->availableSellersQuery()->get();
    }

    /**
     * Associated agents for a seller
     *
     * @return mixed
     */
    public function getAgentsQuery()
    {
        /*return $this->account
            ->relatedAgentAccounts()
            ->belongsTo('\App\User');
            //->join('users', 'users.id', '=', 'account_connections.id')->with('App\User');*/
        $ids = $this->account
            ->relatedAgentAccounts()->lists('accounts.id');

        $usersQuery = User::query()->whereIn('account_id', $ids);
        return $usersQuery;

    }

    public function account()
    {
        return $this->belongsTo('\App\Account');
    }

    /**
     * get all coupons shared for this seller(user)
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasManyThrough
     */
    public function couponsSharedFromThisSeller()
    {
        return $this->hasManyThrough('App\Coupon', 'App\CouponTemplate');
    }

    public function couponTemplates()
    {
        return $this->hasMany('App\CouponTemplate', 'user_id', 'id');
    }

    public function deals()
    {
        return $this->hasMany('App\CouponTemplate', 'user_id', 'id')
            ->where('status', '<>', CouponTemplate::STATUS_DRAFT);
    }

    /**
     * Relation with referrals (users who has been invited by current user)
     *
     * @return mixed
     */
    public function referrals()
    {
        return $this->hasMany('App\User', 'referred_by')
            ->withoutGlobalScopes();
    }

    /**
     * Relation with invited sellers (sellers who has been invited by current user)
     *
     * @return mixed
     */
    public function invitedSellers()
    {
        return $this->hasMany('App\User', 'invited_by')
            ->withoutGlobalScopes();
    }

    /**
     * Relation with user who invited current user.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function affiliate()
    {
        return $this->hasOne('App\User', 'referred_by');
    }

    /**
     * Relation with user who invited current seller.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function inviter()
    {
        return $this->hasOne('App\User', 'invited_by');
    }

    /**
     * Relation with user's employees.
     *
     * @return mixed
     */
    public function employees()
    {
        return $this->hasMany('App\User', 'boss_id')
            ->where('status', '<>', User::STATUS_DELETED)
            ->withoutGlobalScopes();
    }

    public function roleSeller()
    {
        return $this->hasRole('seller');
    }

    /**
     * Creates a billomat client from this user
     */
    public function transferToBillomat()
    {

        if ($this->hasBillomatAccount()) {
            //Client allready has an accoutn, no need to create
            return false;
        }

        $billomatClientInstance = BillomatClientHelper::getBillomatClientInstance();

        //http://www.billomat.com/en/api/clients -> Create Client
        $ret = $billomatClientInstance->createClient($this->makeBillomatArray());

        $this->billomat_id = $ret["client"]["id"];
        $this->save();

        return $ret;
    }

    /**
     * Update in Billomat, but only if already transfered
     *
     * @return array|boolean
     */
    public function updateInBillomat()
    {

        if (!$this->hasBillomatAccount()) {
            return;
        }

        $update = $this->makeBillomatArray();
        $update['id'] = intval($this->billomat_id);

        $billomatClientInstance = BillomatClientHelper::getBillomatClientInstance();

        try {
            $billomatClientInstance->updateClient($update);
        } catch (\Phobetor\Billomat\Exception\NotFoundException $e) {
            $this->transferToBillomat();
        }
    }

    private function makeBillomatArray()
    {

        $account = $this->account;

        $ret = [
            "client" => [
                "street" => $account->street . " " . $account->housenumber,
                "zip" => $account->zip,
                "city" => $account->city,

                //"state"         => $this->state,
                "name"=> $this->getCompany(),
                "first_name" => $this->firstname,
                "last_name" => $this->surname,
                "salutation" => $this->salutation,
                "phone" => $this->phone,

                "email" => $this->email,

                "tax_number" => $account->tax_id,

                "bank_account_owner" => $account->iban_owner,
                "bank_iban" => $account->iban,
            ]
        ];

        return $ret;
    }

    /**
     * Check if billomat account exists and is valid
     * @return boolean
     */
    public function hasBillomatAccount()
    {
        if ($this->billomat_id !== NULL) {
            $billomatClientInstance = BillomatClientHelper::getBillomatClientInstance();

            try {
                $billomatClientInstance->getClient(["id" => intval($this->billomat_id)]);
            } catch (\Phobetor\Billomat\Exception\NotFoundException $e) {
                $this->billomat_id = NULL;
                $this->save();
                return false;
            } catch (\Phobetor\Billomat\Exception\ExceptionInterface $e) {
                $this->billomat_id = NULL;
                $this->save();
                Log::warning($e, ['user_id'=>$this->id]);
            }
        } else {
            return false;
        }
    }

    /**
     * Retrieves the billomat instance of this user and creates one if no billomat
     * representation of this user exists
     */
    public function getBillomat()
    {

        //TODO sollte der wirklich erstellt werden wenn er nicht existiert,
        //TODO evtl ist das zu viel logik oder zu redundant
        if ($this->hasBillomatAccount()) {
            //Client allready has an accoutn, no need to create
            return $this->transferToBillomat();
        }

        $billomatClientInstance = BillomatClientHelper::getBillomatClientInstance();

        try {
            $billomatClientInstance->getClient(["id" => $this->billomat_id]);
        } catch (\Phobetor\Billomat\Exception\NotFoundException $e) {
            return false;
        } //TODO weitere exceptions, evtl globalen exception handler oder so

    }

    public function transactions()
    {
        return $this->hasMany('App\Transaction', 'user_id', 'id');
    }

    public function getRateableTransactionsCount()
    {
        return count($this->transactions()->where('status', '=', 'rate')->get());
    }

    public function couponsByAgent()
    {
        return $this->hasMany('App\Coupon', 'agent_user_id', 'id');
    }

    /**
     * Get all users (unique) that have a coupon of this seller
     * @return Collection
     * @throws \Exception
     */
    public function getUsersWithCoupons()
    {
        if ($this->hasRole('seller')) {
            throw new \Exception('Must be seller');
        }
        $usersWithCoupons = new Collection();
        foreach ($this->couponTemplates as $couponTemplate) {
            $coupons = $couponTemplate->coupons;
            $usersWithCoupons = $usersWithCoupons->union($coupons->pluck('consumer')->keyBy('id'));
        }
        return $usersWithCoupons;
    }

    /*
     * get all users where password is not set
     */
    public static function allOfRole($role)
    {
        return Role::findByName($role)->users()->where('password', '!=', '');
    }

    public static function allSellers()
    {
        return self::allOfRole('seller');
    }

    public static function allAgents()
    {
        return self::allOfRole('agent');
    }

    public static function allConsumers()
    {
        return self::allOfRole('consumer');
    }

    public function getSharedCoupons()
    {
        return Coupon::where('agent_user_id', '=', $this->id)->get();

    }

    /**
     * Check if the user has to fill the provide now
     *
     * @param $inPaymentStep Whether the user is just about to request the payment
     * @return bool
     */
    public function checkHasToFillTheProfile($isInPaymentStep = false)
    {
        if ($isInPaymentStep) {
            return !$this->completedProfile();
        } else {
            return ($this->hasRole('seller') && !$this->completedProfile());
        }
    }

    public function completedProfile()
    {
        $account = $this->account()->first();
        if (isset($this->salutation, $this->firstname, $this->surname, $this->email, $this->password) && $this->firstname != ' ' && $this->surname != ' ') {
            if ($this->hasRole('agent') || $this->hasRole('consumer')) {
                if (!(empty($account->street) || empty($account->housenumber) || empty($account->zip)
                    || empty($account->city) || empty($account->iban) || empty($account->iban_owner))
                ) {
                    return true;
                }
            }

            if ($this->hasRole('seller')) {
                //if ($account &&  $account->users->first()->phone!='' && isset($account->company, $account->street, $account->housenumber, $account->zip, $account->city, $account->payment, $account->iban, $account->iban_owner)) {
                    return true;
                //}
            }

            if ($this->hasRole('operator')) {
                if (isset($account->iban, $account->iban_owner)) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Replace simple and clean paragraphs by stuff that most mail clients accept render properly.
     *
     * @param $content
     */
    public function renderParagraphs($content)
    {
        return preg_replace_callback('/<p>(.*)<\/p>/Uumsi', function ($text) {
            $view = View::make('emails.html_template.paragraph')->with(['text' => $text[1]]);
            return $view->render();
        }, $content);
    }

    /**
     * Send a mail to an user
     *
     * @param $template Template without emails. prefix
     * @param $fromSystemOrUser false = use auth user, true = systemsending address, App\User = that specified user
     * @param $data array
     */
    public function sendMail($template, $data, $fromSystemOrUser = false, $attachment = false)
    {
        if (isset($data['url'])) {
            $data['url'] = Shortlink::shorten($data['url']);
        }
        $template = 'emails.' . $template;
        if ($fromSystemOrUser instanceof User) {
            $fromUser = $fromSystemOrUser;
        } else {
            $fromUser = Auth::user();
        }
        if ($fromSystemOrUser === true) {
            $fromUser = null;
            $sendingName = 'RECO.MA GmbH';
            $sendingAddress = config('recoma.no-replyAdress');
        } else {
            $sendingName = $fromUser->getDisplayName();
            $sendingAddress = $fromUser->email;
        }

        $toUser = $this;

        $toUserArray = $toUser->toArray(true);
        $data['fromUser'] = $fromUser;
        $data['toUser'] = $toUser;

        if (validator($toUser->toArray(), ['email' => 'required|email'])->failed()) {
            throw new Exception('User has no email');
        }

        $view = View::make($template)->with($data);
        $content = $view->render();
        $contentLines = explode("\n", $content);
        if (count($contentLines) == 0) {
            Log::warning('Empty mail');
            return;
        }
        $subject = $contentLines[0];
        unset($contentLines[0]);
        $content = implode("\n", $contentLines);

        $baseData['fromUser'] = $fromUser;
        $baseData['toUserArray'] = $toUserArray;
        if ($attachment != false) {
            $account = Account::where('id', '=', $toUserArray['account_id'])->first();
            $path = storage_path('app/public/sepa/' . $account->id . '.pdf');
        } else {
            $account = null;
            $path = null;
        }
        if (isset($data['url'])) {  // for unit testing we need the url in the data array
            $baseData['url'] = $data['url'];
        }
        $baseData['homeLink'] = url('/');
        $baseData['imprintLink'] = url('/imprint');
        $baseData['termsLink'] = $this->getContentLinkForUser('a');
        $baseData['faqLink'] = url('/faq');

        $baseData['content'] = $this->renderParagraphs($content);

        // HOLY CRUTCH!!! backward compatibility with verification service
        if ( 'emails.email_verification_link' === $template && $toUser->email_temp ) {
            $toUserArray['email'] = $toUser->email_temp;
        }

        Mail::queue('emails.html_template.html-base', $baseData,
            function (Message $message) use ($toUserArray, $subject, $sendingName, $sendingAddress, $attachment, $path) {
                $message->subject($subject);
                $message->sender(config('recoma.no-replyAdress'), $sendingName . ' via RECO.MA');
                $message->from(config('recoma.no-replyAdress'), $sendingName . ' via RECO.MA');
                $message->returnPath(config('recoma.no-replyAdress'));
                $message->replyTo($sendingAddress, $sendingName);

                $message->to($toUserArray['email'], $toUserArray['displayName']);
                $message->bcc(config('recoma.bcc-address'), 'Archiv');
                if ($attachment != false) {
                    $message->attach($path);
                }
            });
    }

    public function fakeTemplate()
    {
        $fake = new CouponTemplate();
        $fake->user_id = $this->id;
        $fake->title = 'Beispiel';
        $fake->expiration_date = '2016-01-01';
        $fake->description = 'Beschreibung';
        $fake->provision_agent = 0;
        $fake->discount_consumer = 0;
        $fake->quota = 100;
        return $fake;

    }

    /**
     * Profile URLs use always accounts
     *
     * @param $role
     * @return string
     */
    public function getProfileUrl($role)
    {
        if (!$this->account) {
            return '#';
        }
        return $this->account->getProfileUrl($role);
    }

    public function getProfileLink($role)
    {
        $displayName = $this->getDisplayName();
        return new HtmlString('<a href="' . $this->getProfileUrl($role) . '">' . $displayName . '</a>');
    }

    /**
     * Unlock payments to payout or simulate the payout to calculate the payments which are ready for payout
     *
     * @param bool $forReal false: preview / calculation only, true: process for real
     *
     * @return array pendingCount: How many transactions are pending and can not yet be requested
     *               requestAmount: Amount which is requested for payout
     *               requestCount: Count of transactions requested for payout
     */
    public function processCredits($forReal = true)
    {
        $allocatedTransactions = $this->transactions()->whereStatus(Transaction::STATUS_ALLOCATED)->get();

        $requestCount = 0;
        $requestAmount = 0;
        $pendingRatings = 0;
        /**
         * @var $transaction Transaction
         */
        foreach ($allocatedTransactions as $transaction) {
            if (!$transaction->isCredit()) {
                continue;
            }
            if ($transaction->credit_note == null || $transaction->credit_note->status != CreditNote::STATUS_PAYMENT_REQUESTED) {
                if ($transaction->coupon) {
                    if ( $transaction->coupon->status == Coupon::STATUS_PAID && $transaction->type == Transaction::TYPE_CONSUMER_COUPON_DISCOUNT_CREDIT ) {
                        $pendingRatings ++;
                        continue;
                    }
                }

                $requestCount++;
                $requestAmount += $transaction->amount;

                if ($forReal) {
                    $transaction->createCreditNoteAndRequestPayment();
                    $transaction->status = Transaction::STATUS_PAYMENT_REQUESTED;
                    $transaction->save();

                    if ($transaction->coupon) {
                        $consumer = User::find($transaction->coupon->consumer_user_id);
                        if ($consumer->referred_by) {
                            $consumer->referral_status = User::REFERRAL_STATUS_PAID;
                            $consumer->save();
                        }
                    }
                }
            }
        }

        $pendingCount = $this->transactions()->whereStatus(Transaction::STATUS_PENDING)->count();
        $pendingCount += $pendingRatings;

        return compact('pendingCount', 'requestAmount', 'requestCount');
    }

    public function getBalanceRequestable()
    {
        $payments = $this->processCredits(false);
        return $payments['requestAmount'];
    }

    /**
     * Get short code-content link
     *
     * We have
     *
     * a = AGB
     * d = Datenschutz
     *
     * The group based prefix will be added
     * @param string $shortCode
     * @return \Illuminate\Contracts\Routing\UrlGenerator|string
     */
    public function getContentLinkForUser($shortCode = 'a')
    {
        if ($this->hasRole('seller')) {
            $prefix = 'p';
        } else if ($this->hasRole('agent')) {
            $prefix = 'v';
        } else if ($this->hasRole('consumer')) {
            $prefix = 'e';
        } else {
            $prefix = 'p'; // FIXME some stupid fallback to seller, currently not defined, what should happen (i.e. for operator)
        }

        return url('/' . $prefix . $shortCode);
    }

    public function guidanceShown()
    {
        if (!($this->guidance_shown)) {
            $this->guidance_shown = true;
            $this->save();
            return false;
        }
        return true;
    }


    /**
     * Get Home URL for user
     * @return string
     * @throws \Exception
     */
    public function getRealHomeUrl()
    {
        if ($this->hasRole(User::OPERATOR)) {
            return '/dashboard/operator';
        } else if ($this->hasRole(User::SELLER) || $this->hasRole(User::EMPLOYEE)) {
            return route('seller.dashboard');
        } else if ($this->hasRole(User::AGENT)) {
            return route('user.dashboard');
        } else if ($this->hasRole(User::CONSUMER)) {
            return '/dashboard/consumer';
        } else if ($this->hasRole(User::FIELDSERVICE_EMPLOYEE)) {
            return '/user';
        } else {
            throw new \Exception('Home URL for this user type not specified.');
        }
    }


    /**
     * Check if this user can edit another user
     *
     * @param $user
     */
    public function canEditUser(User $user) {
        if ($user->id == $this->id) {
            return true;
        }
        if ($this->can('edit users')) {
            return true;
        }
        if ($this->can('edit new recruited users')) {
            $userAccount=Account::withoutGlobalScopes()->find($user->account_id);
            if ($userAccount->recruiter_user_id === $this->id && !$user->email_verified) {
                return true;
            }
        }
        return false;
    }

    public function getPermanentShortLink(){
        $linkInvitation= LinkInvitations::where('inviting_user_id',Auth::user()->id)->where('is_permanent_link',true)->first();
        if($linkInvitation){
            $url = url('/registration/' . $linkInvitation->token);
            return Shortlink::shorten($url);
        }
        return null;
    }

    /**
     * Method to get available deals.
     *
     * @return mixed
     */
    public function getAvailableCouponTemplates()
    {
        // TODO extract from model

        return $this->couponTemplates()
            ->available()
            ->get();
    }

    /**
     * Method to get available deals except specific for "other deals" section.
     *
     * @param CouponTemplate $except
     * @param int $limit
     * @param bool $randomize
     * @param bool $withMedia
     * @return mixed
     */
    public function getOtherDeals(CouponTemplate $except, $limit = 4, $randomize = true, $withMedia = true)
    {
        $deals = $this->couponTemplates()
            ->available()
            ->where('id', '<>', $except->id)
            ->limit($limit);

        if ($withMedia) {
            $deals->with('media');
        }

        if ($randomize) {
            $deals->inRandomOrder();
        }

        return $deals->get();
    }

    /**
     * Method to get deal for invite widget.
     *
     * @return \App\CouponTemplate;
     */
    public function getInviteWidgetDeal()
    {
        // TODO extract from model

        $available = $this->getAvailableCouponTemplates();
        $options = $this->options->keyBy('key');
        $dealId = isset($options['invite.deal_id']) ? $options['invite.deal_id']->value : 0;
        $deal = CouponTemplate::find($dealId);

        return $available->contains($deal) ? $deal : $available->first();
    }

    /**
     * Full name accessor.
     *
     * @return mixed|string
     */
    public function getFullnameAttribute()
    {
        if ($this->firstname || $this->surname) {
            $name = $this->firstname;
            $surname = strlen($this->surname) ? $this->surname[0] . '.' : '';
            return trim($name . ' ' . $surname);
        } else {
            return substr($this->email, 0, 3) . '...' . substr($this->email, strpos($this->email, '@'));
        }
    }

    /**
     * Userpic accessor.
     *
     * @return string
     */
    public function getUserpicAttribute()
    {
        if (self::STATUS_DELETED === $this->status) {
            return env('APP_URL', 'https://reco.ma') . '/account/portraits/userpic_rm.jpg';
        } else if ($this->upload_portrait) {
            return env('APP_URL', 'https://reco.ma') . '/account/portraits/' . $this->upload_portrait;
        } else {
            return env('APP_URL', 'https://reco.ma') . '/account/portraits/userpic.jpg';
        }
    }

    /**
     * Method to generate API token.
     *
     * @return mixed|string
     */
    public function generateToken()
    {
        $this->api_token = str_random(60);
        $this->save();

        return $this->api_token;
    }

    /**
     * Referral links accessor.
     *
     * @return mixed
     */
    public function getLinksAttribute()
    {
        return (new ReferralLinkService(Auth::user(), $this))->getAllLinks();
    }

    /**
     * Paypal data accessor.
     *
     * @return array|bool|mixed|\stdClass
     */
    public function getPayPalDataAttribute()
    {
        return $this->paypal ? json_decode($this->paypal) : false;
    }

    /**
     * Method to check whether user has any valid payment type.
     *
     * @return bool
     */
    public function hasPaymentType()
    {
        return $this->paypal || ($this->account->iban && $this->account->iban_owner);
    }

    /**
     * Method to check whether user has filled minimal requirements for seller account.
     *
     * @return bool
     */
    public function hasSellerProfileFilled()
    {
        return $this->account->hasSellerProfileFilled();
    }


    /**
     * Method to check whether user has filled minimal requirements for his account.
     *
     * @return bool
     */
    public function hasProfileFilled()
    {
        return $this->hasFullname();
    }

    /**
     * Method to return seller, attached to user.
     *
     * @return User|null
     */
    public function getAttachedSeller()
    {
        $connection = AccountConnection::where('account_id', $this->account->id)
            ->where('type', 'agent')
            ->first();

        if ($connection) {
            return User::where('account_id', $connection->related_account_id)->first();
        }

        return null;
    }

    /**
     * Method to check whether user has address.
     *
     * @return bool
     */
    public function hasAddress()
    {
        return $this->account->hasAddress();
    }

    /**
     * Method to check whether user has fullname.
     *
     * @return bool
     */
    public function hasFullname()
    {
        return $this->firstname && $this->surname;
    }

    /**
     * Method to check whether user has userpic.
     *
     * @return bool
     */
    public function hasUserpic()
    {
        return $this->upload_portrait;
    }

    /**
     * Method to return user profile progress in percents.
     *
     * @return int
     */
    public function getProfileProgressAttribute()
    {
        $total  = 12;
        $filled = 0;

        if ( $this->hasUserpic() ) {
            $filled += 1;
        }

        if ( $this->hasFullname() ) {
            $filled += 3;
        }

        if ( $this->hasAddress() ) {
            $filled += 4;
        }

        if ($this->email_verified) {
            $filled += 1;
        }

        if ($this->alias != $this->id) {
            $filled += 1;
        }

        if ($this->account->iban && $this->account->iban_owner) {
            $filled += 2;
        }

        return ceil(100 * $filled / $total);
    }

    /**
     * Check whether user is seller.
     *
     * @return bool
     */
    public function isSeller()
    {
        return $this->hasRole('seller');
    }

    /**
     * Check whether user can directly (!) do some action.
     * This is a stupid workaround, written for overriding protected method and do some query optimization.
     *
     * @param $permission
     * @param bool $allowEverythingForSeller
     * @return mixed
     */
    public function canDo($permission, $allowEverythingForSeller = false)
    {
        if ( $allowEverythingForSeller && $this->isSeller() ) {
            return true;
        }

        return $this->permissions->contains('name', $permission);
    }
}