<?php

namespace App;

use App\Library\Facades\Format;
use App\Library\Billomat\BillomatClientHelper;
use Carbon\Carbon;
use App\Presenters\TransactionPresenter;
use App\Presenters\PresentsData;

class Transaction extends AbstractModel
{
    use PresentsData;

    protected $presenter = TransactionPresenter::class;

    protected $dates = ['status_pending_at', 'status_allocated_at', 'status_paid_at', 'status_payment_requested_at'];
    protected $guarded = [];

    protected $casts = [
        'billed_until' => 'date',
        'billed_from' => 'date',
    ];

    /**
     * Transaction was generated but is open
     */
    const STATUS_PENDING = 'pending';

    /**
     * Transaction was rated -> money is ready to be payed out
     */
    const STATUS_ALLOCATED = 'allocated';

    /**
     * Payment was requested, but not yet paid out
     */
    const STATUS_PAYMENT_REQUESTED = 'payment_requested';

    /**
     * Transaction was completed
     */
    const STATUS_PAID = 'paid';

    /**
     * Considered payed-out
     */
    const STATUS_GROUP_PAID_OUT = [
        self::STATUS_PAYMENT_REQUESTED,
        self::STATUS_PAID,
    ];

    const STATUS_COLORS = [
        self::STATUS_PENDING => 'default',
        self::STATUS_ALLOCATED => 'info',
        self::STATUS_PAID => 'success',
        self::STATUS_PAYMENT_REQUESTED => 'primary',
    ];

    const TYPE_OPERATOR_COUPON_PROVISION_CREDIT = 'operator-coupon-provision-credit';

    const TYPE_AGENT_COUPON_PROVISION_CREDIT = 'agent-coupon-provision-credit';
    const TYPE_AGENT_RECOMMENDATION_PROVISION_CREDIT = 'agent-recommendation-provision-credit';

    const TYPE_CONSUMER_COUPON_DISCOUNT_CREDIT = 'consumer-coupon-discount-credit';

    const TYPE_SELLER_OPERATOR_PROVISION_PAYMENT = 'seller-operator-provision-payment';
    const TYPE_SELLER_AGENT_PROVISION_PAYMENT = 'seller-agent-provision-payment';
    const TYPE_SELLER_CONSUMER_DISCOUNT_PAYMENT = 'seller-consumer-discount-payment';

    const TYPE_SELLER_FEE_YEARLY_PAYMENT = 'seller-fee-yearly-payment';

    const TYPE_SELLER_FEE_MONTHLY_PAYMENT = 'seller-fee-monthly-payment';

    const TYPE_SELLER_FREE_MONTHLY_PAYMENT = 'seller-free-monthly-payment';

    const TYPE_REIMBURSED_FEES = [
        self::TYPE_SELLER_OPERATOR_PROVISION_PAYMENT,
        self::TYPE_OPERATOR_COUPON_PROVISION_CREDIT
    ];

    const TYPE_CREDIT = [
        self::TYPE_OPERATOR_COUPON_PROVISION_CREDIT,
        self::TYPE_AGENT_COUPON_PROVISION_CREDIT,
        self::TYPE_CONSUMER_COUPON_DISCOUNT_CREDIT,
        self::TYPE_AGENT_RECOMMENDATION_PROVISION_CREDIT,
    ];

    const RELEASEABLE =[
        self::TYPE_OPERATOR_COUPON_PROVISION_CREDIT,
        self::TYPE_AGENT_COUPON_PROVISION_CREDIT,
        self::TYPE_CONSUMER_COUPON_DISCOUNT_CREDIT,
        self::TYPE_AGENT_RECOMMENDATION_PROVISION_CREDIT
    ];

    const TYPE_PAYMENT = [
        self::TYPE_SELLER_OPERATOR_PROVISION_PAYMENT,
        self::TYPE_SELLER_AGENT_PROVISION_PAYMENT,
        self::TYPE_SELLER_CONSUMER_DISCOUNT_PAYMENT,
        self::TYPE_SELLER_FEE_YEARLY_PAYMENT,
        self::TYPE_SELLER_FEE_MONTHLY_PAYMENT,
        self::TYPE_SELLER_FREE_MONTHLY_PAYMENT,
    ];

    const TYPE_FEE = [
        self::TYPE_SELLER_FEE_MONTHLY_PAYMENT,
        self::TYPE_SELLER_FEE_YEARLY_PAYMENT
    ];

    /**
     * Map of the payments to the respective credit transaction types
     */
    const MAP_PAYMENT_TO_CREDIT = [
        self::TYPE_SELLER_AGENT_PROVISION_PAYMENT => self::TYPE_AGENT_COUPON_PROVISION_CREDIT,
        self::TYPE_SELLER_CONSUMER_DISCOUNT_PAYMENT => self::TYPE_CONSUMER_COUPON_DISCOUNT_CREDIT
    ];

    public function user()
    {
        return $this->belongsTo('App\User');
    }

    public function recruitedAccount()
    {
        return $this->belongsTo('App\Account', 'recruited_account_id', 'id');
    }

    public function isCredit()
    {
    	return in_array($this->type, self::TYPE_CREDIT);
    }

    public function isPayment()
    {
    	return in_array($this->type, self::TYPE_PAYMENT);
    }

    public function typeToString()
    {
        switch ($this->type) {
            case Transaction::TYPE_OPERATOR_COUPON_PROVISION_CREDIT:
                return "Transaktionsgebühr Credit";
            case Transaction::TYPE_AGENT_COUPON_PROVISION_CREDIT:
                return "RECO.CASH";
            case Transaction::TYPE_AGENT_RECOMMENDATION_PROVISION_CREDIT:
                return "Werberprovision";
            case Transaction::TYPE_CONSUMER_COUPON_DISCOUNT_CREDIT:
                return "Cashback";
            case Transaction::TYPE_SELLER_OPERATOR_PROVISION_PAYMENT:
                return "Transaktionsgebühr";
            case Transaction::TYPE_SELLER_AGENT_PROVISION_PAYMENT:
                return "Empfehler Provision";
            case Transaction::TYPE_SELLER_CONSUMER_DISCOUNT_PAYMENT:
                return "Kunden Rabatt";
            case Transaction::TYPE_SELLER_FEE_YEARLY_PAYMENT:
                return "Jährlicher Mitgliedsbeitrag";
            case Transaction::TYPE_SELLER_FEE_MONTHLY_PAYMENT:
                return "Monatlicher Mitgliedsbeitrag";
            case Transaction::TYPE_SELLER_FREE_MONTHLY_PAYMENT:
                return "Freimonat";
        }
    }

    public function __toString()
    {
        return $this->typeToString() . ' ' . $this->typeToDescription();
    }

    public function typeToDescription()
    {
        switch ($this->type) {
            case Transaction::TYPE_OPERATOR_COUPON_PROVISION_CREDIT:
                return "";
            case Transaction::TYPE_AGENT_COUPON_PROVISION_CREDIT:
                return 'RECO.CASH'
                . ' von ' . $this->coupon->template->seller->getDisplayName()
                . ' für die Empfehlung an ' . $this->coupon->consumer->getDisplayName()
                . ' mit RECO.CODE "' . $this->coupon->code . '" '
                . ' aus RECO.BON "' . $this->coupon->template->title . '"';
            case Transaction::TYPE_AGENT_RECOMMENDATION_PROVISION_CREDIT:
                return 'für den RECO.MA Vertragsabschluss'
                       . ' der Firma ' . $this->recruitedAccount->getDisplayName();
            case Transaction::TYPE_CONSUMER_COUPON_DISCOUNT_CREDIT:
                $netto = $this->coupon->netto
                    ? ' mit netto ' . Format::currency($this->coupon->netto, false)
                    : '';
                return 'Cashback auf Ihren Einkauf'
                . ' mit RECO.CODE "' . $this->coupon->code . '"'
                . ' aus RECO.BON "' . $this->coupon->template->title . '"'
                . ' bei ' . $this->coupon->template->seller->getDisplayName()
                . $netto;
            case Transaction::TYPE_SELLER_OPERATOR_PROVISION_PAYMENT:
                return "Vermittelter Auftragswert netto "
                . Format::currency($this->coupon->netto, false)
                . ' mit RECO.CODE "' . $this->coupon->code . '"'
                . ' aus RECO.BON "' . $this->coupon->template->title . '"';
            case Transaction::TYPE_SELLER_AGENT_PROVISION_PAYMENT:
                return "RECO.CASH für die Empfehlung an " . $this->coupon->agent->getDisplayName();
            case Transaction::TYPE_SELLER_CONSUMER_DISCOUNT_PAYMENT:
                return "Cashback an " . $this->coupon->consumer->getDisplayName();
            case Transaction::TYPE_SELLER_FEE_YEARLY_PAYMENT:
                return "von " . Format::date($this->billed_from) . " bis " . Format::date($this->billed_until);
            case Transaction::TYPE_SELLER_FEE_MONTHLY_PAYMENT:
                return "von " . Format::date($this->billed_from) . " bis " . Format::date($this->billed_until);
            case Transaction::TYPE_SELLER_FREE_MONTHLY_PAYMENT:
                return "von " . Format::date($this->billed_from) . " bis " . Format::date($this->billed_until);
        }
    }

	/**
	 * Convert to billomat item
	 * @return array
	 */
    public function toBillomatItem()
    {
	    $item = [
		    'unit' => 'Stück',
		    'quantity' => 1.0,
		    'unit_price' => abs(floatval($this->amount)),
		    'title' => $this->typeToString(),
		    'description' => $this->typeToDescription(),
	    ];

	    if (isset(config('billomat.articles')[$this->type])) {
		    $articleId = BillomatClientHelper::resolveArticle( config( 'billomat.articles' )[ $this->type ] );

		    if ( $articleId ) {
			    $item['article_id'] = $articleId;
		    }
	    }
	    return $item;
    }
    /**
     * Transfers this transaction to billomat as an invoice item
     * @param type $invoice_id
     * @return type
     */
    public function transferToBillomat($invoice_id)
    {

    	$invoiceItem = $this->toBillomatItem();
    	$invoiceItem['invoice_id'] = $invoice_id;

        $arr = [
	        'invoice-item' =>  $invoiceItem
        ];

        $billomatClient = BillomatClientHelper::getBillomatClientInstance();

        $ret = $billomatClient->createInvoiceItem($arr);

        $this->billomat_id = $ret['invoice-item']['id'];
        $this->save();

        return $ret;
    }

    public function coupon()
    {
        return $this->belongsTo('\App\Coupon', 'coupon_id', 'id');
    }

    public function credit_note()
    {
        return $this->belongsTo('\App\CreditNote');
    }

    public function invoice()
    {
        return $this->belongsTo('\App\Invoice');
    }

    /**
     * Get credit note or invoice
     */
    public function getReceipt()
    {
        if (in_array($this->type, self::TYPE_PAYMENT)) {
            return $this->invoice;
        } else if (in_array($this->type, self::TYPE_CREDIT)) {
            return $this->credit_note;
        }
    }

    public function getPDFLink()
    {
        if ($this->getReceipt()) {
            return $this->getReceipt()->getPDFLink();
        }
        return false;
    }

    public function isFee()
    {
        return in_array($this->type, self::TYPE_FEE);
    }

    public function isRecurring()
    {
        return $this->isFee();
    }

    /**
     * Check if credit note transactions which are assigned to a coupon where the fee was paid by the seller
     * after a grace period
     *
     * @param bool $override Override grace period
     */
    public function isEligibleForAllocation($override = false)
    {
        if ($this->coupon) {
            if ( ! in_array( $this->coupon->status, Coupon::STATUS_GROUP_IS_PAID ) ) {
                return false;
            }

            $mapCreditToPayment = array_flip( static::MAP_PAYMENT_TO_CREDIT );
            $paymentType        = $mapCreditToPayment[ $this->type ];
            $paymentTransaction = $this->coupon->paymentTransactions()->whereType( $paymentType )->first();
        } else if ($this->recruitedAccount) {
            $paymentTransaction =
                $this->recruitedAccount
                    ->users()
                    ->first()
                    ->transactions()
                    ->orderBy('created_at', 'ASC')
                    ->whereIn('type', Transaction::TYPE_FEE)
                    ->first();
        } else {
            return false;
            //Type of transaction can not be pending, because it is not defined, whewn it will be released;
        }

        if ($paymentTransaction == null) {
            return false;
        }

        if ($paymentTransaction->status == Transaction::STATUS_PAID) {
            $diff = $paymentTransaction->status_allocated_at->diffInDays(Carbon::now(), true);
            if ($override || $diff >= config('recoma.graceDays')) {
                return true;
            }
        }

        return false;
    }

    /**
     * Release an eligible transaction, send notification mails
     * (only if related to coupon)
     *
     * @param bool $override
     */
    public function releaseEligible($override = false)
    {
	    if ($this->isEligibleForAllocation($override)) {
		    $url = route('transaction');
		    $coupon = $this->coupon;
		    $data = compact('coupon', 'url');
		    $this->status = Transaction::STATUS_ALLOCATED;
		    $this->save();

		    if ($this->coupon && 0 != (float)$this->amount) {
                if ( $this->user == $this->coupon->consumer ) {
                    $this->coupon->consumer->sendMail( 'money-is-available-after-seller-accepts-to-consumer', $data, true );
                } elseif ( $this->user == $this->coupon->agent ) {
                    $this->coupon->agent->sendMail( 'money-is-available-after-seller-accepts-to-agent', $data, true );
                }
            } else {
		        // no mail for non-coupon related transactions
            }
	    }
    }

    /**
     * Release Eligble
     *
     * @param bool $override Override grace period
     */
    public static function releaseAllEligible($override = false)
    {
        $transactions = static::whereStatus(static::STATUS_PENDING)->whereIn('type', static::RELEASEABLE)->get();

        /**
         * @var $transaction Transaction
         */
        foreach ($transactions as $transaction) {
        	$transaction->releaseEligible($override);
        }
    }

    /**
     * Get combined status of credit note and transaction
     * FIXME: might need improvement - currently uses only credit note, not invoice status
     *
     * @param bool $html
     * @return mixed|string
     */
    public function getStatusText($html = true, $prefix = null)
    {
        if ($this->status == static::STATUS_ALLOCATED) {
        	if ($this->isCredit()) {
		        if ( $this->credit_note && $this->credit_note->status != CreditNote::STATUS_OPEN ) {
			        return ( $this->credit_note->getStatusText() );
		        }
	        }
	        if ($this->isPayment()) {
		        if ($this->invoice && $this->invoice->status != Invoice::STATUS_OPEN) {
			        return ($this->invoice->getStatusText());
		        }
	        }
        }
        $prefix = get_class($this) . ($this->isPayment() ? '.payment' : '.credit');
        return parent::getStatusText($html, $prefix);
    }


    /**
     * Create the credit note for a user which is realated to the coupon (agent or consumer)
     * and directly request the payment
     *
     * @param User $forUser
     * @return \App\CreditNote
     */
    public function createCreditNoteAndRequestPayment()
    {
        $transaction = $this;
        $creditNote = new CreditNote();
        $creditNote->user_id = $transaction->user->id;
        $creditNote->transaction_id = $transaction->id;
        $creditNote->transferToBillomat();
        $creditNote->completeCreditNote();

        $creditNote->storePDF();
        $creditNote->status = CreditNote::STATUS_PAYMENT_REQUESTED;

        $creditNote->save();

        return $creditNote;
    }

}
