<?php

namespace App\Services;


use App\Events\SimpleTokenGenerated;
use App\Library\GoogleAuth;
use App\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;

class TFAService
{
    /**
     * Current user.
     *
     * @var User
     */
    private $user;

    /**
     * TFAService constructor.
     * @param User $user
     */
    public function __construct(User $user)
    {
        $this->user = $user;
    }

    /**
     * Method to check whether user has simple TFA (code on email).
     *
     * @return bool
     */
    public function hasSimpleTFA()
    {
        return !empty($this->user->tfa_code);
    }

    /**
     * Method to check whether user has Google TFA.
     *
     * @return bool
     */
    public function hasGoogleTFA()
    {
        return !empty($this->user->google_tfa_secret);
    }

    /**
     * Method to check whether user has two factor auth enabled.
     *
     * @return bool
     */
    public function hasTwoFactorAuth()
    {
        return $this->hasGoogleTFA() || $this->hasSimpleTFA();
    }

    /**
     * Method to check whether user need to be authenticated via TFA.
     *
     * @return bool
     */
    public function needTwoFactorAuth()
    {
        // if device is trusted, there is no need ti be authenticated via TFA
        $device = (new DeviceService())->current($this->user);

        if ( null === $device ) {
            //Auth::logout();
            return false;
        }

        if ( $device->isTrusted() ) {
            return false;
        }

        return $this->hasGoogleTFA() || ($this->hasSimpleTFA());
    }

    /**
     * Method to generate key for simple TFA.
     *
     * @param bool $silent
     */
    public function generateSimpleToken($silent = false)
    {
        if ($this->hasGoogleTFA()) { // simple key is not necessary, cause google TFA will have first priority
            return;
        }

        // generate temporary key
        $token = rand(100000, 999999);

        $this->user->timestamps = false;
        $this->user->tfa_code = $token;
        $this->user->tfa_code_expires_at = Carbon::now()->addMinutes(config('services.tfa.simple.expiration_time'));
        $this->user->save();

        activity($this->user)->register('logged in from untrusted device', null, ['token' => $token]);

        if (!$silent) { // fire event, if no need to suppress it
            event(new SimpleTokenGenerated($this->user));
        }
    }

    /**
     * Method to check whether token is expired.
     *
     * @return mixed
     */
    public function isTokenExpired()
    {
        if ($this->hasGoogleTFA()) {
            return false; // google TFA can't be expired
        }

        return $this->user->tfa_code_expires_at->lt(Carbon::now());
    }

    /**
     * Method to check whether token is valid.
     *
     * @param $token
     * @return bool
     */
    public function isTokenValid($token)
    {
        if ( $this->hasGoogleTFA() ) {
            return (new GoogleAuth())->verifyCode($this->user->google_tfa_secret, $token, 2);
        }

        return $token === $this->user->tfa_code;
    }

    /**
     * Method to reset TFA code.
     */
    public function reset()
    {
        if ( $this->hasGoogleTFA() ) { // it is not necessary when google TFA enabled
            return;
        }

        $this->user->timestamps = false;
        $this->user->tfa_code = null;
        $this->user->tfa_code_expires_at = null;
        $this->user->save();
    }

    /**
     * Method to disconnect Google TFA.
     */
    public function googleDisconnect()
    {
        $this->user->google_tfa_secret = null;
        $this->user->save();

        activity($this->user)->register('tfa disconnected');
    }

    /**
     * Method to get TFA connection data.
     *
     * @return array
     */
    public function getTFAConnectionData()
    {
        $ga = new GoogleAuth();

        $name = $this->user->email;
        $secret = $ga->createSecret();
        $qrcode = $ga->getQRCodeGoogleUrl('RECOMA:' . $name, $secret, 'RECOMA');

        session(['google_auth_secret' => $secret]);

        return [
            'name'   => $name,
            'secret' => $secret,
            'qrcode' => $qrcode,
        ];
    }

    /**
     * Method to finish connection with TFA app.
     *
     * @param $code
     * @return bool
     */
    public function finishTFAConnection($code)
    {
        $secret = session('google_auth_secret');

        if ( (new GoogleAuth())->verifyCode($secret, $code, 2) ) {
            session()->forget('google_auth_secret');

            $this->user->google_tfa_secret = $secret;
            $this->user->save();

            activity($this->user)->register('tfa connected', null, [
                'name' => $this->user->email,
                'secret' => $secret,
            ]);

            return true;
        }

        return false;
    }
}