<?php

namespace App\Services;


use App\Coupon;
use App\Transaction;
use App\User;
use App\Library\Facades\Format;
use Illuminate\Support\Facades\DB;

class StatisticsService
{
    public static $monthes = [
        'Jan',
        'Feb',
        'Mär',
        'Apr',
        'Mai',
        'Jun',
        'Jul',
        'Aug',
        'Sep',
        'Okt',
        'Nov',
        'Dez',
    ];

    /**
     * Method to return account numbers, depended of user transactions and coupons.
     *
     * @param User $user
     * @return array
     */
    public static function calculateTransactionsNumbers(User $user)
    {
        $sellerCoupons = $user->couponsSharedFromThisSeller;

        $transactions = $user->transactions->filter(function ($item) {
            return $item->status !== Transaction::STATUS_PAID;
        });

        return [
            'netto'    => Format::currency($sellerCoupons->sum('netto')),
            'consumer' => Format::currency($transactions->where('type', Transaction::TYPE_SELLER_CONSUMER_DISCOUNT_PAYMENT)->sum('amount') * -1),
            'agent'    => Format::currency($transactions->where('type', Transaction::TYPE_SELLER_AGENT_PROVISION_PAYMENT)->sum('amount') * -1),
            'recoma'   => Format::currency($transactions->where('type', Transaction::TYPE_SELLER_OPERATOR_PROVISION_PAYMENT)->sum('amount') * -1),
        ];
    }

    /**
     * Method to return count of activated coupons counts by deal, grouped by month.
     *
     * @param User $user
     * @param int $dealsCount
     * @param int $monthesCount
     * @return mixed
     */
    public static function getActivatedCouponsByPeriod(User $user, $dealsCount = 0, $monthesCount = 6)
    {
        // TODO refactoring ?

        // get all activated coupons
        $items = DB::table('coupons')
            ->select(['coupon_templates.title', DB::raw('COUNT(`coupons`.`id`) AS cnt, DATE_FORMAT(`coupons`.`created_at`, "%Y%m") AS period')])
            ->join('coupon_templates', 'coupon_templates.id', '=', 'coupons.coupon_template_id')
            ->where('coupon_templates.user_id', $user->id)
            ->where('coupons.created_at', '>=', DB::raw('curdate() - interval (dayofmonth(curdate()) - 1) day - interval ' . $monthesCount . ' month'))
            ->groupBy(DB::raw('coupon_templates.title, DATE_FORMAT(coupons.created_at, "%Y%m")'))
            ->get();

        // group by deal title
        $items = collect($items)
            ->groupBy('title')
            ->sortByDesc(function ($item) {
                return $item->sum('cnt');
            });

        // limit if need
        if ($dealsCount) {
            $items = $items->take($dealsCount);
        }

        // set for collection keys as period (e.g. 201910)
        $items = $items->map(function ($item) {
            return $item->keyBy('period');
        });

        // result collection
        $result = collect([]);
        $y = date('Y');
        $m = date('m') + 1;

        // for each month
        for ($i = 0; $i < $monthesCount; $i++) {
            // make key (year + month with leading zero)
            $m--;
            if (0 == $m) {
                $y--;
                $m = 12;
            }
            $d = $y . ($m < 10 ? '0' : '') . $m;

            // collection for values
            $values = collect([]);
            $j = 0;
            foreach ($items as $k => $v) {
                $values->push((object)[
                    'name'  => $k,
                    'key'   => 'y' . $j++,
                    'value' => isset($v[$d]) ? $v[$d]->cnt : 0,
                ]);
            }

            $v = (object)[
                'period' => self::$monthes[$m - 1] . ' ' . $y,
                'values' => $values,
            ];

            $result[$d] = $v;
        }

        return $result->reverse();
    }

    /**
     * Method to return count of successfully paid coupons counts by deal, grouped by month.
     *
     * @param User $user
     * @param int $dealsCount
     * @param int $monthesCount
     * @return mixed
     */
    public static function getPaidCouponsByPeriod(User $user, $dealsCount = 0, $monthesCount = 6)
    {
        $items = DB::table('coupons')
            ->select(['coupon_templates.title', DB::raw('COUNT(`coupons`.`id`) AS cnt, DATE_FORMAT(`coupons`.`status_paid_at`, "%Y-%m") AS period')])
            ->join('coupon_templates', 'coupon_templates.id', '=', 'coupons.coupon_template_id')
            ->whereIn('coupons.status', [Coupon::STATUS_PAID, Coupon::STATUS_DONE])
            ->where('coupon_templates.user_id', $user->id)
            ->where('coupons.status_paid_at', '>=', DB::raw('curdate() - interval (dayofmonth(curdate()) - 1) day - interval ' . $monthesCount . ' month'))
            ->groupBy(DB::raw('coupon_templates.title, DATE_FORMAT(coupons.status_paid_at, "%Y-%m")'))
            ->get();

        $items = collect($items)
            ->groupBy('title')
            ->sortByDesc(function ($item) {
                return $item->sum('cnt');
            });

        if ($dealsCount) {
            $items = $items->take($dealsCount);
        }

        return $items;
    }
}