<?php

namespace App\Services;

use App\Sortings\NewestSorting;
use App\Sortings\MostVievedSorting;
use App\Sortings\MostSharedSorting;
use App\Sortings\DistanceSorting;
use Illuminate\Database\Eloquent\Builder;

class SortingService
{
    /**
     * List of available sorting classes.
     *
     * @var array
     */
    private $availableSortings = [
        NewestSorting::class,
        MostVievedSorting::class,
        MostSharedSorting::class,
        DistanceSorting::class,
    ];

    /**
     * Deafult sorting class.
     *
     * @var string
     */
    private $defaultSorting = NewestSorting::class;

    /**
     * List of sorting instances.
     *
     * @var array
     */
    private $sortings;

    /**
     * Current sorting class instance.
     *
     * @var \App\Sortings\AbstractSorting
     */
    private $currentSorting;

    /**
     * Query builder.
     *
     * @var Builder|null
     */
    private $builder = null;

    /**
     * SortingService constructor.
     *
     * @param Builder|null $builder
     */
    public function __construct(Builder $builder = null)
    {
        $this->builder = $builder;

        $this->initSortings();
        $this->setCurrentSorting();
    }

    /**
     * Method to init sortings from its classes.
     */
    private function initSortings()
    {
        $this->sortings = [];

        foreach ($this->availableSortings as $availableSorting) {
            $sorting = new $availableSorting($this->builder);
            if ( $sorting->isPossible() ) {
                $this->sortings[$sorting->param] = $sorting;
            }
        }
    }

    /**
     * Method to set current sorting.
     */
    private function setCurrentSorting()
    {
        $sort = request('sort');

        if ( isset($this->sortings[$sort]) ) {
            $this->currentSorting = $this->sortings[$sort];
        } else {
            $this->currentSorting = new $this->defaultSorting($this->builder);
        }

        $this->currentSorting->current = true;
    }

    /**
     * Method to apply current sorting.
     *
     * @return mixed
     */
    public function apply()
    {
        return $this->currentSorting->apply();
    }

    /**
     * Method to get sortings.
     *
     * @return array
     */
    public function getSortings()
    {
        return $this->sortings;
    }
}