<?php

namespace App\Services;


use App\CouponTemplate;
use App\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class SellerService
{
    private $seller;

    /**
     * SellerService constructor.
     * @param User $seller
     */
    public function __construct(User $seller = null)
    {
        $this->seller = $seller;
    }

    /**
     * Method to return best deal (by criterion).
     *
     * @param string $criterion
     * @return mixed
     */
    public function getBestDeal($criterion = 'discount_consumer')
    {
        return $this->seller
            ->deals()
            ->available()
            ->orderBy($criterion, 'desc')
            ->first();
    }

    /**
     * Method to get all attached to user sellers with gready loaded accounts and deals.
     *
     * @param User $user
     * @return mixed
     */
    public function getAttachedSellers(User $user)
    {
        // get invited sellers
        $invitedSellers = DB::table('users')
            ->select('id')
            ->where('status', User::STATUS_ACTIVE)
            ->where('invited_by', $user->id);

        // get sellers via user's coupons
        $sellersViaCoupons = DB::table('coupons')
            ->join('coupon_templates', 'coupon_templates.id', '=', 'coupons.coupon_template_id')
            ->select(DB::raw('DISTINCT(coupon_templates.user_id)'))
            ->where('coupons.agent_user_id', $user->id);

        // union with attached seller
        $sellerIds = DB::table('users')
            ->select('id')
            ->where('status', User::STATUS_ACTIVE)
            ->whereRaw('
                account_id = (
                    SELECT related_account_id FROM `account_connections` WHERE `account_id` = (
                        SELECT account_id FROM `users` WHERE id = ' . $user->id . ' 
                    )
                )'
            )
            ->union($invitedSellers)
            ->union($sellersViaCoupons)
            ->pluck('id');

        // get sellers with accounts and deals
        $sellers = User::whereIn('id', $sellerIds)
            ->with(['account', 'deals' => function ($query) {
                $query
                    ->withCount('coupons')
                    ->where('expiration_date', '>=', Carbon::today())
                    ->where('status',          '=',  CouponTemplate::STATUS_ACTIVE)
                    ->having('quota',          '>',  DB::raw('coupons_count'));
            }])
            ->get()
            ->sortBy('account.company');

        // find best deals
        $sellers = $sellers->map(function ($seller) {
            $seller->bestCashback = $seller->deals->sortByDesc('discount_consumer')->first();
            $seller->bestRecocash = $seller->deals->sortByDesc('provision_agent')->first();
            return $seller;
        });

        return $sellers;
    }
}