<?php

namespace App\Services;


use App\Account;
use App\Events\SellerRegistered;
use Carbon\Carbon;
use App\Transaction;
use App\User;
use DB;

class SellerRegistrationService extends AbstractRegistrationService
{
    protected $initialPassword;

    /**
     *
     * Apply invitation bonus to the already existing account
     *
     * @param $account
     *
     * @throws \Exception
     */
    public static function applyInvitationBonus(Account $account)
    {
        $inviter = $account->recruiterUser;

        if (!$inviter) {
            return;
        }

        //Only seller and agents can invite seller to recoma
        if ($inviter->hasRole(User::SELLER)) {
            $account->createFreeMonthMembershipInvoice();
            $inviter->account->createFreeMonthMembershipInvoice();
            return;
        }

        if ($inviter->hasRole(User::AGENT) && !$inviter->hasRole(User::FIELDSERVICE_EMPLOYEE)) {
            $account->createFreeMonthMembershipInvoice();
        }

        if ($inviter->hasRole(User::AGENT) || $inviter->hasRole(User::FIELDSERVICE_EMPLOYEE)) {
            // FIXME Extract - but where to ?

            // Create provision
            switch ($account->payment) {
                case 'month':
                    $amount = config('recoma.fix-provision');
                    break;
                case 'year':
                    $amount = config('recoma.fix-provision');
                    break;
            }

            Transaction::create([
                'user_id' => $inviter->id,
                'recruited_account_id' => $account->id,
                'amount' => $amount,
                'status' => Transaction::STATUS_PENDING,
                'type' => Transaction::TYPE_AGENT_RECOMMENDATION_PROVISION_CREDIT,
            ]);

            return;
        } else {
            throw new \Exception('Not supported');
        }
    }

    /**
     * Create initial password which is available in clear text during the service instance
     */

    public function createInitialPassword()
    {
        $this->initialPassword = str_random(15);
        $this->user->password = bcrypt($this->initialPassword );
    }


    /**
     * Register:
     * Creates User/Account, assigns roles and gives permissions
     */
    public function register($active = false)
    {
        DB::transaction(function () use ($active) {
            $this->create();
            $this->user->assignRole(User::SELLER);
            if ($active) {
                $this->user->assignRole('seller-active');
            }
            $this->user->givePermissionTo('edit account');
            $this->user->givePermissionTo('edit account data');
        });

        $this->user->account->onboarding_process_level = 6;
        $this->user->account->save();

        event(new SellerRegistered($this->user));

        return $this->user;
    }

    public function sendDoubleOptInMail($user, $template = 'confirm-registration-to-seller')
    {
        $token = $user->email_verification_token;
        $data['url'] = url('/activate', ['id' => $this->user->id, 'token' => $token]);

        if ($this->initialPassword) {
            $data['initialPassword'] = $this->initialPassword;
        }
        $user->sendMail($template, $data, true);
    }

    /**
     * Sets free trial to passed value or unlimited free trial
     * @param $trial
     */
    public function setTrial($trial)
    {

        if($trial == 'free'){
            $this->account->billed_until = Carbon::createFromDate('9999', '12', '31')->toDateString();
        }
        elseif($trial != '0'){
            $this->account->billed_until = Carbon::now()->addMonth($trial)->toDateString();
        }
    }
}
