<?php

namespace App\Services;


use App\Account;
use App\Category;
use App\CouponTemplate;

class SearchService
{
    /**
     * String to search by.
     *
     * @var string
     */
    private $searchString;

    /**
     * Limitation of rows count. No limitation if 0.
     *
     * @var int
     */
    private $limit;

    /**
     * Safe mode for API search. If true, in results will be only minimal necessary fields.
     *
     * @var boolean
     */
    private $safeMode = true;

    /**
     * SearchService constructor.
     *
     * @param $searchString
     * @param int $limit
     */
    public function __construct($searchString, $limit = 5, $safeMode = true)
    {
        $this->searchString = '%' . $searchString . '%';
        $this->safeMode     = $safeMode;
        $this->limit        = $limit;
    }

    /**
     * Method to remove limit.
     */
    public function noLimit()
    {
        $this->limit = 0;
    }

    /**
     * Method to set limit.
     *
     * @param $limit
     */
    public function setLimit(int $limit)
    {
        $this->limit = $limit;
    }

    /**
     * Method to find all data.
     *
     * @return array
     */
    public function findAll()
    {
        return [
            'deals'      => $this->findDeals(),
            'sellers'    => $this->findSellers(),
            'categories' => $this->findCategories(),
        ];
    }

    /**
     * Method to find deals.
     *
     * @return mixed
     */
    public function findDeals()
    {
        $query = CouponTemplate::available()
            ->with(['media', 'seller.account'])
            ->where('title', 'LIKE', $this->searchString)
            ->latest();

        if ($this->limit > 0) {
            $query->limit($this->limit);
        }

        $result = $query->get();

        if ($this->safeMode) {
            $items = collect([]);
            foreach ($result as $deal) {
                $items->push([
                    'id'     => $deal->id,
                    'link'   => route('deal.index', ['couponTemplate' => $deal, 'alias' => $deal->seller->alias]),
                    'image'  => $deal->mainImage,
                    'title'  => $deal->title,
                    'seller' => $deal->seller->account->company,
                ]);
            }
            return $items;
        }

        return $result;
    }

    /**
     * Method to find sellers.
     *
     * @return mixed
     */
    public function findSellers()
    {
        $query = Account::with('users')
            ->where('company', 'LIKE', $this->searchString)
            ->where('status', 'active')
            ->orderBy('company');

        if ($this->safeMode) {
            $query->select('id', 'company');
        }

        if ($this->limit > 0) {
            $query->limit($this->limit);
        }

        $result = $query->get();

        if ($this->safeMode) {
            $result->map(function ($item) {
                $item->link = route('seller.index', ['alias' => $item->users->first()->alias]);
                $item->logo = $item->logo200;
                unset($item->users);
            });
        }

        return $result;
    }

    /**
     * Method to find categories.
     *
     * @return mixed
     */
    public function findCategories()
    {
        $query = Category::where('name', 'LIKE', $this->searchString)
            ->orderBy('name');

        if ($this->safeMode) {
            $query->select('id', 'name', 'url');
        }

        if ($this->limit > 0) {
            $query->limit($this->limit);
        }

        $result = $query->get();

        if ($this->safeMode) {
            $result->map(function ($item) {
                $item->link = route('category.index', ['hierarchy' => $item->url]);
            });
        }

        return $result;
    }
}