<?php

namespace App\Services;

use App\CouponTemplate;
use App\User;
use hisorange\BrowserDetect\Facade\Parser as BrowserDetect;
use Illuminate\Support\Facades\Auth;

class ReferralLinkService
{
    protected $user;
    protected $seller;
    protected $deal;
    protected $rawLink;
    protected $linkText;
    protected $isMobile;

    /**
     * ReferralLinkService constructor.
     * @param User|null $user
     * @param User|null $seller
     * @param CouponTemplate|null $deal
     */
    public function __construct(User $user = null, User $seller = null, CouponTemplate $deal = null)
    {
        $this->user   = ($user) ? $user : Auth::user();
        $this->seller = $seller;
        $this->deal   = $deal;

        $this->isMobile = BrowserDetect::isMobile();

        $this->buildRawLink();

        $this->initText();
    }

    /**
     * Method to initialize link text according shareable content.
     */
    protected function initText()
    {
        if ($this->user) {
            $text = 'Ich empfehle Dir RECO.MA';

            if ( null !== $this->seller ) {
                $text = 'Ich empfehle Dir ' . $this->seller->account->company;
                $deal = (new SellerService($this->seller))->getBestDeal();
                if ( null !== $deal ) {
                    $text .= '%0a' . 'Du erhältst bis zu ' . $deal->present()->discount_consumer_text . ' Cashback beim Einkauf, wenn Du meiner Empfehlung folgst.';
                }
            }

            if ( null !== $this->deal ) {
                $text = 'Ich empfehle Dir ' . $this->deal->seller->account->company;
                $text .= '%0a' . 'Du erhältst ' . $this->deal->present()->discount_consumer_text . ' Cashback beim Einkauf, wenn Du meiner Empfehlung folgst.';
            }

            $seller = null;

            if ( $this->user->hasRole('seller') ) {
                $seller = $this->user;
            }

            if ( $this->user->hasRole('employee') ) {
                $seller = $this->user->boss;
            }

            if ( null !== $seller ) {
                $text = 'Ich lade Dich zu RECO.MA ein.';
                $deal = (new SellerService($seller))->getBestDeal('provision_agent');
                if ( null !== $deal && CouponTemplate::TYPE_NONE !== $deal->type_provision ) {
                    $text .= '%0a' . 'Empfehle uns an Deine Freunde und erhalte bis zu ' . $deal->present()->provision_agent_text . ' Prämie.';
                } else {
                    $text .= '%0a' . 'Empfehle uns an Deine Freunde und erhalte Prämie.';
                }
            }

            $this->linkText = $text . '%0a' . $this->rawLink;
        } else {
            $this->linkText = $this->rawLink;
        }
    }

    /**
     * Method to get basic route.
     *
     * @param $alias
     * @return string
     */
    protected function getBasicRoute($alias)
    {
        return route('user.referral-link', $alias);
    }

    /**
     * Method to build raw link;
     */
    private function buildRawLink()
    {
        if ($this->user) {
            $link = $this->getBasicRoute($this->user->alias);

            if ( null !== $this->seller ) {
                $link .= '?s=' . $this->seller->id;
            }

            if ( null !== $this->deal ) {
                $link .= '?d=' . $this->deal->id;
            }
        } else {
            $link = url('/login');
        }

        $this->rawLink = $link;
    }

    /**
     * Method to return raw referral link.
     *
     * @return string
     */
    public function getRawLink()
    {
        return $this->rawLink;
    }

    /**
     * Method to get link for sharing via Whatsapp.
     *
     * @return string
     */
    public function getWhatsappLink()
    {
        return $this->isMobile
            ? 'whatsapp://send?text=' . $this->linkText
            : 'http://web.whatsapp.com/send?text=' .$this->linkText;
    }

    /**
     * Method to get link for sharing via Telegram.
     *
     * @return string
     */
    public function getTelegramLink()
    {
        return $this->isMobile
            ? 'tg://msg_url?url=' . $this->linkText
            : 'http://telegram.me/share/url?url=' . $this->linkText;
    }

    /**
     * Method to get all links.
     *
     * @return object
     */
    public function getAllLinks()
    {
        return (object)[
            'raw'      => $this->getRawLink(),
            'whatsapp' => $this->getWhatsappLink(),
            'telegram' => $this->getTelegramLink(),
        ];
    }
}