<?php

namespace App\Services;


use App\Place;
use Illuminate\Support\Facades\Auth;

class PlaceService
{
    private $place = null;

    private $attributesToSkip = [
        'id',
        'status',
        'seller_id',
        'created_at',
        'updated_at',
    ];

    /**
     * PlaceService constructor.
     * @param Place|null $place
     */
    public function __construct(Place $place = null)
    {
        $this->place = $place;
    }

    /**
     * Method to prepare place data.
     *
     * @param $data
     * @return mixed
     */
    private function prepareData($data)
    {
        if (null === $this->place) {
            $user = Auth::user();

            if ( $user->hasRole('employee') ) {
                $user = $user->boss;
            }

            $data['seller_id'] = $user->id;
        }

        $geo = (new GeoService())->getCoordinatesByAddress($data);

        $data['lat'] = $geo->lat;
        $data['lon'] = $geo->lon;

        return $data;
    }

    /**
     * Method to prepare attributes array for saving into log.
     *
     * @return array
     */
    private function prepareAttributes()
    {
        $attributes = $this->place->getAttributes();

        foreach ($this->attributesToSkip as $key) {
            unset($attributes[$key]);
        }

        return $attributes;
    }

    /**
     * Method to create new place.
     *
     * @param $data
     * @return Place
     */
    public function create($data)
    {
        $data = $this->prepareData($data);

        $place = Place::create($data);

        activity(Auth::user())->register('place added', null, $data, Place::class, $place->id);

        return $place;
    }

    /**
     * Method to update place.
     *
     * @param $data
     * @return bool|int
     */
    public function update($data)
    {
        $data = $this->prepareData($data);

        $oldData = $this->prepareAttributes();

        $place = $this->place->update($data);

        activity(Auth::user())->register('place edited', $oldData, $data, Place::class, $this->place->id);

        return $place;
    }

    /**
     * Method to activate place.
     */
    public function activate()
    {
        $this->place->status = Place::STATUS_ACTIVE;
        $this->place->save();

        activity(Auth::user())->register('place activated', null, null, Place::class, $this->place->id);
    }

    /**
     * Method to deactivate place.
     */
    public function deactivate()
    {
        $this->place->status = Place::STATUS_INACTIVE;
        $this->place->save();

        activity(Auth::user())->register('place deactivated', null, null, Place::class, $this->place->id);
    }

    /**
     * Method to delete place.
     */
    public function delete()
    {
        $this->place->status = Place::STATUS_DELETED;
        $this->place->save();

        activity(Auth::user())->register('place deleted', null, null, Place::class, $this->place->id);
    }
}