<?php

namespace App\Services;

use App\CreditNote;
use App\Events\CouponFinishedClientTransactions;
use App\Transaction;
use GuzzleHttp\Client as GuzzleClient;
use Illuminate\Support\Facades\Auth;
use Monolog\Logger;
use Monolog\Handler\StreamHandler;

class PayPalService
{
    private $mode;
    private $secret;
    private $endpoint;
    private $clientId;
    private $basicToken;

    private $guzzleClient;

    private $log;

    /**
     * PayPalService constructor.
     */
    public function __construct()
    {
        $this->init();
    }

    /**
     * Method to initialize service.
     */
    private function init()
    {
        $this->clientId   = config('paypal.client_id');
        $this->secret     = config('paypal.secret');
        $this->mode       = config('paypal.settings.mode');
        $this->endpoint   = config('paypal.api_endpoint.' . $this->mode);
        $this->basicToken = base64_encode($this->clientId . ':' . $this->secret);

        $this->log = new Logger('paypal');
        $this->log->pushHandler(new StreamHandler(storage_path('logs/paypal.log')), Logger::INFO);

        $this->guzzleClient = new GuzzleClient();
    }

    /**
     * Method to get access token.
     *
     * @param $code
     * @return mixed
     */
    private function getAccessToken($code)
    {
        $response = $this->guzzleClient->request('POST', $this->endpoint . 'v1/identity/openidconnect/tokenservice', [
            'query' => [
                'grant_type' => 'authorization_code',
                'code' => $code,
            ],
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Basic ' . $this->basicToken,
            ],
        ]);

        $statusCode = $response->getStatusCode();

        if (200 !== $statusCode) {
            return false;
        }

        $content = json_decode($response->getBody()->getContents());

        if ($content && isset($content->access_token)) {
            return $content->access_token;
        } else {
            return false;
        }
    }

    private function getUserData($accessToken)
    {
        $response = $this->guzzleClient->request('GET', $this->endpoint . 'v1/identity/oauth2/userinfo?schema=paypalv1.1', [
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $accessToken,
            ],
        ]);

        $statusCode = $response->getStatusCode();

        if (200 !== $statusCode) {
            return false;
        }

        $content = json_decode($response->getBody()->getContents());

        return $content;
    }

    /**
     * Method to connect user with PayPal account.
     *
     * @param $code
     * @return bool
     */
    public function connect($code)
    {
        $accessToken = $this->getAccessToken($code);

        if (!$accessToken) {
            return false;
        }

        $userData = $this->getUserData($accessToken);

        if ($userData) {
            $user = Auth::user();
            $user->paypal = json_encode($userData);
            $user->save();
        }

        return $userData;
    }

    /**
     * Method to disconnect PayPal account.
     */
    public function disconnect()
    {
        $user = Auth::user();
        $user->paypal = null;
        $user->save();
    }

    /**
     * Method to get OAuth API access token.
     *
     * @return array|mixed|\stdClass
     */
    private function getAppAccessToken()
    {
        $response = $this->guzzleClient->request('POST', $this->endpoint . 'v1/oauth2/token', [
            'query' => [
                'grant_type' => 'client_credentials',
            ],
            'headers' => [
                'Accept' => 'application/json',
            ],
            'auth' => [
                $this->clientId, $this->secret
            ],
        ]);

        $statusCode = $response->getStatusCode();

        if (200 !== $statusCode) {
            return false;
        }

        $content = json_decode($response->getBody()->getContents());

        if ($content && isset($content->access_token)) {
            return $content->access_token;
        } else {
            return false;
        }
    }

    /**
     * Method to return payout batch header.
     *
     * @param CreditNote $creditNote
     * @return object
     */
    private function getPayOutBatchHeader(CreditNote $creditNote)
    {
        return (object)[
            'sender_batch_id' => $creditNote->number . '2152',
            'email_subject'   => 'RECO.MA Auszahlung',
            'email_message'   => 'RECO.MA Auszahlung (RECO.BON ' . $creditNote->transaction->coupon->code . ')',
        ];
    }

    /**
     * Method to return payout items (currently one item).
     *
     * @param CreditNote $creditNote
     * @return array
     */
    private function getPayOutItems(CreditNote $creditNote)
    {
        return [
            (object)[
                'recipient_type' => 'PAYPAL_ID',
                'amount' => (object)[
                    'value' => $creditNote->gross_amount,
                    'currency' => 'EUR',
                ],
                'note' => 'RECO.MA Auszahlung',
                'sender_item_id' => $creditNote->number . '_' . $creditNote->id,
                'receiver' => $creditNote->paypal_payer_id
            ]
        ];
    }

    /**
     * Method to return payout data.
     *
     * @param CreditNote $creditNote
     * @return object
     */
    private function getPayOutData(CreditNote $creditNote)
    {
        return (object)[
            'sender_batch_header' => $this->getPayOutBatchHeader($creditNote),
            'items' => $this->getPayOutItems($creditNote),
        ];
    }

    /**
     * Method to process one payout batch (currently one transaction per credit note).
     *
     * @param CreditNote $creditNote
     * @param $accessToken
     * @return bool
     */
    private function processPayOut(CreditNote $creditNote, $accessToken)
    {
        $response = $this->guzzleClient->request('POST', $this->endpoint . 'v1/payments/payouts', [
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $accessToken,
            ],
            'body' => json_encode($this->getPayOutData($creditNote)),
            'http_errors' => false
        ]);

        $statusCode = $response->getStatusCode();

        $content = json_decode($response->getBody()->getContents());

        if (201 !== $statusCode) {
            $this->log->error('PayOut send', (array)$content);
            return false;
        } else {
            return $content;
        }
    }

    /**
     * Method to process all PayPal credit notes, requested to payout.
     *
     * @return int
     */
    public function processPayOuts()
    {
        $creditNotes = CreditNote::where('status', CreditNote::STATUS_PAYMENT_REQUESTED)
            ->where('payment_type', 'paypal')
            ->get();

        if ( $creditNotes && $accessToken = $this->getAppAccessToken() ) {
            foreach ($creditNotes as $creditNote) {
                $payoutResult = $this->processPayOut($creditNote, $accessToken);
                if (false !== $payoutResult) {
                    $creditNote->status = CreditNote::STATUS_EXPORTED;
                    $creditNote->paypal_payload = json_encode($payoutResult);
                    $creditNote->save();
                }
            }
        }

        return $creditNotes->count();
    }

    /**
     * Method to get batch details.
     *
     * @param CreditNote $creditNote
     * @param $accessToken
     * @return array|bool|mixed|\stdClass
     */
    private function getBatchDetails(CreditNote $creditNote, $accessToken)
    {
        $batchId = json_decode($creditNote->paypal_payload)->batch_header->payout_batch_id;

        $response = $this->guzzleClient->request('GET', $this->endpoint . 'v1/payments/payouts/' . $batchId, [
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $accessToken,
            ],
        ]);

        $statusCode = $response->getStatusCode();

        $content = json_decode($response->getBody()->getContents());

        if (200 !== $statusCode) {
            $this->log->error('PayOut check', (array)$content);
            return false;
        } else {
            return $content;
        }
    }

    /**
     * Method to check batch statuses and finalize credit notes.
     *
     * @return mixed
     */
    public function finalizePayOuts()
    {
        $creditNotes = CreditNote::where('status', CreditNote::STATUS_EXPORTED)
            ->where('payment_type', 'paypal')
            ->get();

        if ( $creditNotes && $accessToken = $this->getAppAccessToken() ) {
            foreach ($creditNotes as $creditNote) {
                $batchDetails = $this->getBatchDetails($creditNote, $accessToken);
                if (false !== $batchDetails && 'SUCCESS' === $batchDetails->batch_header->batch_status) {
                    $creditNote->status = CreditNote::STATUS_COMPLETE;
                    $creditNote->save();
                    $creditNote->transaction->status = Transaction::STATUS_PAID;
                    $creditNote->transaction->save();
                    event(new CouponFinishedClientTransactions($creditNote->coupon));
                }
            }
        }

        return $creditNotes->count();
    }
}