<?php

namespace App\Services;


use App\Account;
use App\Events\EmployeeAdded;
use App\Events\EmployeeEdited;
use App\User;
use Illuminate\Support\Facades\Auth;
use Spatie\Permission\Models\Permission;

class EmployeeService
{
    private $seller;
    
    private $employee = null;

    /**
     * EmployeeService constructor.
     * @param User|null $employee
     */
    public function __construct(User $employee = null)
    {
        $this->seller = Auth::user();
        $this->employee = $employee;
    }

    /**
     * Method to create new employee.
     *
     * @param $data
     * @return User
     */
    public function create($data)
    {
        $account = Account::create();
        $employee = User::create([
            'email'      => $data['email'],
            'salutation' => $data['salutation'],
            'firstname'  => $data['firstname'],
            'surname'    => $data['surname'],
            'account_id' => $account->id,
            'boss_id'    => $this->seller->id,
            'status'     => User::STATUS_PENDING,
            'alias'      => time(),
        ]);

        $employee->alias = $employee->id;
        $employee->save();

        $employee->assignRole('employee');

        $data['permissions'] = isset($data['permissions']) ? $data['permissions'] : [];

        foreach ($data['permissions'] as $permission) {
            $employee->givePermissionTo($permission);
        }

        event(new EmployeeAdded($this->seller, $employee, $data));

        $logData = [
            'email'       => $data['email'],
            'salutation'  => $data['salutation'],
            'firstname'   => $data['firstname'],
            'surname'     => $data['surname'],
            'permissions' => $data['permissions'],
        ];

        activity($this->seller)->register('seller added employee', null, $logData, User::class, $employee->id);
        activity($employee)->register('employee added', null, $logData);

        return $employee;
    }

    /**
     * Method to update employee data.
     *
     * @param $data
     * @return User|null
     */
    public function update($data)
    {
        $data['permissions'] = isset($data['permissions']) ? $data['permissions'] : [];

        $oldData = [
            'permissions' => $this->employee->permissions->pluck('name')->implode(','),
        ];

        $newData = [
            'permissions' => implode(',', $data['permissions']),
        ];

        if (User::STATUS_PENDING === $this->employee->status) {
            $oldData = array_merge($oldData, [
                'email'      => $this->employee->email,
                'salutation' => $this->employee->salutation,
                'firstname'  => $this->employee->firstname,
                'surname'    => $this->employee->surname,
            ]);

            $newData = array_merge($newData, [
                'email'      => $data['email'] ?? '',
                'salutation' => $data['salutation'] ?? '',
                'firstname'  => $data['firstname'] ?? '',
                'surname'    => $data['surname'] ?? '',
            ]);

            $this->employee->email      = $data['email'];
            $this->employee->salutation = $data['salutation'];
            $this->employee->firstname  = $data['firstname'];
            $this->employee->surname    = $data['surname'];

            $this->employee->save();
        }

        $permissionIDs = Permission::whereIn('name', isset($data['permissions']) ? $data['permissions'] : '')
            ->pluck('id')->toArray();
        $this->employee->permissions()->sync($permissionIDs);

        event(new EmployeeEdited($this->seller, $this->employee, $oldData, $data));

        activity($this->seller)->register('seller edited employee', $oldData, $newData, User::class, $this->employee->id);
        activity($this->employee)->register('employee edited by seller', $oldData, $newData);

        return $this->employee;
    }


    /**
     * Method to activate employee.
     */
    public function activate()
    {
        $this->employee->status = User::STATUS_ACTIVE;
        $this->employee->save();

        activity($this->seller)->register('seller activated employee', null, null, User::class, $this->employee->id);
        activity($this->employee)->register('employee activated');
    }

    /**
     * Method to deactivate employee.
     */
    public function deactivate()
    {
        $this->employee->status = User::STATUS_INACTIVE;
        $this->employee->save();

        activity($this->seller)->register('seller deactivated employee', null, null, User::class, $this->employee->id);
        activity($this->employee)->register('employee deactivated');
    }

    /**
     * Method to delete employee.
     */
    public function delete()
    {
        (new UserService($this->employee))->delete();

        activity($this->seller)->register('seller deleted employee', null, null, User::class, $this->employee->id);
        activity($this->employee)->register('employee deleted');
    }

    /**
     * Method to finish employee activation.
     *
     * @param $data
     */
    public function register($data)
    {
        $this->employee->status                   = User::STATUS_ACTIVE;
        $this->employee->password                 = bcrypt($data['password']);
        $this->employee->email_verified           = 1;
        $this->employee->email_verification_token = null;
        $this->employee->save();

        activity($this->employee)->register('employee registered');
    }
}