<?php

namespace App\Services;


use App\CouponTemplate;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Spatie\MediaLibrary\Media;

class DealService
{
    private $deal;

    private $attributesToSkip = [
        'id',
        'user_id',
        'deal_id',
        'expiration_date',
        'img',
        'status',
        'provision_gift_id',
        'discount_gift_id',
        'lat',
        'lon',
        'online',
        'offline',
        'redirect_link',
        'visits',
        'shares',
        'created_at',
        'updated_at',
    ];

    /**
     * MediaService constructor.
     * @param CouponTemplate $couponTemplate
     */
    public function __construct(CouponTemplate $couponTemplate)
    {
        $this->deal = $couponTemplate;
    }

    /**
     * Method to remove draft deals older than some value in minutes.
     *
     * @param int $minutes
     * @return mixed
     */
    public static function removeDraft($minutes = 180)
    {
        $deals = CouponTemplate::draft()
            ->olderThan($minutes)
            ->get();

        $dealsCount = $deals->count();

        $deals->each(function ($item) {
            $item->delete(); // delete each separately to fire observer event
        });

        return $dealsCount;
    }

    /**
     * Method to update videos collection.
     *
     * @param $data
     */
    private function updateVideos($data)
    {
        if ( isset($data['video']) ) {
            $videos = Media::find($data['video']);
            foreach ($videos as $video) {
                $video->collection_name = 'videos';
                $video->save();
            }
            $videosOld = Media::where('model_id', $this->deal->id)
                ->where('collection_name', 'videos')
                ->whereNotIn('id', $data['video'])
                ->get();
            foreach ($videosOld as $video) {
                $video->delete();
            }
        } else {
            $this->deal->clearMediaCollection('videos');
        }

        $this->deal->clearMediaCollection('videos_draft');
    }

    /**
     * Method to update image gallery.
     *
     * @param $data
     */
    private function updateGallery($data)
    {
        if ( isset($data['gallery']) ) {
            $images = Media::find($data['gallery']);
            foreach ($images as $image) {
                $image->collection_name = 'images';
                $image->save();
            }

            $imagesToRemove = Media::where('model_id', $this->deal->id)
                ->where('collection_name', 'images')
                ->whereNotIn('id', $data['gallery'])
                ->get();
            foreach ($imagesToRemove as $image) {
                $image->delete();
            }

            Media::setNewOrder($data['gallery']);
        }

        $this->deal->clearMediaCollection('images_draft');
    }

    /**
     * Method to prepare attributes array for saving into log.
     *
     * @return array
     */
    private function prepareAttributes()
    {
        $attributes = $this->deal->getAttributes();

        foreach ($this->attributesToSkip as $key) {
            unset($attributes[$key]);
        }

        $attributes['places']  = $this->deal->places->pluck('id')->implode(',');
        $attributes['gallery'] = $this->deal->gallery->pluck('id')->implode(',');

        $attributes['cashback'] = $this->deal->type_discount . '|' . $this->deal->discount_consumer;
        $attributes['recocash'] = $this->deal->type_provision . '|' . $this->deal->provision_agent;

        return $attributes;
    }

    /**
     * Method to update deal from constructor.
     *
     * @param $data
     * @return
     */
    public function update($data)
    {
        $oldData = (CouponTemplate::STATUS_DRAFT === $this->deal->status)
            ? null
            : $this->prepareAttributes();

        $action = (CouponTemplate::STATUS_DRAFT === $this->deal->status)
            ? 'deal added'
            : 'deal edited';

        $this->deal->title             = $data['title'];
        $this->deal->expiration_date   = '2999-01-01';
        $this->deal->description       = $data['description'];
        $this->deal->terms             = $data['terms'];
        $this->deal->quota             = $data['quota'];
        $this->deal->validity          = $data['validity'];
        $this->deal->type_provision    = $data['type_provision'];
        $this->deal->type_discount     = $data['type_discount'];
        $this->deal->provision_agent   = (CouponTemplate::TYPE_GIFT === $data['type_provision']) ? 0 : $data['provision_agent'];
        $this->deal->discount_consumer = (CouponTemplate::TYPE_GIFT === $data['type_discount']) ? 0 : $data['discount_consumer'];
        $this->deal->provision_gift_id = (CouponTemplate::TYPE_GIFT === $data['type_provision']) ? $data['provision_gift_id'] : null;
        $this->deal->discount_gift_id  = (CouponTemplate::TYPE_GIFT === $data['type_discount']) ? $data['discount_gift_id'] : null;
        $this->deal->category_id       = $data['category_id'];
        $this->deal->online            = 0; // temporary
        $this->deal->offline           = 1; // temporary
        $this->deal->status            = CouponTemplate::STATUS_ACTIVE;

        $this->deal->places()->sync($data['places']);

        $this->updateVideos($data);
        $this->updateGallery($data);

        $this->deal->save();

        unset($data['deal_id']);

        $data['cashback'] = $this->deal->type_discount . '|' . $this->deal->discount_consumer;
        $data['recocash'] = $this->deal->type_provision . '|' . $this->deal->provision_agent;

        activity(Auth::user())->register($action, $oldData, $data, CouponTemplate::class, $this->deal->id);

        return $this->deal;
    }

    /**
     * Method to activate deal.
     */
    public function activate()
    {
        $this->deal->status = CouponTemplate::STATUS_ACTIVE;
        $this->deal->save();

        activity(Auth::user())->register('deal activated', null, null, CouponTemplate::class, $this->deal->id);
    }

    /**
     * Method to deactivate deal.
     */
    public function deactivate()
    {
        $this->deal->status = CouponTemplate::STATUS_INACTIVE;
        $this->deal->save();

        activity(Auth::user())->register('deal deactivated', null, null, CouponTemplate::class, $this->deal->id);
    }

    /**
     * Method to delete deal.
     */
    public function delete()
    {
        $this->deal->status = CouponTemplate::STATUS_DELETED;
        $this->deal->save();

        activity(Auth::user())->register('deal deleted', null, null, CouponTemplate::class, $this->deal->id);
    }
}