<?php

namespace App\Services;

use App\City;
use App\Library\Utils;

class CityService
{
    /**
     * Method to find N rows by some search string.
     *
     * @param $searchString
     * @param int $limit
     * @return mixed
     */
    public function find($searchString, $limit = 5)
    {
        $searchString = Utils::prepareSearchString($searchString);

        $citiesByNameStart = City::where('name', 'like', $searchString . '%')
            ->limit($limit)
            ->get()
            ->unique('loc_id');

        $citiesByNameEverywhere = City::where('name', 'like', '%' . $searchString . '%')
            ->limit($limit)
            ->get()
            ->unique('loc_id');

        $citiesByStringStart = City::where('s', 'like', $searchString . '%')
            ->limit($limit)
            ->get()
            ->unique('loc_id');

        $citiesByStringEverywhere = City::where('s', 'like', '%' . $searchString . '%')
            ->limit($limit)
            ->get()
            ->unique('loc_id');

        $cities = $citiesByNameStart
            ->merge($citiesByNameEverywhere)
            ->merge($citiesByStringStart)
            ->merge($citiesByStringEverywhere);

        return $cities->unique('loc_id')->slice(0, $limit);
    }

    /**
     * Method to save city search into session.
     */
    public function saveSearchToSession()
    {
        if ( request('plz') && request('city') && request('loc_id') && request('lat') && request('lon') ) {
            $geoSearches = session('geo_searches', collect([]));
            if ( !$geoSearches->contains('loc_id', request('loc_id')) ) {
                $geoSearches->prepend([
                    'loc_id' => request('loc_id'),
                    'plz'    => request('plz'),
                    'city'   => request('city'),
                    'lat'    => request('lat'),
                    'lon'    => request('lon'),
                ]);
            }
            $geoSearches = $geoSearches->slice(0, 5);
            session(['geo_searches' => $geoSearches]);
        }
    }

    /**
     * Method to find city by location identifier.
     *
     * @param $locId
     * @return mixed
     */
    public function getCityByLocId($locId)
    {
        return City::where('loc_id', $locId)->first();
    }

    /**
     * Method to find closest city by geocoordinates.
     *
     * @param $latitude
     * @param $longitude
     * @return mixed
     */
    public function getCityByCoordinates($latitude, $longitude)
    {
        if ( !$latitude || !$longitude ) {
            return null;
        }

        return City::selectRaw('*, ( 6371 * acos( cos( radians(' . $latitude . ') ) * cos( radians( lat ) ) * cos( radians( lon ) - radians(' . $longitude . ') ) + sin( radians(' . $latitude . ') ) * sin(radians(lat)) ) ) AS distance')
            ->orderBy('distance')
            ->first();
    }

    /**
     * Method to find city by request.
     *
     * @return mixed|null
     */
    public function getCity()
    {
        if ( (int)request('loc_id') ) {
            return $this->getCityByLocId((int)request('loc_id'));
        }

        if ( (float)request('lat') && (float)request('lon') ) {
            return $this->getCityByCoordinates((float)request('lat'), (float)request('lon'));
        }

        return null;
    }
}