<?php

namespace App\Services;


use AqBanking\Account;
use AqBanking\Bank;
use AqBanking\BankCode;
use AqBanking\Command\AddUserCommand;
use AqBanking\Command\AddUserFlagsCommand;
use AqBanking\Command\GetAccountsCommand;
use AqBanking\Command\GetSysIDCommand;
use AqBanking\Command\ListUsersCommand;
use AqBanking\Command\RenderContextFileToXMLCommand;
use AqBanking\Command\RequestCommand;
use AqBanking\Command\SetITanModeCommand;
use AqBanking\ContextFile;
use AqBanking\ContextXmlRenderer;
use AqBanking\HbciVersion;
use AqBanking\PinFile\PinFile;
use AqBanking\PinFile\PinFileCreator;
use AqBanking\User;
use AqBanking\UserMatcher;
use Carbon\Carbon;
use DateTime;
use Illuminate\Support\Facades\Storage;

class AqBankingService
{
    private $aqbBankCode;
    private $hbciVersion;
    private $aqbBank;
    private $aqbAccount;
    private $aqbUser;
    private $aqbCurrentUID;
    private $aqbPinFile;
    private $aqbITANMode;

    /**
     * AqBankingService constructor.
     */
    public function __construct()
    {
        $this->aqbBankCode = new BankCode(config('banking.bankCodeString'));
        $this->hbciVersion = new HbciVersion('300');
        $this->aqbBank     = new Bank($this->aqbBankCode, config('banking.aqBankUrl'), $this->hbciVersion);
        $this->aqbAccount  = new Account($this->aqbBankCode, config('banking.accountNumber'));
        $this->aqbUser     = new User(config('banking.userId'), config('banking.accountOwner'), $this->aqbBank);
        $this->aqbPinFile  = new PinFile($this->aqbGetStoragePath(), $this->aqbUser);
        $this->aqbITANMode = config('banking.ITANMode');

        $this->aqbInit();
    }

    /**
     * Method to get current user ID.
     */
    private function aqbGetCurrentUID()
    {
        $listUsers = new ListUsersCommand();
        $userList = $listUsers->execute();
        $userMatcher = new UserMatcher($userList);
        $this->aqbCurrentUID = $userMatcher->getExistingUser($this->aqbUser);
    }

    /**
     * Method to add new user.
     *
     * @throws AddUserCommand\UserAlreadyExistsException
     * @throws \AqBanking\Command\ShellCommandExecutor\DefectiveResultException
     */
    private function aqbAddUser()
    {
        $addUser = new AddUserCommand();
        $addUser->execute($this->aqbUser);
    }

    /**
     * Method to get current user (create if doesn't exist).
     *
     * @throws AddUserCommand\UserAlreadyExistsException
     * @throws \AqBanking\Command\ShellCommandExecutor\DefectiveResultException
     */
    private function aqbGetUser()
    {
        $this->aqbGetCurrentUID();

        // if user doesn't yet exist
        if ( null === $this->aqbCurrentUID ) {
            $this->aqbAddUser();
            $this->aqbGetCurrentUID();

            // additional check
            if ( null === $this->aqbCurrentUID ) {
                throw new \RuntimeException('User not found, even after creating');
            }
        }
    }

    /**
     * Method to add flags to current user.
     *
     * @param $flags
     * @throws AddUserCommand\UserAlreadyExistsException
     * @throws \AqBanking\Command\ShellCommandExecutor\DefectiveResultException
     */
    private function aqbAddFlags($flags)
    {
        $addUserFlags = new AddUserFlagsCommand();
        foreach ($flags as $flag) {
            $addUserFlags->execute($this->aqbCurrentUID, $flag);
        }
    }

    /**
     * Method to get storage path (create folder if doesn't exist).
     *
     * @return string
     */
    private function aqbGetStoragePath()
    {
        $path = storage_path('app/aqbanking');

        if ( !file_exists($path) ) {
            mkdir($path, 0770, true);
        }

        return $path;
    }

    /**
     * Method to create PIN file if it doesn't exist.
     */
    private function aqbInitPin()
    {
        $createPinFile = new PinFileCreator($this->aqbGetStoragePath());
        $createPinFile->createFile(config('banking.pin'), $this->aqbUser);
    }

    /**
     * Method to get sys ID.
     *
     * @throws \AqBanking\Command\ShellCommandExecutor\DefectiveResultException
     */
    private function aqbGetSysID()
    {
        $getSysId = new GetSysIDCommand();
        $getSysId->execute($this->aqbCurrentUID, $this->aqbPinFile);
    }

    /**
     * Method to set ITAN mode.
     *
     * @throws AddUserCommand\UserAlreadyExistsException
     * @throws \AqBanking\Command\ShellCommandExecutor\DefectiveResultException
     */
    private function aqbSetITAN()
    {
        $setITanMode = new SetITanModeCommand();
        $setITanMode->execute($this->aqbCurrentUID, $this->aqbITANMode);
    }

    /**
     * Method to get bank accounts.
     *
     * @throws \AqBanking\Command\ShellCommandExecutor\DefectiveResultException
     */
    private function aqbGetAccounts()
    {
        $getAccounts = new GetAccountsCommand();
        $getAccounts->execute($this->aqbCurrentUID, $this->aqbPinFile);
    }

    /**
     * Method to initialize AqBanking.
     */
    private function aqbInit()
    {
        $this->aqbGetUser();
        //$this->aqbAddFlags([AddUserFlagsCommand::FLAG_SSL_QUIRK_IGNORE_PREMATURE_CLOSE]);
        $this->aqbInitPin();
        $this->aqbGetSysID();
        $this->aqbSetITAN();
        $this->aqbGetAccounts();
    }

    /**
     * Method to get transactions.
     *
     * @param DateTime|null $from
     * @return
     * @throws \AqBanking\Command\ShellCommandExecutor\DefectiveResultException
     */
    public function getTransactions(\DateTime $from = null)
    {
        $storagePath = $this->aqbGetStoragePath();
        $contextFile = new ContextFile($storagePath . '/aqBanking.ctx');

        $runRequest = new RequestCommand($this->aqbAccount, $contextFile, $this->aqbPinFile);
        $runRequest->execute($from);

        $render = new RenderContextFileToXMLCommand();
        $dom = $render->execute($contextFile);
        $result = new ContextXmlRenderer($dom);

        Storage::copy('aqbanking/aqBanking.ctx', 'aqbanking/old/aqBanking_' . Carbon::now()->format('Y-m-d-H-i-s') . '.ctx');

        return $result->getTransactions();
    }

    /**
     * Method to get balance.
     */
    public function getBalance()
    {
        // TODO get balance
    }
}