<?php

namespace App\Services;

use App\Account;
use App\AccountConnection;
use App\LinkInvitations;
use App\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Redirect;

class AbstractRegistrationService
{
    protected $user = null;
    protected $account = null;
    protected $data = null;

    public function __construct($data)
    {
        if ( isset($data['referred_by']) ) {
            $data['referral_status'] = User::REFERRAL_STATUS_ACTIVE;
        }

        $this->account = new Account();
        $this->account->fill($data);
        $this->account->status = User::STATUS_ACTIVE;

        $email_verification_token = strtoupper(str_random(5));

        $this->user = new User();
        $data['alias'] = time();
        $data['newsletter'] = isset($data['newsletter']) ? 1 : 0;
        $this->user->fill($data);

        $this->user->status = User::STATUS_ACTIVE;

        $this->user->email_verified = 0;
        if (isset($data['password'])) {
            $this->user->password = bcrypt( $data['password'] );
            unset($data['password']);
        }
        $this->user->email_verification_token = $email_verification_token;

        $this->user->account()->associate($this->account);

        $this->data = $data;
    }

    protected function create()
    {
        $this->account->save();
        $this->user->account()->associate($this->user->account); // reload ID
        $this->user->save();
        $this->user->alias = $this->user->id;
        $this->user->save();

        activity($this->user)->register('registered', null, $this->data);
    }

    /**
     * check if email is already present in database
     * @param $email
     * @return \App\User or Null
     */
    public static function emailTaken($email)
    {
        return User::where('email', '=', $email)->first();
    }

    /**
     * checks if the invitation is valid
     *  -recieving user is not inviting user
     *  -recieving user can't exist and become seller
     * @param $data array containing requestdata
     * @return mixed
     */
    public static function checkInvitation($data)
    {
        if ($linkInvitation = LinkInvitations::where('token', '=', $data['token'])->first()) {
            if ($user = self::emailTaken($data['email'])) {
                if ($user->id == $linkInvitation->inviting_user_id) {
                    return Redirect::to('/')->with('notification', array(
                        'type' => 'error',
                        'title' => 'Fehler',
                        'description' => 'Sich selbst einen Coupon zu empfehlen ist verboten.'
                    ));
                }
                if ($linkInvitation->reason == 'recommend-recoma') {
                    return Redirect::to('/')->with('notification', array(
                        'type' => 'error',
                        'title' => 'Fehler',
                        'description' => 'Bestehende Benutzer können keine Unternehmen werden.'
                    ));
                }
            }
        }
        return $linkInvitation;
    }

    /**
     * Add invitation information
     *
     * @param LinkInvitations $invitation
     */
    public function setRecruiterFromInvitation(LinkInvitations $invitation)
    {
        $this->setRecruiter($invitation->invitingUser);
    }

    /**
     * Add recruiter information
     *
     * @param Account $account
     */
    public function setRecruiter(User $user)
    {
        $this->account->recruiter_user_id = $user->id;
    }

    /**
     * @return User|null
     */
    public function getUser()
    {
        return $this->user;
    }

    /**
     * @return Account|null
     */
    public function getAccount()
    {
        return $this->account;
    }
}