<?php

namespace App\Providers;

use App\Account;
use App\CouponTemplate;
use App\FinishedJobs;
use App\Http\Requests\Request;
use App\Observer\AccountObserver;
use App\Observer\ModelStatusObserver;
use App\Services\PasswordService;
use App\Services\TFAService;
use App\User;
use App\Coupon;
use App\Transaction;

use App\Observer\CouponObserver;
use App\Observer\CouponTemplateObserver;
use App\Observer\TransactionObserver;
use App\Observer\UserObserver;

use App\UsersGift;
use Illuminate\Support\Facades\Auth;
use Queue;
use Illuminate\Support\ServiceProvider;
use Illuminate\Queue\Events\JobProcessed;
use Illuminate\Auth\Passwords\DatabaseTokenRepository as DbRepository;
use Illuminate\Support\Facades\Validator;
use Validator as Factory;
use ZxcvbnPhp\Zxcvbn;

class AppServiceProvider extends ServiceProvider
{

    /**
     * Bootstrap any application services.
     *
     * @return void
     */
    public function boot()
    {
        $this->customValidations();

        Coupon::observe(new CouponObserver());
        CouponTemplate::observe(new CouponTemplateObserver());
        User::observe(new UserObserver());
        Transaction::observe(new ModelStatusObserver());
        Coupon::observe(new ModelStatusObserver());
        User::observe(new ModelStatusObserver());
        Transaction::observe(new TransactionObserver());
        Account::observe(new AccountObserver());
        UsersGift::observe(new ModelStatusObserver());

        Queue::after(function (JobProcessed $event){
            $event->connectionName;
            $event->job;
            $event->data;

            $data['view'] = isset($event->data['data']['view']) ? $event->data['data']['view'] : '';
            $data['data'] = isset($event->data['data']['callback']) ? $event->data['data']['callback'] : '';
            $data['receiver'] = $event->data['data']['data']['toUser']['email'] ?? '';
            if ($data['view'] && $data['data']) {
                FinishedJobs::create($data);
            }
        });
    }

    /**
     * Register any application services.
     *
     * @return void
     */
    public function register()
    {
        $this->app->singleton('password.resets.tokens', function($app)
        {
            $connection = $app['db']->connection();

            $table = 'password_resets';

            $key = $app['config']['app.key'];

            $expire = 60 * 24 * 7;

            return new DbRepository($connection, $table, $key, $expire);
        });
    }

    public function customValidations()
    {
        // custom validators

        Validator::extend('email_or_phone_number', function($attribute, $value, $parameters, $validator) {

            $mobileNumber = Factory::make([$attribute => $value], [
                $attribute => Request::MOBILE_NUMBER
            ]);

            $email = Factory::make([$attribute => $value], [
                $attribute => Request::MOBILE_NUMBER
            ]);

            if($mobileNumber->fails() || $email->fails()) {
                return false;
            }

            return $value;
        });

        Validator::extend('category_is_leaf', function ($attribute, $value, $parameters, $validator) {
            return \App\Category::find($value)->isLeaf();
        });

        Validator::extend('belongs_to', function ($attribute, $value, $parameters, $validator) {
            $uid = (int)$parameters[0];
            return $uid === (int)\App\CouponTemplate::find($value)->user_id;
        });

        Validator::extend('active_deal', function ($attribute, $value, $parameters, $validator) {
            return \App\CouponTemplate::find($value)->isActive();
        });

        Validator::extend('min_words', function ($attribute, $value, $parameters, $validator) {
            $words = preg_split('@\s+@i', trim($value));
            if ( count($words) >= (int)$parameters[0] ) {
                return true;
            }
            return false;
        });

        Validator::extend('password_strength', function ($attribute, $value, $parameters, $validator) {
            $strength = (new Zxcvbn())->passwordStrength($value);

            return ($strength['score'] >= (int)$parameters[0]);
        });

        Validator::extend('tfa_valid', function ($attribute, $value, $parameters, $validator) {
            $user = Auth::user();
            return (new TFAService($user))->isTokenValid($value);
        });

        // replacers

        Validator::replacer('min_words', function($message, $attribute, $rule, $parameters) {
            return str_replace(':min_words', $parameters[0], $message);
        });
    }
}