<?php

namespace App\Presenters;

use Carbon\Carbon;
use App\CouponTemplate;
use App\Library\Facades\Format;
use Illuminate\Support\HtmlString;

class CouponTemplatePresenter extends AbstractPresenter
{
    /**
     * @var array Type icons.
     */
    public static $typeIcons = [
        CouponTemplate::TYPE_AMOUNT => '€',
        CouponTemplate::TYPE_PERCENT => '%',
        CouponTemplate::TYPE_GIFT => '',
        CouponTemplate::TYPE_NONE => '',
    ];

    /**
     * @var array Type names.
     */
    public static $typeNames = [
        CouponTemplate::TYPE_AMOUNT => 'Betrag',
        CouponTemplate::TYPE_PERCENT => 'Prozent',
        CouponTemplate::TYPE_GIFT => 'Geschenk',
        CouponTemplate::TYPE_NONE => 'Keine',
    ];

    /**
     * @var array Status titles.
     */
    public static $statusTitles = [
        CouponTemplate::STATUS_ACTIVE   => 'Aktiv',
        CouponTemplate::STATUS_INACTIVE => 'Inaktiv',
        CouponTemplate::STATUS_DRAFT    => 'Draft',
    ];

    /**
     * @var array Status badges (bootstrap 3, 4 compatible)
     */
    public static $statusBadges = [
        CouponTemplate::STATUS_ACTIVE   => 'success',
        CouponTemplate::STATUS_INACTIVE => 'danger',
        CouponTemplate::STATUS_DRAFT    => 'secondary',
    ];

    /**
     * Method to present status.
     *
     * @return HtmlString
     */
    public function presentStatus()
    {
        $badge = self::$statusBadges[$this->entity->status];
        $title = self::$statusTitles[$this->entity->status];
        return new HtmlString('<span class="badge badge-pill badge-' . $badge . '">&nbsp;</span>&nbsp;<span>' . $title . '</span>');
    }

    /**
     * Method to present status badge.
     *
     * @return HtmlString
     */
    public function presentStatusBadge()
    {
        $badge = self::$statusBadges[$this->entity->status];
        return new HtmlString('<span class="badge badge-pill badge-' . $badge . '">&nbsp;</span>');
    }

    /**
     * Method to present status badge (metronic theme).
     *
     * @return HtmlString
     */
    public function presentStatusMetronic()
    {
        $badge = self::$statusBadges[$this->entity->status];
        $title = self::$statusTitles[$this->entity->status];
        return new HtmlString('<span class="kt-badge kt-badge--' . $badge . ' kt-badge--inline kt-badge--pill">' . $title . '</span>');
    }

    /**
     * Method to present deal type
     *
     * @return mixed
     */
    public function presentType()
    {
        return array_get(self::$typeIcons, $this->entity->type);
    }

    /**
     * Method to present agent provision.
     *
     * @return HtmlString|string
     */
    public function presentProvisionAgent()
    {
        if (CouponTemplate::TYPE_NONE === $this->entity->type_provision) {
            return 'Keine';
        } else if (CouponTemplate::TYPE_GIFT === $this->entity->type_provision) {
            return new HtmlString(view('redesign.front.pieces.tables.gift_cell', ['gift' => $this->entity->provisionGift]));
        }
        return Format::fee($this->entity->provision_agent, $this->entity->type_provision);
    }

    /**
     * Method to present cashback.
     *
     * @return HtmlString
     */
    public function presentDiscountConsumer()
    {
        if (CouponTemplate::TYPE_GIFT === $this->entity->type_discount) {
            return new HtmlString(view('redesign.front.pieces.tables.gift_cell', ['gift' => $this->entity->discountGift]));
        }
        return Format::fee($this->entity->discount_consumer, $this->entity->type_discount);
    }

    /**
     * Method to present agent provision for deal view.
     *
     * @return HtmlString|string
     */
    public function presentProvisionAgentDeal()
    {
        if (CouponTemplate::TYPE_NONE === $this->entity->type_provision) {
            return 'Keine';
        } else if (CouponTemplate::TYPE_GIFT === $this->entity->type_provision) {
            return new HtmlString('<div class="deal-gift"><i class="fas fa-gift"></i></div>');
        }
        return new HtmlString(str_replace(',00', '', Format::fee($this->entity->provision_agent, $this->entity->type_provision)));
    }

    /**
     * Method to present cashback for deal view.
     *
     * @return HtmlString
     */
    public function presentDiscountConsumerDeal()
    {
        if (CouponTemplate::TYPE_GIFT === $this->entity->type_discount) {
            return new HtmlString('<div class="deal-gift"><i class="fas fa-gift"></i></div>');
        }
        return new HtmlString(str_replace(',00', '', Format::fee($this->entity->discount_consumer, $this->entity->type_discount)));
    }

    /**
     * Method to present agent provision for text view.
     *
     * @return HtmlString|string
     */
    public function presentProvisionAgentText()
    {
        if (CouponTemplate::TYPE_NONE === $this->entity->type_provision) {
            return 'Keine';
        } else if (CouponTemplate::TYPE_GIFT === $this->entity->type_provision) {
            return 'Geschenk';
        } else if (CouponTemplate::TYPE_PERCENT === $this->entity->type_provision) {
            return str_replace(',00', '', $this->entity->provision_agent) . ' %';
        }
        return str_replace(',00', '', $this->entity->provision_agent) . ' €';
    }

    /**
     * Method to present cashback for text view.
     *
     * @return HtmlString
     */
    public function presentDiscountConsumerText()
    {
        if (CouponTemplate::TYPE_GIFT === $this->entity->type_discount) {
            return 'Geschenk';
        } else if (CouponTemplate::TYPE_PERCENT === $this->entity->type_discount) {
            return str_replace(',00', '', $this->entity->discount_consumer) . ' %';
        }
        return str_replace(',00', '', $this->entity->discount_consumer) . ' €';
    }

    /**
     * Method to present agent provision for invite view.
     *
     * @return HtmlString|string
     */
    public function presentProvisionAgentInvite()
    {
        if (CouponTemplate::TYPE_NONE === $this->entity->type_provision) {
            return 'Keine';
        } else if (CouponTemplate::TYPE_GIFT === $this->entity->type_provision) {
            $gift = $this->entity->provisionGift;
            return new HtmlString('<img src="' . asset($gift->image) . '">');
        }
        // TODO fix crutch
        return new HtmlString(str_replace(',00', '', Format::fee($this->entity->provision_agent, $this->entity->type_provision)));
    }

    /**
     * Method to present cashback for invite view.
     *
     * @return HtmlString|string
     */
    public function presentDiscountConsumerInvite()
    {
        if (CouponTemplate::TYPE_GIFT === $this->entity->type_discount) {
            $gift = $this->entity->discountGift;
            return new HtmlString('<img src="' . asset($gift->image) . '">');
        }
        // TODO fix crutch
        return new HtmlString(str_replace(',00', '', Format::fee($this->entity->discount_consumer, $this->entity->type_discount)));
    }

    /**
     * Method to present cashback for map marker view.
     *
     * @return HtmlString
     */
    public function presentDiscountConsumerMapMarker()
    {
        if (CouponTemplate::TYPE_GIFT === $this->entity->type_discount) {
            $gift = $this->entity->discountGift;
            return new HtmlString('<span class="showGift" data-image="' . $gift->image . '" data-name="' . $gift->name . '">' . $gift->name . '</span>');
        }
        // TODO fix crutch
        return new HtmlString(str_replace(',00', '', Format::fee($this->entity->discount_consumer, $this->entity->type_discount)));
    }

    /**
     * Method to present cashback information.
     *
     * @return HtmlString
     */
    public function presentDiscountConsumerInfo()
    {
        if ( CouponTemplate::TYPE_GIFT === $this->entity->type_discount ) {
            return new HtmlString('Ein <strong>Geschenk</strong>');
        }
        return new HtmlString('<strong>' . $this->entity->discount_consumer . self::$typeIcons[$this->entity->type_discount] . '</strong>');
    }

    /**
     * Method to present distance.
     *
     * @return string
     */
    public function presentDistance()
    {
        return number_format($this->entity->distance, 2) . ' km';
    }

    /**
     * Method to present creation date.
     *
     * @return string
     */
    public function presentExpirationDate()
    {
        return Carbon::parse($this->entity->expiration_date)->format('d.m.Y');
    }
}