<?php

namespace App\Presenters;

use App\Coupon;
use App\CouponTemplate;
use App\Library\Facades\Format;
use Carbon\Carbon;
use Illuminate\Support\HtmlString;

class CouponPresenter extends AbstractPresenter
{
    /**
     * @var array Status titles.
     */
    public static $statusTitles = [
        Coupon::STATUS_PENDING          => 'Empfohlen',
        Coupon::STATUS_ACTIVE           => 'Empfohlen',
        Coupon::STATUS_REDEEMED         => 'Eingelöst',
        Coupon::STATUS_PAYMENT_ASSERTED => 'Eingelöst',
        Coupon::STATUS_PAID             => 'Bezahlt',
        Coupon::STATUS_DONE             => 'Abgeschlossen',
    ];

    /**
     * @var array Status badges (bootstrap 3, 4 compatible)
     */
    public static $statusBadges = [
        Coupon::STATUS_PENDING          => 'secondary',
        Coupon::STATUS_ACTIVE           => 'secondary',
        Coupon::STATUS_REDEEMED         => 'warning',
        Coupon::STATUS_PAYMENT_ASSERTED => 'warning',
        Coupon::STATUS_PAID             => 'primary',
        Coupon::STATUS_DONE             => 'success',
    ];

    /**
     * Method to present status.
     *
     * @return HtmlString
     */
    public function presentStatus()
    {
        $badge = self::$statusBadges[$this->entity->status];
        $title = self::$statusTitles[$this->entity->status];
        return new HtmlString('<span class="badge badge-pill badge-' . $badge . '">&nbsp;</span>&nbsp;<span>' . $title . '</span>');
    }

    /**
     * Method to present status badge.
     *
     * @return HtmlString
     */
    public function presentStatusBadge()
    {
        $badge = self::$statusBadges[$this->entity->status];
        return new HtmlString('<span class="badge badge-pill badge-' . $badge . '">&nbsp;</span>');
    }

    /**
     * Method to present status badge (metronic theme).
     *
     * @return HtmlString
     */
    public function presentStatusMetronic()
    {
        $badge = self::$statusBadges[$this->entity->status];
        $title = self::$statusTitles[$this->entity->status];
        return new HtmlString('<span class="kt-badge kt-badge--' . $badge . ' kt-badge--inline kt-badge--pill">' . $title . '</span>');
    }

    /**
     * Method to present agent provision.
     *
     * @return HtmlString|string
     */
    public function presentProvisionAgent()
    {
        if ( CouponTemplate::TYPE_NONE === $this->entity->type_provision ) {
            return 'Keine';
        } else if ( CouponTemplate::TYPE_GIFT === $this->entity->type_provision ) {
            return new HtmlString(view('redesign.front.pieces.tables.gift_cell', ['gift' => $this->entity->provisionGift]));
        }
        return Format::fee($this->entity->provision_agent, $this->entity->type_provision);
    }

    /**
     * Method to present cashback.
     *
     * @return HtmlString|string
     */
    public function presentDiscountConsumer()
    {
        if ( CouponTemplate::TYPE_NONE === $this->entity->type_discount ) {
            return 'Keine';
        } else if ( CouponTemplate::TYPE_GIFT === $this->entity->type_discount ) {
            return new HtmlString(view('redesign.front.pieces.tables.gift_cell', ['gift' => $this->entity->discountGift]));
        }
        return Format::fee($this->entity->discount_consumer, $this->entity->type_discount);
    }

    /**
     * Method to present agent provision for coupon view.
     *
     * @return HtmlString|string
     */
    public function presentProvisionAgentCoupon()
    {
        if ( CouponTemplate::TYPE_NONE === $this->entity->type_provision ) {
            return 'Keine';
        } else if ( CouponTemplate::TYPE_GIFT === $this->entity->type_provision ) {
            return new HtmlString('<div class="deal-gift"><i class="fas fa-gift"></i></div>');
        }
        return new HtmlString(str_replace(',00', '', Format::fee($this->entity->provision_agent, $this->entity->type_provision)));
    }

    /**
     * Method to present cashback for coupon view.
     *
     * @return HtmlString
     */
    public function presentDiscountConsumerCoupon()
    {
        if ( CouponTemplate::TYPE_GIFT === $this->entity->type_discount ) {
            return new HtmlString('<div class="deal-gift"><i class="fas fa-gift"></i></div>');
        }
        return new HtmlString(str_replace(',00', '', Format::fee($this->entity->discount_consumer, $this->entity->type_discount)));
    }

    /**
     * Method to present coupon code.
     *
     * @return mixed
     */
    public function presentCode()
    {
        return implode(" ", str_split($this->entity->code, 3));
    }

    /**
     * Method to present netto sum.
     *
     * @return string
     */
    public function presentNetto()
    {
        return (float)$this->entity->netto ? Format::currency($this->entity->netto) : '';
    }

    /**
     * Method to present redeem date.
     *
     * @return mixed
     */
    public function presentRedeemDate()
    {
        return $this->entity->status_redeemed_at ? Carbon::parse($this->entity->status_redeemed_at)->format('d.m.Y') : '';
    }
}