<?php

namespace App\Presenters;


use App\Category;
use App\CouponTemplate;
use App\Place;
use Carbon\Carbon;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\HtmlString;

class ActivityPresenter extends AbstractPresenter
{
    /**
     * Color fields.
     *
     * @var array
     */
    private $colorFields = [
        'main_bg',
        'main_txt',
        'main_bg_h',
        'main_txt_h',
    ];

    /**
     * Translatable fields.
     *
     * @var array
     */
    private $transFields = [
        'payment',
    ];

    /**
     * Currency fields.
     *
     * @var array
     */
    private $currencyFields = [
        'netto',
    ];

    /**
     * Fields to skip
     *
     * @var array
     */
    private $skipFields = [
        'lat',
        'lon',
        'seller_id',
        'type_discount',
        'type_provision',
        'discount_consumer',
        'provision_agent',
    ];

    /**
     * Default ordering value.
     *
     * @var int
     */
    private $defaultOrdering = 0;

    /**
     * Field ordering values.
     *
     * @var array
     */
    private $fieldsOrder = [
        // place
        'plz'         => 10,
        'ort'         => 20,
        'street'      => 30,
        'housenumber' => 40,
        'email'       => 50,
        'phone'       => 60,
        'wt_mon'      => 100,
        'wt_tue'      => 110,
        'wt_wed'      => 120,
        'wt_thu'      => 130,
        'wt_fri'      => 140,
        'wt_sat'      => 150,
        'wt_sun'      => 160,
        // deal
        'title'       => 10,
        'description' => 20,
        'quota'       => 30,
        'validity'    => 40,
        'terms'       => 50,
        'category_id' => 60,
        'cashback'    => 70,
        'recocash'    => 80,
        'places'      => 90,
        'gallery'     => 100,
    ];

    /**
     * Method to present creation date.
     *
     * @return string
     */
    public function presentCreationDate()
    {
        return Carbon::parse($this->entity->created_at)->format('d.m.Y h:i');
    }

    /**
     * Permissions list chages.
     *
     * @param $k
     * @param $v
     * @return string
     * @throws \Throwable
     */
    private function presentPermissionsChanges($k, $v)
    {
        $old = isset($this->entity->changes->old->{$k}) ? array_filter(explode(',', $this->entity->changes->old->{$k})) : [];
        $new = is_array($v) ? $v : array_filter(explode(',', $v));

        return view('redesign.back.metronic.employees.pieces.activity.changes.permissions', [
            'title'   => trans('actions.' . $k),
            'added'   => array_diff($new, $old),
            'removed' => array_diff($old, $new),
        ])->render();
    }

    /**
     * Invite widget deal changes.
     *
     * @param $k
     * @param $v
     * @return string
     * @throws \Throwable
     */
    private function presentInviteWidgetDealChanges($k, $v)
    {
        $oldId = $this->entity->changes->old->{$k} ?? 0;

        $old = Cache::remember('deal_' . $oldId, 15, function() use ($oldId) {
            return CouponTemplate::find($oldId);
        });

        $new = Cache::remember('deal_' . $v, 15, function() use ($v) {
            return CouponTemplate::find($v);
        });

        return view('redesign.back.metronic.employees.pieces.activity.changes.widget_deal', [
            'title' => trans('actions.' . $k),
            'old'   => $old,
            'new'   => $new,
        ])->render();
    }

    /**
     * Deal places changes.
     *
     * @param $k
     * @param $v
     * @return string
     * @throws \Throwable
     */
    private function presentPlacesChanges($k, $v)
    {
        $old = isset($this->entity->changes->old->{$k}) ? explode(',', $this->entity->changes->old->{$k}) : [];
        $new = is_array($v) ? $v : explode(',', $v);

        $places = Place::find(array_unique(array_merge($old, $new)))->keyBy('id');

        return view('redesign.back.metronic.employees.pieces.activity.changes.places', [
            'title'   => trans('actions.' . $k),
            'added'   => array_diff($new, $old),
            'removed' => array_diff($old, $new),
            'places'  => $places,
        ])->render();
    }

    /**
     * Deal gallery changes.
     *
     * @param $k
     * @param $v
     * @return string
     * @throws \Throwable
     */
    private function presentGalleryChanges($k, $v)
    {
        $old = isset($this->entity->changes->old->{$k}) ? explode(',', $this->entity->changes->old->{$k}) : [];
        $new = is_array($v) ? $v : explode(',', $v);

        return view('redesign.back.metronic.employees.pieces.activity.changes.gallery', [
            'title'   => trans('actions.' . $k),
            'added'   => count(array_diff($new, $old)),
            'removed' => count(array_diff($old, $new)),
        ])->render();
    }

    /**
     * Color field change.
     *
     * @param $k
     * @param $v
     * @return string
     * @throws \Throwable
     */
    private function presentColorChanges($k, $v)
    {
        return view('redesign.back.metronic.employees.pieces.activity.changes.color', [
            'title' => trans('actions.' . $k),
            'old'   => $this->entity->changes->old->{$k} ?? '',
            'new'   => $v,
        ])->render();
    }

    /**
     * Get string for provision according type.
     *
     * @param $string
     * @return string
     */
    private function getProvisionString($string)
    {
        list($type, $value) = explode('|', $string ?? '');

        if (CouponTemplate::TYPE_NONE === $type) {
            return 'Keine';
        } else if (CouponTemplate::TYPE_PERCENT === $type) {
            return str_replace(',00', '', $value) . ' %';
        } else {
            return str_replace(',00', '', $value) . ' €';
        }
    }

    /**
     * Deal provision (cashback or RECO.CASH).
     *
     * @param $k
     * @param $v
     * @return string
     * @throws \Throwable
     */
    private function presentProvisionChanges($k, $v)
    {
        $old = isset($this->entity->changes->old->{$k}) ? $this->getProvisionString($this->entity->changes->old->{$k}) : '';
        $new = $this->getProvisionString($v);

        return view('redesign.back.metronic.employees.pieces.activity.changes.provision', [
            'title' => trans('actions.' . $k),
            'old'   => $old,
            'new'   => $new,
        ])->render();
    }

    /**
     * Deal category.
     *
     * @param $k
     * @param $v
     * @return string
     * @throws \Throwable
     */
    private function presentCategoryChanges($k, $v)
    {
        $oldId = $this->entity->changes->old->{$k} ?? 0;

        $old = Cache::remember('deal_' . $oldId, 15, function() use ($oldId) {
            return Category::find($oldId);
        });

        $new = Cache::remember('deal_' . $v, 15, function() use ($v) {
            return Category::find($v);
        });

        return view('redesign.back.metronic.employees.pieces.activity.changes.category', [
            'title' => trans('actions.' . $k),
            'old'   => $old,
            'new'   => $new,
        ])->render();
    }

    /**
     * Any scalar field change.
     *
     * @param $k
     * @param $v
     * @return string
     * @throws \Throwable
     */
    private function presentScalarChanges($k, $v)
    {
        return view('redesign.back.metronic.employees.pieces.activity.changes.scalar', [
            'title' => trans('actions.' . $k),
            'old'   => $this->entity->changes->old->{$k} ?? '',
            'new'   => $v,
        ])->render();
    }

    /**
     * Any currency field change.
     *
     * @param $k
     * @param $v
     * @return string
     * @throws \Throwable
     */
    private function presentCurrencyChanges($k, $v)
    {
        return view('redesign.back.metronic.employees.pieces.activity.changes.scalar', [
            'title' => trans('actions.' . $k),
            'old'   => isset($this->entity->changes->old->{$k}) ? (str_replace(',00', '', $this->entity->changes->old->{$k}) . ' €') : '',
            'new'   => str_replace(',00', '', $v) . ' €',
        ])->render();
    }

    /**
     * Any translatable field change.
     *
     * @param $k
     * @param $v
     * @return string
     * @throws \Throwable
     */
    private function presentTranslatableChanges($k, $v)
    {
        return view('redesign.back.metronic.employees.pieces.activity.changes.scalar', [
            'title' => trans('actions.' . $k),
            'old'   => isset($this->entity->changes->old->{$k}) ? trans('actions.' . $this->entity->changes->old->{$k}) : '',
            'new'   => trans('actions.' . $v),
        ])->render();
    }

    /**
     * Method to sort keys.
     */
    private function orderFields()
    {
        $arr = (array)$this->entity->changes->new;

        uksort($arr, function ($a, $b) {
            $aValue = isset($this->fieldsOrder[$a]) ? $this->fieldsOrder[$a] : $this->defaultOrdering;
            $bValue = isset($this->fieldsOrder[$b]) ? $this->fieldsOrder[$b] : $this->defaultOrdering;
            return ($aValue <=> $bValue);
        });

        return $arr;
    }

    /**
     * Method to present changes.
     *
     * @return mixed
     * @throws \Throwable
     */
    public function presentChanges()
    {
        $html = '';

        $newArr = $this->orderFields();

        foreach ($newArr as $k => $v) {
            if ( in_array($k, $this->skipFields) ) {
                // do nothing
            } else if ('permissions' === $k) {
                $html .= $this->presentPermissionsChanges($k, $v);
            } else if ('places' === $k) {
                $html .= $this->presentPlacesChanges($k, $v);
            } else if ('gallery' === $k) {
                $html .= $this->presentGalleryChanges($k, $v);
            } else if ('category_id' === $k) {
                $html .= $this->presentCategoryChanges($k, $v);
            } else if ('cashback' === $k || 'recocash' === $k) {
                $html .= $this->presentProvisionChanges($k, $v);
            } else if ( in_array($k, $this->transFields) ) {
                $html .= $this->presentTranslatableChanges($k, $v);
            } else if ( in_array($k, $this->currencyFields) ) {
                $html .= $this->presentCurrencyChanges($k, $v);
            } else if ( 'invite widget edited' === $this->entity->action->action && 'deal_id' === $k ) {
                $html .= $this->presentInviteWidgetDealChanges($k, $v);
            } else if ( in_array($k, $this->colorFields) ) {
                $html .= $this->presentColorChanges($k, $v);
            } else {
                $html .= $this->presentScalarChanges($k, $v);
            }
        }

        return new HtmlString($html);
    }
}