<?php

namespace App\Observer;

use App\Coupon;
use App\CouponTemplate;
use App\Http\Grids\Tr;
use App\Transaction;
use App\User;
use Exception;

class CouponObserver {

    public function saved(Coupon $coupon)
    {
        switch($coupon->status) {
            case Coupon::STATUS_REDEEMED:
                $this->createTransactions($coupon);
                $this->updateTransactionsStatus($coupon, Transaction::STATUS_PENDING);
                break;
        }
    }

    public function saving(Coupon $coupon)
    {
        switch($coupon->status){
            case Coupon::STATUS_PAID:
                foreach($coupon->paymentTransactions as $paymentTransaction) {
                    $paymentTransaction->status = Transaction::STATUS_ALLOCATED;
                    $paymentTransaction->save();
                }
                $coupon->createInvoice();
                break;
        }
    }

    protected function updateTransactionsStatus($coupon, $status)
    {
        $transactions = Transaction::where('coupon_id','=',$coupon->id)->get();
        foreach($transactions as $transaction) {
            $transaction->update(['status' => $status]);
        }
    }
    
    protected function createTransactions($coupon)
    {
        $agent    = User::find($coupon->agent_user_id);
        $consumer = User::find($coupon->consumer_user_id);

        if ($consumer->referred_by && CouponTemplate::TYPE_NONE !== $coupon->type_provision) {
            if (User::REFERRAL_STATUS_HAS_COUPONS === $consumer->referral_status) {
                $consumer->referral_status = User::REFERRAL_STATUS_HAS_REDEEMED;
                $consumer->save();
            }

            $otherCoupons = Coupon::where('id', '!=', $coupon->id)
                ->where('consumer_user_id', $consumer->id)
                ->where('agent_user_id', $agent->id)
                ->get();
            foreach ($otherCoupons as $c) {
                $c->agent_user_id     = $c->consumer_user_id;
                $c->provision_agent   = 0;
                $c->type_provision    = CouponTemplate::TYPE_NONE;
                $c->provision_gift_id = null;
                $c->save();
            }
        }

        $transactionDefaultData = [
            'user_id'   => null,
            'coupon_id' => $coupon->id,
            'amount'    => 0,
            'type'      => null,
            'status'    => Transaction::STATUS_PENDING,
        ];

        $operatorAmount       = config('recoma.additionalCharge');
        $agentAmount          = 0;
        $consumerAmount       = 0;
        $sellerAgentAmount    = 0;
        $sellerConsumerAmount = 0;

        if (CouponTemplate::TYPE_PERCENT === $coupon->type_provision) {
            $agentAmount        = $coupon->netto * $coupon->provision_agent / 100;
            $sellerAgentAmount += $agentAmount;
        } else if (CouponTemplate::TYPE_AMOUNT === $coupon->type_provision) {
            $agentAmount        = $coupon->provision_agent;
            $sellerAgentAmount += $agentAmount;
        } else if (CouponTemplate::TYPE_GIFT === $coupon->type_provision) {
            $sellerAgentAmount += $coupon->provision_agent;
        }

        if (CouponTemplate::TYPE_PERCENT === $coupon->type_discount) {
            $consumerAmount        = $coupon->netto * $coupon->discount_consumer / 100;
            $sellerConsumerAmount += $consumerAmount;
        } else if (CouponTemplate::TYPE_AMOUNT === $coupon->type_discount) {
            $consumerAmount        = $coupon->discount_consumer;
            $sellerConsumerAmount += $consumerAmount;
        } else if (CouponTemplate::TYPE_GIFT === $coupon->type_discount) {
            $sellerConsumerAmount += $coupon->discount_consumer;
        }

        $operatorAmount += $agentAmount / 100 * config('recoma.amount_commision');
        $operatorAmount += $consumerAmount / 100 * config('recoma.amount_commision');

        Transaction::create(array_merge($transactionDefaultData,[
            'type'    => Transaction::TYPE_AGENT_COUPON_PROVISION_CREDIT,
            'user_id' => $coupon->agent_user_id,
            'amount'  => $agentAmount,
        ]));

        Transaction::create(array_merge($transactionDefaultData,[
            'type'    => Transaction::TYPE_CONSUMER_COUPON_DISCOUNT_CREDIT,
            'user_id' => $coupon->consumer_user_id,
            'amount'  => $consumerAmount,
        ]));

        Transaction::create(array_merge($transactionDefaultData,[
            'type'    => Transaction::TYPE_SELLER_AGENT_PROVISION_PAYMENT,
            'user_id' => $coupon->template->user_id,
            'amount'  => $sellerAgentAmount * -1,
        ]));

        Transaction::create(array_merge($transactionDefaultData,[
            'type'    => Transaction::TYPE_SELLER_CONSUMER_DISCOUNT_PAYMENT,
            'user_id' => $coupon->template->user_id,
            'amount'  => $sellerConsumerAmount * -1,
        ]));

        Transaction::create(array_merge($transactionDefaultData,[
            'type'    => Transaction::TYPE_SELLER_OPERATOR_PROVISION_PAYMENT,
            'user_id' => $coupon->template->user_id,
            'amount'  => $operatorAmount * -1,
        ]));
    }
}