<?php

namespace App\Library\Grids;

use App\Exceptions\NotFoundException;
use App\Http\Grids\Tr;
use App\Library\GeoLocation;
use App\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Nayjest\Grids\Components\Base\RenderableRegistry;
use Nayjest\Grids\Components\ColumnHeadersRow;
use Nayjest\Grids\Components\HtmlTag;
use Nayjest\Grids\Components\Laravel5\Pager;
use Nayjest\Grids\Components\OneCellRow;
use Nayjest\Grids\Components\RecordsPerPage;
use Nayjest\Grids\Components\TFoot;
use Nayjest\Grids\Components\THead;
use Nayjest\Grids\EloquentDataProvider;
use Nayjest\Grids\FilterConfig;
use Nayjest\Grids\SelectFilterConfig;

abstract class DefaultGrid extends \Nayjest\Grids\Grid
{
    /**
     * Unique internal grid name (at least unique per page) Customize Name here
     *
     * @var string
     */
    protected $_name = 'grid';

    /**
     * Filters spanning the whole grid or even a full page of grids
     *
     * @var
     */
    protected $globalFilters;

    public function __construct()
    {
        $config = $this->getGridConfig();

        $config->setComponents([
            $this->getGridHeader(),
            $this->getGridFooter()
        ])
            ->setRowComponent(new Tr($this));

        return parent::__construct($config);
    }

    /**
     * Allow to finally modify the query, for example to add joins
     * @return mixed
     */

    protected function getGridFinalQuery()
    {
        return $this->getGridQuery();
    }

    /**
     * @return GridConfig
     */
    protected function getGridConfig()
    {
        return (new GridConfig)
            ->setDataProvider($this->getDataProvider())
            ->setName($this->_name)
            ->setColumns($this->getGridColumns());
    }

    protected function getDataProvider()
    {
        return new EloquentDataProvider($this->getGridFinalQuery());
    }

    /**
     * @return array
     */
    protected function getGridColumns()
    {
        return [];
    }

    /**
     * @return mixed
     */
    abstract protected function getGridQuery();

    public function getTableRowCssClass($row)
    {
        return '';
    }

    protected function getGridFooter()
    {
        return (new TFoot)
            ->setComponents([
                (new OneCellRow)
                    ->setRenderSection(RenderableRegistry::SECTION_END)
                    ->setComponents([
                        (new Pager),
                        (new RecordsPerPage)->setVariants([
                            15,
                            50,
                            100,
                            300,
                            1000
                        ]),
                        (new HtmlTag)
                            ->setContent('<span class="glyphicon glyphicon-refresh"></span>')
                            ->setTagName('button')
                            ->setRenderSection(RenderableRegistry::SECTION_END)
                            ->setAttributes([
                                'class' => 'btn btn-primary btn-sm'
                            ])
                    ])
            ]);
    }

    protected function getGridHeader()
    {
        return (new THead)
            ->setComponents([
                (new ColumnHeadersRow),
            ]);
    }

    public function date($input, $provider)
    {
        if ($input != 'DD.MM.YYYY') {
            if (preg_match('/[0-9]{1,2}[.][0-9]{1,2}[.][0-9]{4}/', $input)) {
                $inputDate = explode('.', $input);

                $date = Carbon::createFromDate($inputDate[2], $inputDate[1], $inputDate[0], 'Europe/Berlin');
                dd($date);

                $date = $inputDate[2] . '-' . $inputDate[1] . '-' . $inputDate[0];

                $provider->getBuilder()
                    ->where('date', 'like', '%' . Carbon::parse() . '%');
            };
        }
    }

    /**
     * Copy global filters to other grids
     *
     * @param Grid|array $target
     */
    public function exportGlobalFilters($target)
    {
        if (!is_array($target)) {
            $target = [$target];
        }

        foreach ($this->getGlobalFilters() as $filterKey => $globalFilter) {
            $globalValue = $globalFilter->getValue();
            foreach ($target as $targetGrid) {
                $targetGrid->getGlobalFilters()[$filterKey]->setValue($globalValue);
            }
        }
    }

    /**
     * Global filters accross multiple columns
     *
     * @return array
     */
    public function getGlobalFiltersConfig()
    {
        return [];
    }

    /**
     * Cached version of build global filters, overwrite build global filters
     * @return array
     */
    public function getGlobalFilters()
    {
        if ($this->globalFilters == null) {
            $this->globalFilters = $this->getGlobalFiltersConfig();
        }

        return $this->globalFilters;
    }

    public function prepare()
    {
        $return = parent::prepare();

        foreach ($this->getGlobalFilters() as $filter) {
            $filter->apply();
        }

        return $return;
    }

    /**
     * Library function which can come in handy for different grids
     * @FIXME: maybe as a trait ;-) ?
     * @return mixed
     */
    public function libraryAccountFullTextFilterConfig()
    {
        $config =
            (new AutoCompleteFilterConfig)
                ->setLabel('Unternehmen')
                ->setName('fulltext')
                ->setOperator(FilterConfig::OPERATOR_LIKE)
                ->setFilteringFunc(function ($val, EloquentDataProvider $provider) {
                    $provider->getBuilder()
                        ->where(function ($query) use ($val) {
                            $query->where('accounts.company', 'like', '%' . $val . '%')
                                ->orWhere(DB::raw('CONCAT(users.firstname, " ", users.surname)'), 'like', '%' . $val . '%');
                        });
                });

        return $config;
    }

    public function libraryUserFilterConfig()
    {
        $config =
            (new AutoCompleteFilterConfig)
                ->setLabel('Unternehmen / E-Mail')
                ->setName('fulltext')
                ->setOperator(FilterConfig::OPERATOR_LIKE)
                ->setFilteringFunc(function ($val, EloquentDataProvider $provider) {
                    $provider->getBuilder()
                        ->where(function ($query) use ($val) {
                            $query->where('accounts.company', 'like', '%' . $val . '%')
                                ->orWhere(DB::raw('CONCAT(users.firstname, " ", users.surname)'), 'like', '%' . $val . '%')
                                ->orWhere('users.email', 'like', '%' . $val . '%');
                        });
                });

        return $config;
    }

    public function libraryGroupFilterConfig()
    {
        $config =
            (new SelectFilterConfig)
                ->setName('id')
                ->setLabel('User')
                ->setSubmittedOnChange(true)
                ->setOptions([
                    '1' => trans('translations.operator'),
                    '2' => trans('translations.seller'),
                    '4' => trans('translations.agent'),
                    '5' => trans('translations.consumer'),
                ])
                ->setFilteringFunc(function ($val, EloquentDataProvider $provider) {
                    $provider->getBuilder()
                        ->leftJoin('user_has_roles as user_has_roles', 'users.id', '=', 'user_has_roles.user_id')
                        ->where('user_has_roles.role_id', '=', $val)// $val is value entered (id of the group), not the user id in this case
                        ->groupBy('users.id');
                });

        return $config;
    }

    public function libraryPlaceConfig()
    {
        $config =
            (new PlaceRadiusFilterConfig)
                ->setLabel('PLZ / Ort')
                ->setName('place')
                ->setFilteringFunc(function ($val, EloquentDataProvider $provider) {
                    $value = trim($val['place']);
                    if (empty($value)) {
                        return;
                    }

                    try {
                        $location = GeoLocation::getInstanceByString($value);

                        $radius = is_numeric($val['radius']) ? $val['radius'] : 10;
                        if ($location != null) {
                            $provider->getBuilder()->whereRaw($location->getRadiusQuery($radius));
                        }
                    } catch (NotFoundException $e) {
                        // search textual
                        $provider->getBuilder()
                            ->where(function ($query) use ($value) {
                                $query->where('accounts.zip', 'like', '%' . $value . '%')
                                    ->orWhere('accounts.city', 'like', '%' . $value . '%');
                            });
                    }
                });

        return $config;
    }

    public function libraryTransactionStatusConfig()
    {
        $config =
            (new SelectFilterConfig)
                ->setName('transactions.status')
                ->setLabel('Status')
                ->setSubmittedOnChange(true)
                ->setOptions([
                    'pending' => trans('translations.pending'),
                    'active' => trans('translations.active'),
                    'redeemed' => trans('translations.redeemed'),
                    'paid' => trans('translations.paid'),
                    'done' => trans('translations.done'),
                ]);

        return $config;
    }

    public function libraryAgentTransactionCompanyAndConsumerConfig()
    {
        $config =
            (new FilterConfig)
                ->setLabel('Unternehmen / Verbraucher')
                ->setName('fulltext')
                ->setOperator(FilterConfig::OPERATOR_LIKE)
                ->setFilteringFunc(function ($val, EloquentDataProvider $provider) {
                    $provider->getBuilder()
                        ->where(function ($query) use ($val) {
                            $query->where('company', 'like', '%' . $val . '%')
                                ->orWhere(User::getDisplayNameDBExpression(false, 'consumers'), 'like', '%' . $val . '%');
                        });
                });

        return $config;
    }

    public function libraryAgentSharedCouponsStatusConfig()
    {
        $config =
            (new SelectFilterConfig)
                ->setName('coupons.status')
                ->setLabel('Status')
                ->setSubmittedOnChange(true)
                ->setOptions([
                    'pending' => trans('translations.pending'),
                    'active' => trans('translations.active'),
                    'redeemed' => trans('translations.redeemed'),
                    'paid' => trans('translations.paid'),
                    'done' => trans('translations.done'),
                ]);

        return $config;
    }
}