<?php

namespace App\Library;


use App\Action;
use App\Location;
use App\Services\LocationService;
use App\Services\UserAgentService;
use App\User;
use App\Useragent;

class Activity
{
    private $user;

    /**
     * Activity constructor.
     * @param User $user
     */
    public function __construct(User $user)
    {
        $this->user = $user;
    }

    /**
     * Method to prepare changes for DB.
     *
     * @param $oldData
     * @param $newData
     * @return object
     */
    private function prepareChanges($oldData, $newData)
    {
        if (null === $oldData) {
            $oldData = [];
        }

        if (null === $newData) {
            $newData = [];
        }

        foreach ($oldData as &$item) {
            if ( is_array($item) ) {
                $item = implode(',', $item);
            }
        }

        foreach ($newData as &$item) {
            if ( is_array($item) ) {
                $item = implode(',', $item);
            }
        }

        unset($newData['_token']);

        $old = array_diff_assoc($oldData, $newData);
        $new = array_diff_assoc($newData, $oldData);

        if ($old === $new) {
            return false;
        }

        return (object)[
            'old' => $old,
            'new' => $new,
        ];
    }

    /**
     * Method to get action id. If action doesn't yet exist, create it at first.
     *
     * @param $action
     * @return mixed
     */
    private function getActionId($action)
    {
        return Action::firstOrCreate([
            'action' => $action,
        ])->id;
    }

    /**
     * Method to get location id. If location doesn't yet exist, create it with all details from 3rd party service.
     *
     * @param $ipAddress
     * @return mixed
     */
    private function getLocationId($ipAddress)
    {
        $location = Location::where('ip', $ipAddress)->first();

        if (null === $location) {
            $location = Location::create([
                'ip'      => $ipAddress,
                'details' => (new LocationService())->getDataByIPAddress($ipAddress),
            ]);
        }

        return $location->id;
    }

    /**
     * Method to get useragent id. If useragent doesn't yet exist, create it at first.
     *
     * @param $useragent
     * @return mixed
     */
    private function getUserAgentId($useragent)
    {
        $ua = Useragent::where('useragent', $useragent)->first();

        if (null === $ua) {
            $ua = Useragent::create([
                'useragent' => $useragent,
                'details'   => (new UserAgentService($useragent))->getAllData(),
            ]);
        }

        return $ua->id;
    }

    /**
     * Method to add new row into user log.
     *
     * @param $action
     * @param null $oldData
     * @param null $newData
     * @param null $model
     * @param null $modelId
     * @param bool $skipIfNothingChanged
     */
    public function register($action, $oldData = null, $newData = null, $model = null, $modelId = null, $skipIfNothingChanged = true)
    {
        if (null === $oldData && null === $newData) {
            $changes = null;
            $skipIfNothingChanged = false;
        } else {
            $changes = $this->prepareChanges($oldData, $newData);
        }

        if ( $changes || !$skipIfNothingChanged ) {
            \App\Activity::create([
                'user_id'      => $this->user->id,
                'action_id'    => $this->getActionId($action),
                'changes'      => $changes,
                'model_type'   => $model,
                'model_id'     => $modelId,
                'location_id'  => $this->getLocationId(request()->ip()),
                'useragent_id' => $this->getUserAgentId(request()->header('User-Agent')),
            ]);
        }
    }
}