<?php

namespace App;

use App\Library\Billomat\PDFDummyResponse;
use App\Library\Facades\Format;
use Digitick\Sepa\PaymentInformation;
use Digitick\Sepa\TransferFile\Factory\TransferFileFacadeFactory;
use Illuminate\Support\Facades\Log;
use Phobetor\Billomat\Exception\BadRequestException;
use Storage;

use Illuminate\Database\Eloquent\Model;
use App\Library\Billomat\BillomatClientHelper;

class Invoice extends AbstractReceipt
{

    const TYPE_RECURRING = 'recurring';
    const TYPE_INVOICE = 'invoice';

    protected $fillable =[
        'coupon_id',
        'billomat_id',
        'pdf_path',
        'status',
        'type',
        'user_id',
        'gross_amount',
        'number',
    ];

    public function coupon()
    {
        return $this->belongsTo('App\Coupon');
    }

    public function transactions()
    {
        return $this->hasMany('App\Transaction', 'invoice_id');
    }

    /**
     * Transfer coupon invoice to billomat
     * The coupon is taken from the relation in the invoice
     */
    public function transferCouponInvoiceToBillomat()
    {
        $this->transferInvoiceTransactionsToBillomat($this->coupon->paymentTransactions);
    }


    public function getInvoiceTitle()
    {
        if ($this->coupon) {
            return $this->coupon->title;
        }

        return 'Rechnung';
    }

    /**
     * @param $transactions Collection of Transactions or single
     * @return mixed
     */
    public function transferInvoiceTransactionsToBillomat($transactions)
    {
        $billomatClient = BillomatClientHelper::getBillomatClientInstance();

        if ($transactions instanceof \App\Transaction) {
            $transactions = [$transactions];
        }

        $seller = null;

        /**
         * @var $transaction \App\Transaction
         */
        foreach ($transactions as $transaction) {
            if (!($seller instanceof User)) {
                $seller = $transaction->user;
                break;
            }
        }

        $seller->transferToBillomat();

        $invoiceArray = [
            'invoice' => [
                'client_id' => (int)$seller->billomat_id,
                'title' => $this->getInvoiceTitle(),
            ]
        ];

        $ret = $billomatClient->createInvoice($invoiceArray);

        $this->billomat_id = $ret['invoice']['id'];

        //Values returned by billomat are not typesave
        $invoice_billomat_id = intval($this->billomat_id);

        //Create Invoice items (invoice positions)
        foreach ($transactions as $transaction) {
            $transaction->transferToBillomat($invoice_billomat_id);
        }

        foreach ($transactions as $transaction) {
            $transaction->save();
        }

        $this->status = $ret['invoice']['status'];

        $this->type = self::TYPE_INVOICE;

        $this->save();

        foreach ($transactions as $transaction) {
            $transaction->invoice()->associate($this)->save();
        }

        return $ret['invoice']['id'];
    }

    /**
     * Complete Invoice
     *
     * sets billomat status from draft to complete and also the status of this
     * instance.
     *
     * TODO: Static wrapper?
     */
    public function completeInvoice()
    {

        $billomatClient = BillomatClientHelper::getBillomatClientInstance();

        $arr = [
            'id' => intval($this->billomat_id),
            'complete' => [],
        ];

        if ($this->type == self::TYPE_INVOICE && config('billomat.pdf.type.invoice.templateid', null) != null) {
            $arr['complete']['template_id'] = (int)config('billomat.pdf.type.invoice.templateid');
        } else if ($this->type == self::TYPE_RECURRING && config('billomat.pdf.type.recurring.templateid', null) != null) {
            $arr['complete']['template_id'] = (int)config('billomat.pdf.type.recurring.templateid');
        }

        $billomatClient->completeInvoice($arr);

        $receiptData = $billomatClient->getInvoice([
            'id' => intval($this->billomat_id),
        ]);

        $this->gross_amount = $receiptData['invoice']['total_gross'];
        $this->number = $receiptData['invoice']['invoice_number'];

        //Todo
        $this->status = self::STATUS_OPEN;
        $this->save();
    }

    public function getPDF()
    {
        //Invoice has not to be in draft stage
        if ($this->status === self::STATUS_DRAFT) {
            return false;
        }

        $billomatClient = BillomatClientHelper::getBillomatClientInstance();


        $ret = $billomatClient->getInvoicePdf([
            'id' => intval($this->billomat_id),
            'format' => 'pdf'
        ]);


        return $ret;
    }

    public function storePDF()
    {


        $pdf = $this->getPDF();
        //BILLOMAT_INTERFACE_DOC_STORRAGE_BASE

        if ($pdf === false) {
            return false;
        }

        if ($this->type === self::TYPE_INVOICE) {
            $this->pdf_path = "invoice_" . $this->billomat_id . ".pdf";
        } else if ($this->type === self::TYPE_RECURRING) {
            $this->pdf_path = "recurring_inv_" . $this->billomat_id . ".pdf";
        }

        if (!($pdf instanceof PDFDummyResponse)) {
            BillomatClientHelper::storePDF('invoicepdf', $this->pdf_path, $pdf);
        }
    }

    public function getPDFLink()
    {

        if ($this->pdf_path === null) {
            $this->storePDF();
            $this->save();
        }

        return route('invoice-downloadpdf', ['id' => $this->id]);
    }

    public function transferAsRecurringInvoice($recurring_id, $user)
    {

        $arr = [
            'invoice' => [
                'client_id' => intval($user->billomat_id),
                'recurring_id' => $recurring_id,
            ],
        ];

        $billomatClient = BillomatClientHelper::getBillomatClientInstance();

        $ret = $billomatClient->createInvoice($arr);

        $this->billomat_id = $ret['invoice']['id'];
    }

    public static function generateSepaDebitXml($markExported = true)
    {
        $accountOwner = config('banking.accountOwner');
        $iban = config('banking.iban');
        $bic = config('banking.bic');
        $creditorId = config('banking.creditorId');

        $msgId = self::generateMsgId('deb');
        $directDebit = TransferFileFacadeFactory::createDirectDebit($msgId, $accountOwner, 'pain.008.003.02');

        $receipts = self::whereStatus(self::STATUS_OPEN)->get();

        if (count($receipts) == 0) {
            return false;
        }

        /**
         * @var $r AbstractReceipt
         */
        foreach ($receipts as $receipt) {
            $paymentId = 'payment' . $receipt->id;
            $purpose = $receipt->getPurpose();
            $directDebit->addPaymentInfo($paymentId, [
                'id' => $paymentId,
                'creditorName' => $accountOwner,
                'creditorAccountIBAN' => $iban,
                'creditorAgentBIC' => $bic,
                'seqType' => /*($receipt->isRecurring() ? PaymentInformation::S_RECURRING : */
                    PaymentInformation::S_ONEOFF /*)*/,
                'creditorId' => $creditorId,
            ]);

            // Add a Single Transaction to the named payment
            $directDebit->addTransfer($paymentId, array(
                'amount' => (float)$receipt->gross_amount,
                'debtorIban' => $receipt->user->account->iban,
                'debtorName' => $receipt->user->account->iban_owner,
                'debtorMandate' => $receipt->user->account->id,
                'debtorMandateSignDate' => Format::date($receipt->user->account->created_at),
                'remittanceInformation' => $purpose,
            ));

            if ($markExported) {
                $receipt->status = self::STATUS_EXPORTED;
                $receipt->export_run = $msgId;
                $receipt->save();
            }
        }

        $file = self::createSepaFileName($msgId);
        $xml = $directDebit->asXML();
        file_put_contents($file, $xml);
        return $file;
    }

    public function markPaidAtBillomat($date)
    {
        $billomatClient = BillomatClientHelper::getBillomatClientInstance();

        $arr = [
            'invoice-payment' => [
                'invoice_id' => intval($this->billomat_id),
                'mark_invoice_as_paid' => 1,
                'date' => $date,
                'type' => 'BANK_TRANSFER',
                'amount' => $this->gross_amount,
            ]
        ];
        try{
            $billomatClient->createInvoicePayment($arr);
        }catch(BadRequestException $e){
            Log::error($e);
        };
    }

    public function getSignedAmount()
    {
        return $this->gross_amount;
    }

}