<?php

namespace App\Http\Requests;

use App\Place;
use Format;
use Illuminate\Support\Facades\Auth;
use App\CouponTemplate;

class CouponTemplateRequest extends ApiRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        $user   = Auth::user();
        $seller = Auth::user();
        $dealId = (int)$this->get('deal_id') ?? 0;

        if ( $user->hasRole('employee') ) {
            $seller = $user->boss;

            if ( !($user->canDo('create deals') || $user->canDo('edit deals')) ) {
                return false;
            }
        }

        if ( $dealId && CouponTemplate::find($dealId) && $seller->id !== CouponTemplate::find($dealId)->user_id ) {
            return false;
        }

        $places = $this->get('places') ?? [];
        if ($places) {
            $items = Place::find($places)->keyBy('id');
            foreach ($places as $place) {
                if ( $items[$place]->seller_id !== $seller->id ) {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $rules = [
            'title'           => ['required', 'max:100'],
            'category_id'     => ['bail', 'required', 'integer', 'min:1', 'exists:categories,id', 'category_is_leaf'],
            'description'     => ['required', 'min_words:10'],
            'terms'           => ['required', 'min_words:2'],
            'quota'           => ['required', 'integer', 'between:1,10000'],
            'validity'        => ['required', 'integer', 'between:1,24'],
            'type_discount'   => ['required', 'in:' . implode(',', CouponTemplate::DISCOUNT_TYPES)],
            'type_provision'  => ['required', 'in:' . implode(',', CouponTemplate::PROVISION_TYPES)],
            'gallery'         => ['required', 'array', 'min:1', 'max:10'],
            'places'          => ['required', 'array'],
            'places.*'        => ['integer', 'min:1', 'exists:places,id'],
        ];

        $typeProvision = $this->get('type_discount');
        if ( CouponTemplate::TYPE_AMOUNT === $typeProvision ) {
            $rules['discount_consumer'] = ['required', 'numeric', 'min:1'];
        } elseif ( CouponTemplate::TYPE_PERCENT === $typeProvision ) {
            $rules['discount_consumer'] = ['required', 'numeric', 'between:1,100'];
        } elseif ( CouponTemplate::TYPE_GIFT === $typeProvision ) {
            $rules['discount_gift_id'] = ['required', 'integer', 'min:1', 'exists:gifts,id'];
        }

        $typeProvision = $this->get('type_provision');
        if ( CouponTemplate::TYPE_AMOUNT === $typeProvision ) {
            $rules['provision_agent'] = ['required', 'numeric', 'min:1'];
        } elseif ( CouponTemplate::TYPE_PERCENT === $typeProvision ) {
            $rules['provision_agent'] = ['required', 'numeric', 'between:1,100'];
        } elseif ( CouponTemplate::TYPE_GIFT === $typeProvision ) {
            $rules['provision_gift_id'] = ['required', 'integer', 'min:1', 'exists:gifts,id'];
        }

        $online = (int)$this->get('online');
        if ($online) {
            $rules['redirect_link'] = ['required', 'url'];
        }

        $offline = (int)$this->get('offline');
        if ($offline) {
            $rules['lat'] = ['required', 'numeric'];
            $rules['lon'] = ['required', 'numeric'];
        }

        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        return [
            'lat'               => 'Standort',
            'lon'               => 'Standort',
            'terms'             => 'Bedingungen',
            'places'            => 'Standorte',
            'gallery'           => 'Galerie',
            'redirect_link'     => 'Redirect URL',
            'discount_gift_id'  => 'Cashback Geschenk',
            'provision_gift_id' => 'RECO.CASH Geschenk',
            'expiration_date'   => 'Haltbarkeitsdatum',
        ];
    }

    /**
     * Get the error messages for the defined validation rules.
     *
     * @return array
     */
    public function messages()
    {
        return [
            'category_id.exists'           => 'Kategorie existiert nicht',
            'category_id.category_is_leaf' => 'Kategorie sollte Blattknoten sein',
            'gallery.required'             => 'Galerie muss mindestens ein Bild haben',
            'places.required'              => 'Man muss mindestens ein Standort auswählen',
            'description.min_words'        => 'Beschreibung muss mindestens :min_words Wörter haben.',
            'terms.min_words'              => 'Bedingungen muss mindestens :min_words Wörter haben.',
        ];
    }
}
