<?php

namespace App\Http\Grids;

use App\Account;
use App\Library\Facades\Format;
use App\Library\Grids\DummyFieldConfig;
use App\Library\Grids\FieldConfig;
use App\Library\Grids\GlobalFilter;
use App\User;
use App\Library\Grids\DefaultGrid;
use Auth;
use Nayjest\Grids\EloquentDataProvider;
use Nayjest\Grids\FilterConfig;
use Nayjest\Grids\Grid;
use Nayjest\Grids\SelectFilterConfig;

class UserGrid extends DefaultGrid
{
    protected $_name = 'useroverview';

    protected function getGridQuery()
    {
        $query = User::query()->withoutGlobalScopes()
            ->select([
                'users.*',
                'accounts.company as account_company',
                User::getDisplayNameDBExpression('user_display_name', 'users')
            ])
            ->join('accounts', 'account_id', '=', 'accounts.id')
            ->groupBy('accounts.id');

        return $query;
    }

    public function getGlobalFiltersConfig()
    {

        $filters = [];

        $filters['fulltext']
            = new GlobalFilter($this->libraryUserFilterConfig(), new DummyFieldConfig(), $this, 'fulltext');

        $filters['place'] = new GlobalFilter($this->libraryPlaceConfig(), new DummyFieldConfig(), $this, 'place');

        $filters['group'] = new GlobalFilter($this->libraryGroupFilterConfig(), new DummyFieldConfig(), $this, 'group');

        return $filters;
    }

    protected function getGridColumns()
    {
        $gridColumns = parent::getGridColumns();

        $gridColumns[] =
            (new FieldConfig)
                ->setName('account_company')
                ->setLabel('Unternehmen')
                ->setSortable(true);
        $gridColumns[] =
            (new FieldConfig)
                ->setName('user_display_name')
                ->setLabel('Name')
                ->setSortable(true)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                        ->setFilteringFunc(function ($val, EloquentDataProvider $provider) {
                            $provider->getBuilder()
                                ->where(User::getDisplayNameDBExpression(false, 'users'), 'like', '%' . $val . '%');
                        })
                )
                ->setCallback(function ($val, $row) {
                    // FIXME: weitere Profile auflisten
                    return $row->getSrc()->getDisplayName();
                });
        $gridColumns[] =
            (new FieldConfig)
                ->setName('email')
                ->setLabel('E-Mail')
                ->setSortable(true)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                );

        $gridColumns['group'] = (new FieldConfig)
            ->setName('id')
            ->setLabel('Usergruppe')
            ->addFilter(
                (new SelectFilterConfig)
                    ->setName('GroupFilter')
                    ->setSubmittedOnChange(true)
                    ->setOptions([
                        '1' => trans('translations.operator'), // FIXME (LOW PRIO): we should not hard code the role IDs here
                        '2' => trans('translations.seller'),
                        '4' => trans('translations.agent'),
                        '5' => trans('translations.consumer'),
                        '6' => trans('translations.fieldservice_employee'),
                    ])
                    ->setOperator(FilterConfig::OPERATOR_EQ)
                    ->setFilteringFunc(function ($val, EloquentDataProvider $provider) {
                        $provider->getBuilder()
                            ->leftJoin('user_has_roles as user_has_roles', 'users.id', '=', 'user_has_roles.user_id')
                            ->where('user_has_roles.role_id', '=', $val)// $val is value entered, not the user id in this case
                            ->groupBy('users.id');

                    })
            )
            ->setCallback(function ($val) {
                if(auth::user()->hasRole('fieldservice_employee')){
                    $rowUser = User::withoutGlobalScopes()->find($val);
                }else{
                    $rowUser = User::find($val);
                }
                $rolesText  = [];
                foreach ($rowUser->roles as $role) {
                    if ( $role->name != 'seller-active') {
                        $rolesText[] = trans('translations.' . $role->name);
                    }
                }
                return implode(', ', $rolesText);
            });

        $gridColumns[] = (new FieldConfig)
            ->setName('id')
            ->setLabel('Bewertungen')
            ->setCallback(function ($val, $row) {
                $user = $row->getSrc();
                if ($user->hasRole('seller')) {
                    $account = Account::withoutGlobalScopes()->find($user->account_id);
                    return Format::rating($account->averageRating);
                }
                return '';
            });

        $gridColumns[] = (new FieldConfig)
            ->setName('status')
            ->setLabel('Status')
            ->setSortable(true)
            ->addFilter(
                (new SelectFilterConfig)
                    ->setName('users.status')
                    ->setSubmittedOnChange(true)
                    ->setOptions(User::getStatusOptionArray())
            )
            ->setCallback(function ($val, $row) {
                return $row->getSrc()->getStatusText();
            });

        $gridColumns['actions'] = (new FieldConfig)
            ->setName('id')
            ->setLabel(' ')
            ->setCellHtmlAttributes(['class' => 'button-cell'])
            ->setSortable(false)
            ->setCallback(function ($val) {
                return '<a class="btn btn-white " href="' . route('user-edit-id', $val) . '"><i class="fa fa-pencil"></i></a>';
            });

        // Sort by creation date, still allow to sort by other fields
        $gridColumns[] = (new FieldConfig)
            ->setName('users.created_at')
            ->hide()
            ->setSorting(Grid::SORT_DESC);

        return $gridColumns;
    }
}