<?php

namespace App\Http\Grids;

use App\CouponTemplate;
use App\Library\Facades\Format;
use App\Library\Grids\DummyFieldConfig;
use App\Library\Grids\FieldConfig;
use App\Library\Grids\GlobalFilter;
use App\Library\Grids\GridConfig;
use App\Transaction;
use App\User;
use App\Library\Grids\DefaultGrid;
use App\Coupon;
use Auth;
use Illuminate\Support\Facades\DB;
use Nayjest\Grids\EloquentDataProvider;
use Nayjest\Grids\FilterConfig;

class TransactionGrid extends DefaultGrid
{
    public function __construct()
    {
        return parent::__construct($this->getGridConfig());
    }

    public function getGlobalFiltersConfig()
    {
        if (Auth::user()->can('show operator transactions')) {
            $name = 'Unternehmen / Verbraucher / Empfehler';
        } elseif (Auth::user()->can('show seller transactions')) {
            $name = 'Verbraucher / Empfehler';
        } elseif (Auth::user()->can('show agent transactions')) {
            $name = 'Unternehmen / Verbraucher';
        } else {
            $name = 'Unternehmen / Empfehler';
        }

        $config = (new FilterConfig)
            ->setLabel($name)
            ->setName('fulltext')
            ->setOperator(FilterConfig::OPERATOR_LIKE)
            ->setFilteringFunc(function ($val, EloquentDataProvider $provider) {
                $provider->getBuilder()
                    ->where(function ($query) use ($val) {
                        if (Auth::user()->can('show operator transactions')) {
                            $query->where('account.company', 'like', '%' . $val . '%')
                                ->orWhere(User::getDisplayNameDBExpression(false, 'consumer'), 'like', '%' . $val . '%')
                                ->orWhere(User::getDisplayNameDBExpression(false, 'agent'), 'like', '%' . $val . '%');
                        } elseif (Auth::user()->can('show seller transactions')) {
                            $query->where(User::getDisplayNameDBExpression(false, 'consumer'), 'like', '%' . $val . '%')
                                ->orWhere(User::getDisplayNameDBExpression(false, 'agent'), 'like', '%' . $val . '%');
                        } elseif (Auth::user()->can('show agent transactions')) {
                            $query->where('account.company', 'like', '%' . $val . '%')
                                ->orWhere(User::getDisplayNameDBExpression(false, 'consumer'), 'like', '%' . $val . '%');
                        } else {
                            $query->where('account.company', 'like', '%' . $val . '%')
                                ->orWhere(User::getDisplayNameDBExpression(false, 'agent'), 'like', '%' . $val . '%');
                        }
                    });
            });

        $filters['fulltext'] = new GlobalFilter($config, new DummyFieldConfig(), $this, 'fulltext');
        $filters['status'] = new GlobalFilter($this->libraryTransactionStatusConfig(), new DummyFieldConfig(), $this, 'status');

        return $filters;
    }

    protected function getGridConfig()
    {
        return (new GridConfig)
            ->setDataProvider(
                new EloquentDataProvider($this->getGridQuery())
            )
            ->setName('Transaktionen')
            ->setColumns($this->getGridColumns());
    }

    protected function getGridQuery()
    {
        $query = Transaction::query()
            ->select([
                'transactions.*',
                'consumer.surname as consumer_surname',
                'consumer.firstname as consumer_firstname',
                'accounts.company as company',
                'accounts.id as account_id',
                'coupons.title',
                'coupons.code',
                User::getDisplayNameDBExpression('prospective_customer', 'consumer'),
                User::getDisplayNameDBExpression('recommender', 'agent'),
                User::getDisplayNameDBExpression('user_name', 'user'),
                'account.company'
            ])
            ->leftjoin('coupons', 'coupon_id', '=', 'coupons.id')
            ->leftjoin('users as consumer', 'coupons.consumer_user_id', '=', 'consumer.id')
            ->leftjoin('users as agent', 'coupons.agent_user_id', '=', 'agent.id')
            ->leftjoin('accounts', 'consumer.account_id', '=', 'accounts.id')
            ->join('users as user', 'user_id', '=', 'user.id')
            ->join('accounts as account', 'user.account_id', '=', 'account.id')
            ->groupBy(DB::raw('IF(coupon_id IS NULL, -transactions.id, coupon_id)'))
            ->orderBy('transactions.created_at','desc');

        if (!Auth::user()->can('show operator transactions')) {
            $query->whereUserId(Auth::user()->id);
        }

        return $query;
        // FIXME: Uninion with whereNull('coupon_id');
    }

    protected function getGridColumns()
    {
        $gridColumns = [];

        if (Auth::user()->can('show consumer transactions')
            || Auth::user()->can('show agent transactions')
            || Auth::user()->can('show operator transactions')
            || Auth::user()->can('show fieldservice transactions')
        ) {
            $gridColumns[] = (new FieldConfig)
                ->setName('company')
                ->setLabel('Unternehmen')
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                        ->setFilteringFunc(function ($val, EloquentDataProvider $provider) {
                            $provider->getBuilder()
                                ->where(User::getDisplayNameDBExpression(false, 'user'), 'like', '%' . $val . '%')
                                ->orWhere('account.company', 'like', '%' . $val . '%');
                        })
                )
                ->setCallback(function ($val, $row) {
                    if ($row->getSrc()->coupon) {
                        //check if user is null from globalScope!
                        if(!$row->getSrc()->coupon->template->seller){
                            return trans('translations.not-active');
                        }else{
                            return $row->getSrc()->coupon->template->seller->account->getProfileLink(User::SELLER);
                        }

                    } else if ($row->getSrc()->recruitedAccount) {
                        return $row->getSrc()->recruitedAccount->getProfileLink(User::SELLER);
                    } else {
                        return $row->getSrc()->user->account->getProfileLink(User::SELLER);
                    }
                });

        }


/*
        if(Auth::user()->hasRole('consumer')){//consumer
            $gridColumns[] = (new FieldConfig)
                ->setName('id')
                ->setLabel('Chatten')
                ->setCellHtmlAttributes(['class' => 'button-celll'])
                ->setSortable(false)
                ->setCallback(function ($val, $row) {
                    if ($val !== null && $row->getSrc()->coupon->template->seller) { // nicht "ohne Anfrage verfügbar"
                        $sellerId= $row->getSrc()->coupon->template->seller->id;
                        return '<a href="javascript:;" onclick="jqac.arrowchat.chatWith('.$sellerId.');"><img class="chat-image-small" src="/img/icons/chat-bubble.svg"> </a>';
                    }
                    return '';

                });
        }
*/

        $gridColumns[] = (new FieldConfig)
            ->setName('_coupon_code')
            ->setSortable(false)
            ->setLabel('Buchungstext')
            ->addFilter(
                (new FilterConfig)
                    ->setOperator(FilterConfig::OPERATOR_LIKE)
            )
            ->setCallback(function ($val, $row) {
                if ($row->getSrc()->coupon) {
                    return 'RECO.CODE: ' . $row->getSrc()->coupon->getCodeFormatted();
                } else {
                    return $row->getSrc()->typeToString();
                }
            });

        // possible FIXME operator should just have the seller transactions right?
        if (Auth::user()->can('show seller transactions') || Auth::user()->can('show operator transactions')) {
            $gridColumns[] = (new FieldConfig)
                ->setName('transaction_fee')
                ->setLabel('Gebühr')
                ->setSortable(false)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val, $row) {
                    if ($row->getSrc()->coupon) {
                        return $row->getSrc()->coupon->getTransactionAmountFormatted(Transaction::TYPE_SELLER_OPERATOR_PROVISION_PAYMENT);
                    }
                });
        }

        if (Auth::user()->can('show seller transactions') || Auth::user()->can('show consumer transactions') || Auth::user()->can('show operator transactions')) {

            $gridColumns[] = (new FieldConfig)
                ->setName('discount_consumer')
                ->setLabel('Cashback')
                ->setSortable(false)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val, $row) {
                    if ($row->getSrc()->coupon) {
                        if ( (Auth::user()->hasRole(User::AGENT)) && CouponTemplate::TYPE_GIFT === $row->getSrc()->coupon->type_discount ) {
                            return $row->getSrc()->coupon->present()->discount_consumer;
                        } else if (Auth::user()->hasRole(User::SELLER)) {
                            return $row->getSrc()->coupon->getTransactionAmountFormatted(Transaction::TYPE_SELLER_CONSUMER_DISCOUNT_PAYMENT);
                        } else {
                            return $row->getSrc()->coupon->getTransactionAmountFormatted(Transaction::TYPE_CONSUMER_COUPON_DISCOUNT_CREDIT);
                        }
                    }

                });
        }

        if (Auth::user()->can('show seller transactions')
            || Auth::user()->can('show agent transactions')
            || Auth::user()->can('show operator transactions')
            || Auth::user()->can('show fieldservice transactions')
        ) {

            $gridColumns[] = (new FieldConfig)
                ->setName('provision_agent')
                ->setLabel('RECO.CASH')
                ->setSortable(false)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val, $row) {
                    // TODO if gift
                    if ($row->getSrc()->coupon) {
                        if ( (Auth::user()->hasRole(User::AGENT)) && CouponTemplate::TYPE_GIFT === $row->getSrc()->coupon->type_provision ) {
                            return $row->getSrc()->coupon->present()->provision_agent;
                        } else if (Auth::user()->hasRole(User::SELLER)) {
                            return $row->getSrc()->coupon->getTransactionAmountFormatted(Transaction::TYPE_SELLER_AGENT_PROVISION_PAYMENT);
                        } else {
                            return $row->getSrc()->coupon->getTransactionAmountFormatted(Transaction::TYPE_AGENT_COUPON_PROVISION_CREDIT);
                        }
                    }

                    // Recruiting actions are also listed as reco-cash
                    if ($row->getSrc()->recruitedAccount) {
                        return Format::currency($row->getSrc()->amount);
                    }
                });
        }

        if (Auth::user()->can('show operator transactions')) {
            $gridColumns[] = (new FieldConfig)
                ->setName('netto')
                ->setLabel('Warenkorb')
                ->setSortable(false)
                ->setCallback(function ($val, $row) {
                    if ($row->getSrc()->coupon){
                        return Format::currency($row->getSrc()->coupon->netto);
                    }
                });
        }

        if (Auth::user()->can('show seller transactions') || Auth::user()->can('show operator transactions')) {
            $gridColumns[] = (new FieldConfig)
                ->setName('operator-coupon-provision-credit')
                ->setLabel('Beitrag')
                ->setSortable(false)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val, $row) {
                    if (!$row->getSrc()->coupon) {
                        return Format::currency($row->getSrc()->amount); //FIXME: to be implemented
                    }
                });
        }

        if (Auth::user()->can('show seller transactions') || Auth::user()->can('show operator transactions')) {
            // Status Seller Bill
            $gridColumns[] = (new FieldConfig)
                ->setName('primary_transaction_status')
                ->setLabel('Status')
                ->setSortable(false)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val, $row) {
                    if ($row->getSrc()->coupon) {

                        $result =
                            // Does not matter which payment we wtake -> all are on the same invoice
                            $row->getSrc()->coupon->getTransactionStatusText(Transaction::TYPE_SELLER_AGENT_PROVISION_PAYMENT)
                            . ' ' . $row->getSrc()->coupon->getTransactionPDFLink(Transaction::TYPE_SELLER_AGENT_PROVISION_PAYMENT);

                        if (Auth::user()->can('show operator transactions')) {
                            $result .= ' ' . $row->getSrc()->coupon->getReceiptStatusText(Transaction::TYPE_SELLER_AGENT_PROVISION_PAYMENT);
                        }
                        return $result;
                    } else {
                        return $row->getSrc()->getStatusText() . ' ' .
                        '<a href="' . e($row->getSrc()->getPDFLink()) . '" target="_blank"><i class="fa fa-file-pdf-o"></i></a>';
                    }
                });
        }

        if (Auth::user()->can('show seller transactions') || Auth::user()->can('show agent transactions') || Auth::user()->can('show operator transactions')) {
            $gridColumns[] = (new FieldConfig)
                ->setName('prospective_customer')
                ->setSortable(true)
                ->setLabel('Verbraucher')
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                        ->setFilteringFunc(function ($val, EloquentDataProvider $provider) {
                            $provider->getBuilder()
                                ->where(User::getDisplayNameDBExpression(false, 'consumer'), 'like', '%' . $val . '%');
                        })
                )
                ->setCallback(function ($val, $row) {
                    if ($row->getSrc()->coupon) {
                        $transaction = $row->getSrc();
                        //check if consumer is null from globalScope
                        if(!$transaction->coupon->consumer){
                            return trans('translations.not-active');
                        }else{
                            return $transaction->coupon->consumer->getProfileLink(User::CONSUMER);

                        }
                    }
                });
        }

        if (Auth::user()->can('show consumer transactions') || Auth::user()->can('show operator transactions')) {
            // Status Consumer credit note
            $gridColumns[] = (new FieldConfig)
                ->setName('id')
                ->setLabel('Status')
                ->setSortable(false)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val, $row) {
                    if ($row->getSrc()->coupon) {
                        $result = $row->getSrc()->coupon->getTransactionStatusText(Transaction::TYPE_CONSUMER_COUPON_DISCOUNT_CREDIT)
                            . ' ' . $row->getSrc()->coupon->getTransactionPDFLink(Transaction::TYPE_CONSUMER_COUPON_DISCOUNT_CREDIT);

                        if (Auth::user()->can('show operator transactions')) {
                            $result .= ' ' . $row->getSrc()->coupon->getReceiptStatusText(Transaction::TYPE_CONSUMER_COUPON_DISCOUNT_CREDIT);
                        }
                        return $result;
                    }
                });
        }

        if (Auth::user()->can('show consumer transactions') || Auth::user()->can('show seller transactions') || Auth::user()->can('show operator transactions')) {
            $gridColumns[] = (new FieldConfig)
                ->setName('recommender')
                ->setSortable(true)
                ->setLabel('Empfehler')
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                        ->setFilteringFunc(function ($val, EloquentDataProvider $provider) {
                            $provider->getBuilder()
                                ->where(User::getDisplayNameDBExpression(false, 'agent'), 'like', '%' . $val . '%');
                        })
                )
                ->setCallback(function ($val, $row) {
                    if ($row->getSrc()->coupon) {
                        $transaction = $row->getSrc();
                        if(!$transaction->coupon->agent){
                            return trans('translations.not-active');
                        }else{
                            return $transaction->coupon->agent->getProfileLink(User::AGENT);
                        }
                    }
                    if ($row->getSrc()->recruitedAccount) {
                        $transaction = $row->getSrc();
                        if(!$transaction->user){
                            return trans('translations.not-active');
                        }else{
                            return $transaction->user->getDisplayName(); // for now no link, because it is not clear in which role the user was acting (can be multiple)

                        }
                    }
                });
        }

        /*
        if(Auth::user()->hasRole('consumer')){
            $gridColumns[] = (new FieldConfig)
                ->setName('id')
                ->setLabel('Chatten')
                ->setCellHtmlAttributes(['class' => 'button-celll'])
                ->setSortable(false)
                ->setCallback(function ($val, $row) {
                    if ($val !== null  && $agent_user_id = $row->getSrc()->coupon->agent) { // nicht "ohne Anfrage verfügbar"
                        $agent_user_id = $row->getSrc()->coupon->agent->id;
                        return '<a href="javascript:;" onclick="jqac.arrowchat.chatWith('.$agent_user_id.');"><img class="chat-image-small" src="/img/icons/chat-bubble.svg"> </a>';
                    }
                    return '';

                });
        }
        */

        if (Auth::user()->can('show agent transactions')
            || Auth::user()->can('show operator transactions')
            || Auth::user()->can('show fieldservice transactions')
        ) {
            $gridColumns[] = (new FieldConfig)
                ->setName('agent_transaction_status')
                ->setLabel('Status')
                ->setSortable(false)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val, $row) {
                    if ($row->getSrc()->coupon) {
                        $result = $row->getSrc()->coupon->getTransactionStatusText(Transaction::TYPE_AGENT_COUPON_PROVISION_CREDIT)
                            . ' ' . $row->getSrc()->coupon->getTransactionPDFLink(Transaction::TYPE_AGENT_COUPON_PROVISION_CREDIT);

                        if (Auth::user()->can('show operator transactions')) {
                            $result .= ' ' . $row->getSrc()->coupon->getReceiptStatusText(Transaction::TYPE_AGENT_COUPON_PROVISION_CREDIT);
                        }
                        return $result;
                    }

                    // for the operator, the transaction status is already shown in another field
                    if (Auth::user()->can('show operator transactions')) {
                        return;
                    }

                    $result = $row->getSrc()->getStatusText();

                    if ($row->getSrc()->getPDFLink()) {
                        $result .= '<a href="' . e($row->getSrc()->getPDFLink()) . '" target="_blank"><i class="fa fa-file-pdf-o"></i></a>';
                    }

                    return $result;
                });
        }

        $gridColumns[] = (new FieldConfig)
            ->setName('created_at')
            ->setLabel('Datum')
            ->setSortable(false)
            ->addFilter(
                (new FilterConfig)
                    ->setOperator(FilterConfig::OPERATOR_LIKE)
            )
            ->setCallback(function ($val, $row) {
                if ($row->getSrc()->billed_until && $row->getSrc()->billed_from) {
                    return Format::date($row->getSrc()->billed_from) . ' - ' . Format::date($row->getSrc()->billed_until);
                }
                return Format::date($val);
            });
        if (Auth::user()->can('show consumer transactions')) {
            $gridColumns[] = (new FieldConfig)
                ->setName('next-step')
                ->setLabel('Nächster Schritt')
                ->setCellHtmlAttributes(['class' => 'button-cell'])
                ->setSortable(false)
                ->setCallback(function ($val, $row) {
                    $transaction = $row->getSrc()->coupon->getTransactionByType(Transaction::TYPE_CONSUMER_COUPON_DISCOUNT_CREDIT);
                    if($transaction->user_id == Auth::user()->id) {
                        if(!$transaction->coupon->template->seller){
                            return trans('translations.not-active');
                        }elseif($transaction->coupon->status == Coupon::STATUS_REDEEMED){
                            return 'Deine Zahlung muss beim Verkäufer eingehen.<br>' .
                                '<a class="btn btn-warning " href="' . route('coupon-assert-payment', $transaction->coupon['id']) . '">Als bezahlt markieren</a>';
                        }elseif($transaction->coupon->status == Coupon::STATUS_PAYMENT_ASSERTED){
                            return 'Der Verkäufer bestätigt die Zahlung ebenfalls.';
                        }
                        if ($transaction->status == Transaction::STATUS_PAID) {
                            return 'abgeschlossen';
                        }
                        if ($transaction->status == Transaction::STATUS_PAYMENT_REQUESTED) {
                            return '';
                        }
                        if ($transaction->coupon->status == Coupon::STATUS_DONE && $transaction->status == Transaction::STATUS_PENDING) {
                            return 'Dein CASHBACK kann bald ausgezahlt werden. Wir informieren dich per Mail.';
                        }
                        if ($transaction->coupon->status == Coupon::STATUS_PAID) {
                            return '<button
                                    type="button"
                                    class="btn btn-primary"
                                    data-toggle="modal"
                                    data-transactionid=' . $transaction->id . '
                                    data-target="#rateModal">
                                    Bewerten
                                </button>';
                        }
                        if ($transaction->status == Transaction::STATUS_ALLOCATED) {
                            return 'Bitte Auszahlen klicken.';
                        }
                    }
                });

        }
        return $gridColumns;
    }
}