<?php

namespace App\Http\Grids\Seller;

use App\Coupon;
use App\CouponTemplate;
use App\Library\Facades\Format;
use App\Library\Grids\DefaultGrid;
use App\Library\Grids\DummyFieldConfig;
use App\Library\Grids\FieldConfig;
use App\Library\Grids\GlobalFilter;
use App\Library\Grids\GridConfig;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Nayjest\Grids\DbalDataProvider;
use Nayjest\Grids\EloquentDataProvider;
use Nayjest\Grids\FilterConfig;
use Nayjest\Grids\Grid;
use Nayjest\Grids\SelectFilterConfig;

class CouponTemplateGrid extends DefaultGrid
{
    public function __construct()
    {
        return parent::__construct($this->getGridConfig());
    }

    protected function getGridConfig()
    {
        return (new GridConfig)
            ->setDataProvider(
                new EloquentDataProvider($this->getGridQuery())
            )
            ->setName('coupons')
            ->setColumns($this->getGridColumns());
    }

    public function getGlobalFiltersConfig()
    {
        $filters['fulltext'] = new GlobalFilter($this->libraryTitleConfig(), new DummyFieldConfig(), $this, 'fulltext');
        $filters['place'] = new GlobalFilter($this->libraryMySharedCouponsStatusConfig(), new DummyFieldConfig(), $this, 'place');

        return $filters;
    }

    public function libraryTitleConfig()
    {
        $config =
            (new FilterConfig)
                ->setLabel('RECO.BON')
                ->setName('fulltext')
                ->setOperator(FilterConfig::OPERATOR_LIKE)
                ->setFilteringFunc(function ($val, EloquentDataProvider $provider) {
                    $provider->getBuilder()
                        ->where(function ($query) use ($val) {
                            $query->where('title', 'like', '%' . $val . '%');
                        });
                });

        return $config;
    }

    public function libraryMySharedCouponsStatusConfig()
    {
        $config =
            (new SelectFilterConfig)
                ->setName('status')
                ->setLabel('Status')
                ->setSubmittedOnChange(true)
                ->setOptions([
                    'active' => trans('translations.active'),
                    'inactive' => trans('translations.inactive'),
                ])
        ;

        return $config;
    }

    protected function getGridQuery()
    {
        $query = CouponTemplate::where('user_id', Auth::user()->id);

        return $query;
    }

    protected function getGridColumns()
    {
        $gridColumns = [
            (new FieldConfig)
                ->setName('title')
                ->setLabel('RECO.BON')
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
            ,
            (new FieldConfig)
                ->setName('provision_agent')
                ->setLabel('RECO.CASH')
                ->setSorting(Grid::SORT_ASC)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val, $row) {
                    return Format::percent($val);
                })
            ,
            (new FieldConfig)
                ->setName('discount_consumer')
                ->setLabel('Cashback')
                ->setSorting(Grid::SORT_ASC)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val, $row) {
                    return Format::percent($val);
                })
            ,
            (new FieldConfig)
                ->setName('sucess_rate')
                ->setLabel('Erfolgsrate')
                ->setSorting(Grid::SORT_ASC)
                ->setSortable(false)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val, $row) {
                    $successRate = 0;
                    $doneCount = $row->getSrc()->coupons()->whereIn('status', ['paid', 'done'])->count();
                    $count = $row->getSrc()->coupons()->get()->count();
                    if($doneCount != 0 && $count !=0){
                        $successRate = 100 / $count * $doneCount;
                    }
                    return Format::percent($successRate);
                })
            ,
            (new FieldConfig)
                ->setName('discount_consumer')
                ->setLabel('Erstellt')
                ->setSorting(Grid::SORT_ASC)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val, $row) {
                    $count = $row->getSrc()->coupons()->count();
                    return $count;
                })
            ,
            (new FieldConfig)
                ->setName('quota')
                ->setLabel('Verfügbar')
                ->setSorting(Grid::SORT_ASC)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val, $row) {
                    $couponTemplate = $row->getSrc();
                    $couponCount = $row->getSrc()->coupons()->count();
                    return $val - $couponCount;
                })
            ,
            (new FieldConfig)
                ->setName('coupon_id')
                ->setLabel('Eingelöst')
                ->setSorting(Grid::SORT_ASC)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val, $row) {
                    $couponTemplate = $row->getSrc();
                    $couponCount = Coupon::where('coupon_template_id', '=', $couponTemplate->id)
                        ->whereIn('status', ['paid', 'done'])
                        ->count();
                    return $couponCount;
                })
            ,
            (new FieldConfig)
                ->setName('expiration_date')
                ->setLabel('Ablaufdatum')
                ->setSorting(Grid::SORT_ASC)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val, $row) {
                    if ($row->getSrc()->isExpired()) {
                        return '<span class="label label-default">abgelaufen</span>';
                    }
                    return Format::date($val);
                })
            ,
            (new FieldConfig)
                ->setName('status')
                ->setLabel('Status')
                ->setSorting(Grid::SORT_ASC)
                ->addFilter(
                    (new SelectFilterConfig)
                        ->setSubmittedOnChange(true)
                        ->setOptions([
                            'active' => trans('translations.active'),
                            'inactive' => trans('translations.inactive'),
                        ])
                        ->setOperator(FilterConfig::OPERATOR_EQ)
                )
                ->setCallback(function ($val, $row) {
                    if ($row->getSrc()->isExpired()) {
                        return '';
                    }

                    $id = $row->getSrc()->id;
                    $result = (($val == 'active') ?
                        '<a class="active-toggle" href="/coupon/status/' . $id . '/deactivate"><span class="sr-only">Deactivate</span>'
                        : '<a class="active-toggle" href="/coupon/status/' . $id . '/activate"><span class="sr-only">Activate</span>');

                    $result .=  ' ' . $row->getSrc()->getStatusText();

                    $result .= '</a>';

                    return $result;

                })
            ,
            (new FieldConfig)
                ->setName('id')
                ->setLabel(' ')
                ->setSortable(false)
                ->setCellHtmlAttributes(['class' => 'button-cell'])
                ->setCallback(function ($val, $row) {
                    return
                        '<a class="btn btn-white " href="' . route('coupon-id', $val) . '"><i class="fa fa-pencil"></i></a>';
                })
        ];
        return $gridColumns;
    }
}