<?php

namespace App\Http\Grids\Agent;

use App\Account;
use App\Coupon;
use App\Library\Facades\Format;
use App\Library\Grids\DefaultGrid;
use App\Library\Grids\DummyFieldConfig;
use App\Library\Grids\FieldConfig;
use App\Library\Grids\GlobalFilter;
use App\Library\Grids\GridConfig;
use Illuminate\Support\Facades\Auth;
use Nayjest\Grids\EloquentDataProvider;
use Nayjest\Grids\FilterConfig;
use Nayjest\Grids\Grid;

/**
 * Available coupon templates (sometimes called coupons) for the seller to be sent to consumers
 *
 */
class CouponTemplateGrid extends DefaultGrid
{
    public function __construct()
    {
        return parent::__construct($this->getGridConfig());
    }


    protected function getGridConfig()
    {
        return (new GridConfig)
            ->setDataProvider(
                new EloquentDataProvider($this->getGridQuery())
            )
            ->setName('Coupons')
            ->setColumns($this->getGridColumns());
    }

    public function getGlobalFiltersConfig()
    {
        $filters['fulltext'] = new GlobalFilter($this->libraryAccountFullTextFilterConfig(), new DummyFieldConfig(), $this, 'fulltext');
        $filters['place'] = new GlobalFilter($this->libraryPlaceConfig(), new DummyFieldConfig(), $this, 'place');

        return $filters;
    }


    protected function getGridQuery()
    {
        return Auth::user()->availableCouponsQuery()
            ->select([
                'coupon_templates.*',
                'accounts.id' => 'users.account_id',
                'accounts.company',
                'accounts.zip',
                'accounts.city',
            ])
            ->join('users', 'user_id', '=', 'users.id', 'left')
            ->join('accounts', 'users.account_id', '=', 'accounts.id', 'left')
            ->groupBy('coupon_templates.id');
    }

    protected function getGridColumns()
    {
        $gridColumns = [
            (new FieldConfig)
                ->setName('company')
                ->setLabel('Unternehmen')
                ->setSorting(Grid::SORT_ASC)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val, $row) {
                    $couponTemplate = $row->getSrc();
	                $account = Account::find($couponTemplate->account_id);
	                return $account->renderFullLink();
                })
            ,
            (new FieldConfig)
                ->setName('id')
                ->setLabel('PLZ')
                ->setSorting(Grid::SORT_ASC)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                        ->setFilteringFunc(function ($val, EloquentDataProvider $provider) {
                            $provider->getBuilder()
                                ->where('accounts.zip', 'like', '%' . $val . '%');
                        })
                )
                ->setCallback(function ($val, $row) {
                    $couponTemplate = $row->getSrc();

                    return $couponTemplate->zip;
                })
            ,
            (new FieldConfig)
                ->setName('id')
                ->setLabel('Ort')
                ->setSorting(Grid::SORT_ASC)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                        ->setFilteringFunc(function ($val, EloquentDataProvider $provider) {
                            $provider->getBuilder()
                                ->where('accounts.city', 'like', '%' . $val . '%');
                        })
                )
                ->setCallback(function ($val, $row) {
                    $couponTemplate = $row->getSrc();

                    return $couponTemplate->city;
                })
            ,

            (new FieldConfig)
                ->setName('title')
                ->setLabel('RECO.BON')
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
            ,
            (new FieldConfig)
                ->setName('id')
                ->setLabel('Ø Bewertung')
                ->setCallback(function ($val, $row) {
                    return Format::rating($row->getSrc()->user->account->averageRating);
                })
            ,
            (new FieldConfig)
                ->setName('provision_agent')
                ->setLabel('RECO.CASH')
                ->setSorting(Grid::SORT_ASC)
                ->setCallback(function ($val, $row) {
                    return $row->getSrc()->present()->provision_agent;
                })
            ,
            (new FieldConfig)
                ->setName('discount_consumer')
                ->setLabel('Cashback')
                ->setSorting(Grid::SORT_ASC)
                ->setCallback(function ($val, $row) {
                    return $row->getSrc()->present()->discount_consumer;
                })
            ,
            (new FieldConfig)
                ->setName('quota')
                ->setLabel('Verfügbar')
                ->setSorting(Grid::SORT_ASC)
                ->setCallback(function ($val, $row) {
                    $couponTemplate = $row->getSrc();
                    $couponCount = count(Coupon::where('coupon_template_id', '=', $couponTemplate->id)->get());
                    return Format::value($val - $couponCount);
                })
            ,
            (new FieldConfig)
                ->setName('id')
                ->setLabel(' ')
                ->setCellHtmlAttributes(['class' => 'button-cell'])
                ->setSortable(false)
                ->setCallback(function ($val, $row) {
                    $couponTemplate = $row->getSrc();
                    if ($couponTemplate->status == 'active' && $couponTemplate->limitReached() == false) {
                        $html = '<a class="btn btn-primary-cta" id="send-template-' . $val . '" href="' . url('/share-coupon') . '/' . $val . '">Empfehlen</a> ';
                        // TODO check if user hasn't same coupon active ?
                        $html .= '<a class="btn btn-primary-cta" href="' . route('coupon.get-cashback', $val) . '">Get Cashback</a>';
                        return $html;
                    }
                    return 'Limit erreicht';
                })
        ];
        return $gridColumns;
    }
}