<?php

namespace App\Http\Grids\Agent;

use App\Coupon;
use App\CouponTemplate;
use App\Library\Facades\Format;
use App\Library\Grids\DefaultGrid;
use App\Library\Grids\DummyFieldConfig;
use App\Library\Grids\FieldConfig;
use App\Library\Grids\GlobalFilter;
use App\Library\Grids\GridConfig;
use App\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\HtmlString;
use Nayjest\Grids\EloquentDataProvider;
use Nayjest\Grids\FilterConfig;
use Nayjest\Grids\Grid;

class CouponGrid extends DefaultGrid
{
    protected $_name = 'recobons';

    public function getGlobalFiltersConfig()
    {
        $filters['fulltext'] = new GlobalFilter($this->libraryAgentTransactionCompanyAndConsumerConfig(), new DummyFieldConfig(), $this, 'fulltext');
        $filters['status'] = new GlobalFilter($this->libraryAgentSharedCouponsStatusConfig(), new DummyFieldConfig(), $this, 'status');

        return $filters;
    }

    protected function getGridQuery()
    {
        return Coupon::whereAgentUserId(Auth::user()->id)
            ->select([
                'coupons.*',
                User::getDisplayNameDBExpression('consumer_display_name', 'consumers'),
                User::getDisplayNameDBExpression('seller_display_name', 'sellers'),
                'coupon_templates.user_id as coupon_template_user_id',
                'seller_accounts.company as seller_company',
                'seller_accounts.id as seller_account_id',
            ])
            ->join('users as consumers', 'consumer_user_id', '=', 'consumers.id')
            ->join('coupon_templates', 'coupon_template_id', '=', 'coupon_templates.id')
            ->join('users as sellers',  'coupon_templates.user_id', '=', 'sellers.id')
            ->join('accounts as seller_accounts',  'sellers.account_id', '=', 'seller_accounts.id')
            ->where('selection', '=', 0);
    }

    protected function getGridColumns()
    {

            $gridColumns =  [
                (new FieldConfig)
                    ->setName('seller_company')
                    ->setLabel('Unternehmen')
                    ->setSorting(Grid::SORT_ASC)
                    ->addFilter(
                        (new FilterConfig)
                            ->setOperator(FilterConfig::OPERATOR_LIKE)
                            ->setFilteringFunc(function ($val, EloquentDataProvider $provider) {
                                $provider->getBuilder()
                                    ->where( User::getDisplayNameDBExpression(false, 'sellers'), 'like', '%' . $val . '%');

                            })
                    )
                    ->setCallback(function ($val, $row) {
                        if(!$row->getSrc()->template->seller){
                            return trans('translations.not-active');
                        }
                        if ($val == '') {
                            $val = $row->getSrc()->seller_display_name;
                        }
                        return new HTMLString('<a href="' . route('show-seller-account', $row->getSrc()->seller_account_id) . '">' . e($val) . '</a>');
                    })
                ,
                (new FieldConfig)
                    ->setName('consumer_display_name')
                    ->setLabel('Verbraucher')
                    ->setSorting(Grid::SORT_ASC)
                    ->addFilter(
                        (new FilterConfig)
                            ->setOperator(FilterConfig::OPERATOR_LIKE)
                            ->setFilteringFunc(function ($val, EloquentDataProvider $provider) {
                                $provider->getBuilder()
                                    ->where( User::getDisplayNameDBExpression(false, 'consumers'), 'like', '%' . $val . '%');

                            })
                    )
                    ->setCallback(function ($val, $row) {
                        if(!$row->getSrc()->consumer){
                            return trans('translations.not-active');
                        }else{
                            return $row->getSrc()->consumer->getProfileLink(User::CONSUMER);
                        }
                    })
                ,
                (new FieldConfig)
                    ->setName('title')
                    ->setLabel('RECO.BON')
                    ->addFilter(
                        (new FilterConfig)
                            ->setOperator(FilterConfig::OPERATOR_LIKE)
                    )
                ,
                (new FieldConfig)
                    ->setName('code')
                    ->setLabel('RECO.CODE')
                    ->setSortable(false)
                    ->setSorting(Grid::SORT_ASC)
                    ->setCallback(function ($val) {
                        return Format::recoCode($val);
                    })
                ,
                (new FieldConfig)
                    ->setName('provision_agent')
                    ->setLabel('RECO.CASH')
                    ->setSorting(Grid::SORT_ASC)
                    ->setCallback(function ($val, $row) {
                        return $row->getSrc()->present()->provision_agent;
                    })
                ,

            (new FieldConfig)
                ->setName('discount_consumer')
                ->setLabel('Cashback')
                ->setSorting(Grid::SORT_ASC)
                ->setCallback(function ($val, $row) {
                    return $row->getSrc()->present()->discount_consumer;
                })
            ,
            (new FieldConfig)
                ->setName('netto')
                ->setLabel('Warenkorb')
                ->setSorting(Grid::SORT_ASC)
                ->setCallback(function ($val) {
                    return Format::currency($val);
                })
            ,
            (new FieldConfig)
                ->setName('status')
                ->setLabel('Status')
                ->setSorting(Grid::SORT_ASC)
                ->setCallback(function ($val, $row) {
                    return $row->getSrc()->getStatusText();
                })
            ,
            (new FieldConfig)
                ->setName('created_at')
                ->setLabel('Empfohlen')
                ->setSorting(Grid::SORT_ASC)
                ->setCallback(function ($val) {
                    return Format::date($val);
                })
            ];

        return $gridColumns;
    }
}