<?php

namespace App\Http\Grids;

use App\Library\Facades\Format;
use App\Library\Grids\DummyFieldConfig;
use App\Library\Grids\FieldConfig;
use App\Library\Grids\GlobalFilter;
use App\Library\Grids\GridConfig;
use App\Transaction;
use App\User;
use Illuminate\Support\Facades\Auth;
use App\Library\Grids\DefaultGrid;
use Nayjest\Grids\EloquentDataProvider;
use Nayjest\Grids\FilterConfig;

class AgentTransactionGrid extends DefaultGrid
{
    public function __construct()
    {
        return parent::__construct($this->getGridConfig());
    }

    protected function getGridConfig()
    {
        return (new GridConfig)
            ->setDataProvider(
                new EloquentDataProvider($this->getGridQuery())
            )
            ->setName('Transaktionen')
            ->setColumns($this->getGridColumns());
    }

    public function getGlobalFiltersConfig()
    {
        $filters['fulltext'] = new GlobalFilter($this->libraryAgentTransactionCompanyAndConsumerConfig(), new DummyFieldConfig(), $this, 'fulltext');
        $filters['status'] = new GlobalFilter($this->libraryTransactionStatusConfig(), new DummyFieldConfig(), $this, 'status');

        return $filters;
    }

    protected function getGridQuery()
    {
        $query = Transaction::query()
            ->select([
                'transactions.*',
                'coupons.created_at as coupon_created_at',
                'accounts.company',
                User::getDisplayNameDBExpression('consumer_display_name', 'consumers'),
            ])
            ->where('transactions.user_id', '=', Auth::id())
            ->join('coupons', 'coupon_id', '=', 'coupons.id', 'left')
            ->join('coupon_templates', 'coupons.coupon_template_id', '=', 'coupon_templates.id', 'left')
            ->join('users', 'coupon_templates.user_id', '=', 'users.id', 'left')
            ->join('accounts', 'users.account_id', '=', 'accounts.id', 'left')
            ->join('users as consumers', 'coupons.consumer_user_id', '=', 'consumers.id', 'left');

        return $query;
    }

    protected function getGridColumns()
    {
        $gridColumns = [
            (new FieldConfig)
                ->setName('coupon_created_at')
                ->setLabel('Empfohlen')
                ->setSortable(true)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val, $row) {
                    return Format::date($val);
                })
            ,
            (new FieldConfig)
                ->setName('coupon_id')
                ->setSortable(false)
                ->setLabel('Unternehmen')
                ->setSortable(true)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val, $row) {
                    $transaction = $row->getSrc();
                    return '<a class="" href="' . route('show-seller-account', $transaction->coupon->template->seller->account->id) . '">'
                    . $transaction->coupon->template->seller->account->company
                    . '</a>';
                })
            ,
            (new FieldConfig)
                ->setName('coupon_id')
                ->setLabel('Verbraucher')
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val, $row) {
                    $transaction = $row->getSrc();
                    return $transaction->coupon->consumer->getProfileLink(User::CONSUMER);
                })
            ,
            (new FieldConfig)
                ->setName('amount')
                ->setLabel('RECO.CASH')
                ->setSortable(true)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val) {
                    return Format::currency($val);

                })
            ,
            (new FieldConfig)
                ->setName('coupon_id')
                ->setLabel('RECO.CODE')
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val, $row) {
                    $transaction = $row->getSrc();
                    return $transaction->coupon->getCodeFormatted();
                })
            ,
            (new FieldConfig)
                ->setName('transaction.status')
                ->setSortable(false)
                ->setLabel('Status')
                ->setSortable(true)
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val, $row) {
                    /**
                     * @var $transaction Transaction
                     */
                    $transaction = $row->getSrc();
                    /*if(Auth::user()->hasRole('consumer')){
                        return 'Bitte Dienstleister erst bewerten, damit die Auszahlung angefordert werden kann.' . ' - ' . $val;
                    }*/
                    return $transaction->getStatusText();
                })
        ];

        $gridColumns[] = (new FieldConfig)
            ->setName('id')
            ->setSortable(false)
            ->setLabel('Bezahlt')
            ->addFilter(
                (new FilterConfig)
                    ->setOperator(FilterConfig::OPERATOR_LIKE)
            )
            ->setCallback(function ($val, $row) {
                $transaction = $row->getSrc();
                if ($transaction->created_at != $transaction->updated_at) {
                    return Format::date($transaction->updated_at);
                }
            });

        $gridColumns[] = (new FieldConfig)
            ->setName('transaction.status')
            ->setSortable(false)
            ->setLabel('Beleg')
            ->setCellHtmlAttributes(['class' => 'button-cell'])
            ->setCallback(function ($val, $row) {
                /**
                 * @var $transaction Transaction
                 */
                $transaction = $row->getSrc();

                if ($transaction->getPDFLink()) {
                    return '<a class="btn btn-white " href="' . htmlspecialchars($transaction->getPDFLink()) . '" target="_blank">PDF</a>';
                }
            });

        $gridColumns[] = (new FieldConfig)
            ->setName('transaction.user_id')
            ->setLabel(' ')
            ->setCellHtmlAttributes(['class' => 'button-cell'])
            ->setSortable(false)
            ->setCallback(function ($val, $row) {
                $transaction = $row->getSrc();
                if ($transaction->status == 'rate') {
                    return '<a class="btn btn-white " href="/agent/payout/' . $transaction->id . '">Auszahlen</a>';
                }
            });

        return $gridColumns;
    }
}