<?php


namespace App\Http\Grids;

use App\Account;
use App\Library\Facades\Format;
use App\Library\Grids\DefaultGrid;
use App\Library\Grids\DummyFieldConfig;
use App\Library\Grids\FieldConfig;
use App\Library\Grids\GlobalAutoCompleteFilter;
use App\Library\Grids\GlobalFilter;
use App\User;
use Illuminate\Support\Facades\DB;
use Nayjest\Grids\EloquentDataProvider;
use Nayjest\Grids\FilterConfig;


/**
 * Base Grid for User Infos (Name and Zip + City)
 *
 */
class AbstractAccountGrid extends DefaultGrid
{
    /**
     * Empty dummy query for global filter logic
     * @return $this
     */
    protected function getGridQuery()
    {
        return Account::where(false);
    }

    /**
     * Join the users table field to the grid
     */
    protected function getGridFinalQuery()
    {
        $query = parent::getGridFinalQuery();

        $query
            ->addSelect([
                'accounts.id as account_id',
                'accounts.*',
                DB::raw('CONCAT(users.firstname, " ", users.surname) as fallback_name'),
                'users.id as fallback_user_id'
            ])
            ->join('users', 'accounts.id', '=', 'users.account_id')
            ->groupBy('accounts.id');

        return $query;
    }

    public function getGlobalFiltersConfig()
    {

        $filters = [];

        $filters['place'] = new GlobalFilter($this->libraryPlaceConfig(), new DummyFieldConfig(), $this, 'place');

        return $filters;
    }

    protected function getGridColumns()
    {
        $columns = [
            'company' => (new FieldConfig)
                ->setName('company')
                ->setLabel('Unternehmen')
                ->setCallback(function ($val, $row) {
                	$account = $row->getSrc();
	                return $account->renderFullLink(User::SELLER);
                })
            ,
            'zip' => (new FieldConfig)
                ->setName('zip')
                ->setLabel('PLZ')
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val) {
                    if ($val == '') {
                        return 'folgt';
                    }
                    return $val;
                })
            ,
            'city' => (new FieldConfig)
                ->setName('city')
                ->setLabel('Ort')
                ->addFilter(
                    (new FilterConfig)
                        ->setOperator(FilterConfig::OPERATOR_LIKE)
                )
                ->setCallback(function ($val) {
                    if ($val == '') {
                        return 'folgt';
                    }
                    return $val;
                })
            ,
            'rating' => (new FieldConfig)
                ->setName('id')
                ->setLabel('Bewertung')
                ->setCallback(function ($val, $row) {
                    return Format::rating($row->getSrc()->averageRating);
                })
            ,
        ];
        return $columns;
    }
}