<?php

namespace App\Http\Controllers;

use App\Account;
use App\Action;
use App\Activity;
use App\Coupon;
use App\CouponTemplate;
use App\Exceptions\UnsupportedException;
use App\Http\Grids\UserGrid;
use App\Http\Grids\Seller\RatingGrid;
use App\Http\Grids\Fieldservice\UserGrid as FieldserviceUserGrid;
use App\Http\Requests\AccountRequest;
use App\Http\Requests\CompanyPictureRequest;
use App\Http\Requests\CouponSendRequest;
use App\Http\Requests\FullSellerRegistrationRequest;
use App\Http\Requests\SellerRegistrationRequest;
use App\Http\Requests\UserProfilePasswordRequest;
use App\Http\Requests\UserProfileRequest;
use App\Http\Requests\UserRequest;
use App\LinkInvitations;
use App\Rating;
use App\Services\ReferralLinkService;
use App\Services\RouteService;
use App\Services\SellerInvitationLinkService;
use App\Services\SellerRegistrationService;
use App\Services\UserService;
use App\Shortlink;
use App\Transaction;
use App\User;
use Carbon\Carbon;
use Illuminate\Foundation\Auth\ResetsPasswords;
use Illuminate\Http\Request;
use Illuminate\Routing\Route;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Redirect;
use Intervention\Image\Facades\Image;
use SimpleSoftwareIO\SMS\Facades\SMS;
use Spatie\Permission\Models\Role;
use Input;
use App\Http\Requests\AliasRequest;
use App\Http\Requests\BankingDataRequest;
use Validator;

class UserController extends Controller
{

    use ResetsPasswords;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware(['auth']);
        if (!in_array(
            $this->getRouter()->getCurrentRoute()->getAction()['controller'],
            [
                'App\Http\Controllers\UserController@profile',
                'App\Http\Controllers\UserController@store'
            ])
        ) {

            $this->middleware('completed.profile');
        }

    }

    // FIXME: sollte nicht den fall neuer-user beinhalten, das ding heißt edit
    public function edit($id=null)
{
        if(auth::user()->hasRole('fieldservice_employee')){
            $user = \App\User::withoutGlobalScopes()->find($id);
        }else {
            $user = \App\User::find($id);
        }
        $editSelfUrl = '/user/edit/' . Auth::user()->id;

        if ($user === null) { // create user
            if ($this->user->can('create users')) {
                $account = new Account();
                $user = new User();
            } else {
                return Redirect::to($editSelfUrl);
            }
        } else { // edit user
            $account = Account::withoutGlobalScopes()->find($user->account_id);
            if (!$this->user->canEditUser($user)) {
                return Redirect::to($editSelfUrl);
            }
        }

        // FIXME: ist das korrekt / sinnvoll?
        $userAccounts = Account::withoutGlobalScopes()->where('company', 'not like', '')->get();

        return view('user/edit', [
            'user' => $user,
            'account' => $account,
            'userAccounts' => $userAccounts,
            'requestClass' => \App\Http\Requests\UserRequest::class
        ]);
    }

    public function show(User $user)
    {
        $ratings = Rating::where('rated_account_id', '=', $user->account->id)->get();
        return view('user/user-info',
            [
                'user' => $user,
                'ratings' => $ratings
            ]);
    }

    public function showAccount($role, Account $account)
    {
        $user = $account->users()->first();
        if(!$user){
            return back()->with('notification', $this->notificationError(trans('translations.not-active')));
        }

        if (!$user->hasRole($role)) {
            return 'Das Profil existiert (in dieser Ausprägung) nicht.';
        }

        return view('account/account-info',
            [
                'account' => $account,
                'user' => $user,
                'role' => $role,
            ]);
    }

    public function profile()
    {
        $user = Auth::user();
        Auth::setUser($user->fresh());
        $payment = $user->account->payment;
        return view('user/profile', ['payment' => $payment]);
    }

    public function postProfile(UserProfileRequest $request)
    {
        // trim all input
        Input::merge(array_map(function ($value) {
            if (is_string($value)) {
                return trim($value);
            } else {
                return $value;
            }
        }, Input::all()));

        $data = $request->all();
        $user = Auth::user();

        if ($data['mobile'] != $user['mobile']) {
            $data['mobile_verified'] = false;
        }

        $user->fill($data);
        $user->save();

        return $this->returnBackWithSuccess();
    }

    public function publicProfile($id)
    {
        $user = User::findOrFail($id);
        return view('user/public-profile', compact('user'));
    }

    public function postProfilePassword(UserProfilePasswordRequest $request)
    {
        $data = $request->all();
        $user = Auth::user();
        if ($data['password'] && $data['password_new'] == $data['password_confirmation']) {
            $password['password'] = bcrypt($data['password_new']);

            $user->fill($password);
            $user->save();

            return $this->returnHomeWithSuccess('Passwort erfolgreich gespeichert.');
        }
    }

    public function getImage($upload_portrait)
    {

        $path = storage_path('app/portraits/') . $upload_portrait;

        return response()->download($path, null, [], 'inline');
    }

    public function createUserView()
    {
        return view('new-user');
    }

    public function createUser(Request $request)
    {
        $this->validate($request, [
            'email' => 'required|email',
            'password' => 'required'
        ]);

        Input::merge(array_map(function ($value) {
            if (is_string($value)) {
                return trim($value);
            } else {
                return $value;
            }
        }, Input::all()));

        $data = $request->all();
        $replyToUser = Auth::user();

        $user = [
            'firstname' => $data['firstname'],
            'surname' => $data['surname'],
            'street' => $data['street'],
            'housenumber' => $data['housenumber'],
            'zip' => $data['zip'],
            'city' => $data['city'],
            'iban' => $data['iban'],
            'owner' => $data['owner'],
            'mobile' => $data['mobile'],
            'phone' => $data['phone'],
            'fax' => $data['fax'],
            'tax_id' => $data['tax_id'],
            'company' => $data['company'],
            'payment' => $data['payment'],
            'email' => $data['email'],
            'status' => 'pending',
            'password' => bcrypt($data['password']),
            'alias' => time(),
        ];
        User::create($user);
        $user->alias = $user->id;
        $user->save();

        $user = User::where('email', $data['email'])->first();
        if (isset($data['groups'])) {
            foreach ($data['groups'] as $group) {
                $user->assignRole($group);
                if ($group == 'agent' || $group == 'consumer') {
                    $user->givePermissionTo('edit account');
                }
            }
        }
        $user->sendMail('user-registration-confirmation-recoma', compact('userData'));

        return redirect(url('/activate'));
    }

    // FIXME: split out fieldservice to own action
    public function index()
    {
        if ($this->user->can('show users')) {
            $userGrid = new UserGrid();
        } else {
            if ($this->user->can('show recruited users'))  {
                $userGrid = new FieldserviceUserGrid($this->user);
            } else {
                throw new \Exception('Not allowed'); // should never happen, it's covered by the routing
            }
        }

        $userGrid->exportGlobalFilters($userGrid);

        return view('user/index', [
            'grid' => $userGrid,
            'globalFilters' => $userGrid->getGlobalFilters()
        ]);
    }

    public function encryptPasswordIfPresent(&$data)
    {
        if ($data['password'] == '') {
            unset($data['password']);
            unset($data['password_confirmation']);
            return null;
        } else {
            $data['password'] = bcrypt($data['password']);
            return $data['password'];
        }
    }

    public function store(UserRequest $request, $id=null)
    {
        if((auth::user()->hasRole('fieldservice_employee') || (auth::user()->hasRole('operator')))){
            $user = \App\User::withoutGlobalScopes()->find($id);
        }else{
            $user = \App\User::find($id);
        }
        // trim all input
        Input::merge(array_map(function ($value) {
            if (is_string($value)) {
                return trim($value);
            } else {
                return $value;
            }
        }, Input::all()));

        $newData = $request->all();

        // crutch
        if ( !isset($newData['groups']) ) {
            $newData['groups'] = [];
        }

        if (Auth::user()->hasRole('operator')) {
            // $user = User::find($newData['id']);
            if (!isset($user->id) || Auth::user()->id != $user->id) { // operator is editing someone
                return $this->operatorCreatesOrEditsUser($newData, $user);
            }
        }

        if (!$this->user->canEditUser($user)) {
            die('Permission denied');
        }

        if (isset($newData['portrait'])) {
            $this->setAvatar($request, $user);
            unset($newData['portrait']);
        }

        if (isset($newData['upload_portrait'])) {
            $this->setLogo($request, $user);
            unset($newData['upload_portrait']);
        }

        // If mobile number change, request new verification
        if (isset($newData['mobile']) && $user['mobile'] != $newData['mobile']) {
            $user->mobile_verified = false;
        }

        if (isset($newData['tan']) && $newData['tan']!='') {
            if ($newData['tan'] == Auth::user()->mobile_verification_token) {
               $this->verifyMobile($newData['tan']);
               return redirect::back();
            } else {
                return Redirect::to('/user/profile')->with('notification', array(
                    'type' => 'warning',
                    'title' => 'Fehler',
                    'description' => 'Der Eingegebene Code stimmt nicht mit ihrem überein.'
                ));
            }
        }

        if (!empty($newData['password'])) {
            $newData['password'] = bcrypt($newData['password']);
        } else {
            unset($newData['password']);
        }

        if (!$user->hasRole('seller-active') && $user->hasRole('seller')) {
            $user->assignRole('seller-active');
        }

        $user->fill($newData);
        $user->save();

        if ($user->hasPermissionTo('edit account')) {
            if (isset($newData['wt1'])) {
                $newData['worktimes'] = json_encode([
                    $newData['wt1'],
                    $newData['wt2'],
                    $newData['wt3'],
                    $newData['wt4'],
                    $newData['wt5'],
                    $newData['wt6'],
                    $newData['wt7'],
                ]);
            }
            $account = Auth::user()->account;
            if (isset($newData['iban']) && isset($newData['iban_owner']) || Auth::user()->hasRole('seller') || Auth::user()->hasRole('operator') || Auth::user()->hasRole('fieldservice_employee')) {
                if(Auth::user()->hasRole('fieldservice_employee') && $user->id != Auth::user()->id){
                    //ADM edits Seller that he has created
                    $account = Account::withoutGlobalScopes()->find($user->account_id);
                    $account->fill($newData);
                    $account->save();
                }else{
                    $account->fill($newData);
                    $account->save();
                }
            }
        }

        return $this->returnHomeWithSuccess('Die Benutzerdaten wurden aktualisiert.');
    }

    protected function operatorCreatesOrEditsUser($newData, $user){
        if (in_array('seller', $newData['groups']) && in_array('consumer', $newData['groups'])) {
            if ($user->hasRole('seller')) {
                $description = 'Ein Unternehmen darf kein Verbraucher sein';
            } else {
                $description = 'Ein Verbraucher darf kein Unternehmen sein';
            }

            return Redirect::to('/user')->with('notification', array(
                'type' => 'error',
                'title' => 'Fehler',
                'description' => $description
            ));
        }

        if (!isset($user->id)) {
            $token = str_random(25);
            $description = 'Der Benutzer wurde angelegt.';
            $user = new User;
            $password = str_random(15);
            $newData['password'] = bcrypt($password);
            $newData['email_verification_token'] = $token;

            //create a new account
            if($newData['account_id']==-1){
                $newAccount = Account::create([
                    'status'=>$newData['status'],
                    'all_connections'=>0
                ]);
                $newData['account_id'] = $newAccount->id;
            }

            $user->fill($newData);
            $user->save();

            $url = url('/activate', ['id' => $user->id, 'token' => $token]);

            $user->sendMail('operator-creates-new-user', compact('newData', 'url', 'password'));

        } else {
            $description = 'Die Benutzerdaten wurden aktualisiert.';

            $oldData = $user->toArray();
            if ($user['status'] == "pending" AND $newData['status'] == 'active') {
                $user->sendMail('new-user-registration', compact('user'));
            } else {
                $user->sendMail('change-user-profile', compact('oldData', 'user', 'newData'));
            }

            if (isset($newData['password'])) {
                $this->encryptPasswordIfPresent($newData);
            }
            $user->fill($newData);
        }

        try{
            $user->save();
        }catch(UnsupportedException $e){
            return back()->with('notification', $this->notificationError($e->getMessage()));
        }


        if (isset($newData['groupsUpdate'])) {
            $roles = Role::all();
            foreach ($roles as $role) {
                if ($role != 'operator') {
                    $user->removeRole($role);
                }
            }
            $user->load('roles'); // Scheinbar ist der reload Notwendig, damit die gelöschten Rollen in das User MOdel übernommen werden. Siehe auch https://laracasts.com/discuss/channels/testing/refresh-a-model
            if (isset($newData['groups'])) {
                foreach ($newData['groups'] as $group) {
                    $user->assignRole($group);
                    if ($group == 'seller' && $newData['status'] == 'active') {
                        $user->assignRole('seller-active');
                    }
                    if($group == User::AGENT){
                        if($user->hasPermissionTo('edit account')){

                        }else{
                            $user->givePermissionTo('edit account');

                        }
                    }
                }
            }
            $user->load('roles'); // Siehe Zeile 10 Zeilen weiter oben
        }
        return Redirect::to('/user')->with('notification', array(
            'type' => 'success',
            'title' => 'Erfolg',
            'description' => $description
        ));
    }

    protected function setCompanyLogo($request, $user)
    {
        $this->validate($request, [
            'logo' => 'image|mimes:jpeg,png,gif|max:2048'
        ], [], [
            'logo' => 'Firmenlogo',
        ]);

        $account = $user->account;

        if ( $account['upload_portrait'] != '' && file_exists(storage_path('app/portraits/') . $account['upload_portrait']) ) {
            unlink(storage_path('app/portraits/') . $account['upload_portrait']);
        }

        $logo = $request->file('logo');

        $name = preg_replace('/[^[:alnum:].]/', '', $logo->getClientOriginalName());
        $logoName = $account->id . '_logo_' . $name;

        $account->upload_portrait = $logoName;
        $account->save();

        $logo->move(storage_path('app/portraits'), $logoName);

        if ( 'image/svg+xml' !== File::mimeType(storage_path('app/portraits/') . $logoName) ) {
            try {
                $logoImage = Image::make(storage_path('app/portraits/') . $logoName);
                $width = $logoImage->width() < $logoImage->height() ? null : 200;
                $height = $logoImage->width() < $logoImage->height() ? 200 : null;
                $logoImage->resize($width, $height, function ($constraint) {
                    $constraint->aspectRatio();
                    $constraint->upsize();
                });
                $logoImage->save();

                $quad = Image::canvas(200, 200);
                $quad->insert(storage_path('app/portraits/') . $logoName, 'center');
                $quad->save(storage_path('app/portraits/') . $account->id . '_logo_200.png');
            } catch (\Exception $e) {
                // do nothing
            }
        }
    }

    /**
     * Method to set user avastar (userpic).
     *
     * @param Request $request
     * @param User $user
     */
    protected function setAvatar(Request $request, User $user)
    {
        $this->validate($request, [
            'portrait' => 'image|mimes:jpeg,png,gif|max:2048'
        ], [], [
            'portrait' => 'Benutzerportrait',
        ]);

        if ( $user['upload_portrait'] != '' && file_exists(storage_path('app/portraits/') . $user['upload_portrait']) ) {
            unlink(storage_path('app/portraits/') . $user['upload_portrait']);
        }

        $portrait = $request->file('portrait');

        $name = preg_replace('/[^[:alnum:].]/', '', $portrait->getClientOriginalName());
        $portraitName = Auth::id() . '_profile_' . $name;

        $user->upload_portrait = $portraitName;

        $portrait->move(storage_path('app/portraits'), $portraitName);

        try {
            $portraitImage = Image::make(storage_path('app/portraits/') . $portraitName);
            $portraitImage->fit(200, 200);
            $portraitImage->save();
        } catch (\Exception $e) {
            // do nothing
        }
    }

    public function setGroup(User $user, $data)
    {
        if (isset($data['groupsUpdate'])) {
            $roles = Role::all();
            foreach ($roles as $role) {
                if (!$role == 'operator') {
                    $user->removeRole($role);
                }
            }
            if (isset($data['groups'])) {
                foreach ($data['groups'] as $group) {
                    $user->assignRole($group);
                    if ($group == 'seller' && $data['status'] == 'active') {
                        $user->assignRole('seller-active');
                    }
                }
            }
        }
        return $user;
    }

    public function storeAccount(AccountRequest $request, Account $account)
    {
        $newData = $request->all();

        if (Auth::user()->hasRole('operator')) {
            // $user = User::find($newData['id']);

            if (!isset($account->id) || Auth::user()->account->id != $account->id) { // operator is editing someones account

                if (!isset($account->id)) {
                    $isNew = true;
                    $account = new Account;

                } else {
                    $isNew = false;

                    $oldData = $account->toArray();
                    $account->sendMailToUsers('change-account-data', compact('oldData', 'account', 'newData'));
                }

                if(isset($newData['trial'])){
                    if($newData['trial'] == 'free'){
                        $account->billed_until = Carbon::createFromDate('9999', '12', '31')->toDateString();
                    }else{
                        $account->billed_until = Carbon::now()->addMonth($newData['trial'])->toDateString();
                    }
                }

                $account->fill($newData);
                $account->save();

                if ($isNew) {
                    return Redirect::to('/user')->with('notification', array(
                        'type' => 'success',
                        'title' => 'Erfolg',
                        'description' => 'Der Account wurde angelegt.'
                    ));
                } else {
                    return Redirect::to('/user')->with('notification', array(
                        'type' => 'success',
                        'title' => 'Erfolg',
                        'description' => 'Die Accountdaten wurden aktualisiert.'
                    ));
                }
            }
        }


        $account->fill($newData);
        $account->save();

        //Operator edits account
        if (Auth::user()->account != $account) {
            return Redirect::to('/user')->with('notification', array(
                'type' => 'success',
                'title' => 'Erfolg',
                'description' => 'Account Daten erfolgreich gespeichert.'
            ));
        }

        return $this->returnHomeWithSuccess('Der Account wurde erfolgreich geändert.');
    }

    public function delete(Request $request)
    {
        $data = $request->all();
        User::destroy($data['id']);

        return redirect(url('/user'));
    }

    public function sendVerifyMobileToken()
    {
        $user = Auth::user();
        if (!$user->mobile_verified) {

            do {
                try {
                    $user->mobile_verification_token = strtoupper(str_random(5));
                    $user->save();
                    $mobileVerificationTokenSaved = true;
                } catch (Exception $e) {
                    $mobileVerificationTokenSaved = false;
                }
            } while (!$mobileVerificationTokenSaved);

            $url = url('/user/mobile/verify/' . $user->mobile_verification_token);
            $code = $user->mobile_verification_token;

            $url = Shortlink::shorten($url);

            SMS::queue('sms.sms-verification', compact('user', 'url', 'code'), function ($message) use ($user) {
                $message->from('reco.ma');
                $message->to($user->mobile);
            });

            return Redirect::to('/user/profile#mobile')->with('notification', array(
                'type' => 'success',
                'title' => 'Erfolg',
                'description' => 'Ein Verifizierungslink wurde ihnen zugesendet.'
            ));
        } else {
            return Redirect::to('/user/profile#mobile')->with('notification', array(
                'type' => 'warning',
                'title' => 'Bereits verifiziert',
                'description' => 'Ihre Handynummer ' . $user->mobile . ' ist bereits verifiziert.'
            ));
        }
    }

    public function verifyMobile($token)
    {
        if (!$token) {
            die('Error - no token');
        }
        $user = User::where('mobile_verification_token', $token)->first();
        if ($user) {
            $user->mobile_verified = 1;
            $user->mobile_verification_token = null;
            $user->save();
            return Redirect::to('/user/profile#mobile')->with('notification', array(
                'type' => 'success',
                'title' => 'Nummer verifiziert',
                'description' => 'Ihre Handynummer wurde erfolgreich verifiziert.'
            ));
        }
        return Redirect::to('/user/profile#mobile')->with('notification', array(
            'type' => 'warning',
            'title' => 'Nummer bereits verifiziert',
            'description' => 'Ihre Handynummer wurde bereits erfolgreich verifiziert.'
        ));
    }

    public function rate()
    {
        $ownRatings = Rating::where('rated_account_id', '=', Auth::user()->account_id)->get();

        $rating = new RatingGrid();

        return view('user/rate', [
            'ownRatings' => $ownRatings,
            'rating' => $rating
        ]);
    }

    public function sendMailToUser(CouponSendRequest $request)
    {
        $data = $request->all();
        $currentUser = Auth::user();
        $invitation = LinkInvitations::where('id', $request->tokenId)->first();
        $data['url'] = url('registration' . '/' . $invitation->token);


        if ($invitation->coupon_template_id == null && $invitation->shared_account_id == null) {
            $data['inviter'] = $currentUser;
            $data['emailtext'] = 'TEST';
            if($currentUser->hasRole('seller')){
                $template = 'invite-seller-as-seller-to-recoma';
            }else{
                $template = 'invite-seller-to-recoma';
            }
            if ($invitation->reason == "invite-new-agent") {
                $data['sellerAccount'] = $currentUser->account;
                $template = 'seller-invite-new-agent';
            }
        } elseif ($invitation->coupon_template_id == null)//share seller
        {
            $sellerAccount = Account::where('id', $invitation->shared_account_id)->first();

            $template = 'agent-sends-coupon-to-new-consumer';
            $data['sellerAccount'] = $sellerAccount;
            $data['agent'] = $currentUser;
            $sellerAccount->sendMailToUsers('shared-seller-notice-with-no-registered-consumer', compact('user', 'currentUser'), true);

        } else//share coupon
        {
            $couponTemplate = CouponTemplate::where('id', $invitation->coupon_template_id)->first();
            $sellerAccount = $couponTemplate->user->account;

            $template = 'agent-sends-coupon-to-new-consumer';
            $data['sellerAccount'] = $sellerAccount;
            $data['agent'] = $currentUser;
            $sellerAccount->sendMailToUsers('shared-seller-notice-with-no-registered-consumer', compact('user', 'currentUser', 'couponTemplate'), true);

        };
        if ($invitation !== null && $invitation->inviting_user_id == $currentUser->id) {
            $userData = [
                'salutation' => $request->salutation,
                'firstname' => $request->firstname,
                'surname' => $request->surname,
                'email' => $request->email,
            ];
            $toUser = new User();
            $toUser->fill($userData);
        } else {
            //FIXME Throw exception?
            return null;
        }
        //FIXME Always new user-template, check if user is new or already exists?
        // if exists : 'agent-sends-coupon-to-consumer'


        $toUser->sendMail($template, $data);

        if(isset($data['fromOverlay'])){
            return redirect('onboarding-agent')
                ->with('notification', $this->notificationSuccess('Seller erfolgreich empfohlen.'));
        }

        //FIXME macht vielleicht keinen sinn, coupon wird ja nicht angezeigt, redirect auf Dashboard?
        return redirect('coupon/redeem');
    }

    public function sendSMSToUser(CouponSendRequest $request)
    {
        $data = $request->all();
        $currentUser = Auth::user();
        $invitation = LinkInvitations::where('id', $request->tokenId)->first();
        $data['url'] = Shortlink::shorten(url('registration' . '/' . $invitation->token));

        if ($invitation->coupon_template_id == null && $invitation->shared_account_id == null) {
            $data['inviter'] = $currentUser;

            $template = 'sms.agent-sends-coupon-to-consumer';

        } elseif ($invitation->coupon_template_id == null)//share seller
        {
            $sellerAccount = Account::where('id', $invitation->shared_account_id)->first();

            $template = 'sms.agent-sends-coupon-to-consumer';
            $data['sellerAccount'] = $sellerAccount;
            $data['agent'] = $currentUser;

        } else//share coupon
        {
            $couponTemplate = CouponTemplate::where('id', $invitation->coupon_template_id)->first();
            $sellerAccount = $couponTemplate->user->account;

            $template = 'sms.agent-sends-coupon-to-consumer';
            $data['sellerAccount'] = $sellerAccount;
            $data['agent'] = $currentUser;

        };
        if ($invitation !== null && $invitation->inviting_user_id == $currentUser->id) {
            $userData = [
                'salutation' => $request->salutation,
                'firstname' => $request->firstname,
                'surname' => $request->surname,
                'mobile' => $request->mobile,
            ];
            $toUser = new User();
            $toUser->fill($userData);
        } else {
            //FIXME Throw exception?
            return null;
        }


        //FIXME Always new user-template, check if user is new or already exists?
        // if exists : 'agent-sends-coupon-to-consumer'


        SMS::queue($template,
            ['url' => $data['url'], 'smsText' => $data['smsText']],
            function ($message) use ($toUser, $currentUser) {
                $message->from($currentUser->mobile);
                $message->to($toUser->mobile);
            });


        if(isset($data['fromOverlay'])){
            return redirect('onboarding-agent')
                ->with('notification', $this->notificationSuccess('Seller erfolgreich empfohlen.'));
        }

        //FIXME macht vielleicht keinen sinn, coupon wird ja nicht angezeigt, redirect auf Dashboard?
        return redirect('coupon/redeem');
    }

    /**
     * Create a new user (recruit by fieldservice)
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function createRecurited()
    {
        $user = new User();
        $account = new Account();

        $userAccounts = Account::where('company', 'not like', '')->get();

        return view('user/edit', [
            'user' => $user,
            'account' => $account,
            'userAccounts' => $userAccounts,
            'requestClass' => \App\Http\Requests\FullSellerRegistrationRequest::class
        ]);
    }

    /**
     * Fieldservice employee creates new seller
     * @param SellerRegistrationRequest $request FIXME: request has to be changed to the full one
     */
    public function storeCreateRecruited(FullSellerRegistrationRequest $request){
        $data = $request->all();
        $registrationService = new SellerRegistrationService($data);
        $registrationService->setRecruiter($this->user);
        $registrationService->setTrial($data['trial']);
        $registrationService->createInitialPassword();
        $user = $registrationService->register(true);
        $registrationService->sendDoubleOptInMail($user, 'fieldservice-creates-new-user');

        return Redirect::to('/user')
            ->with('notification', $this->notificationSuccess('Das Unternehmen wurde angelegt.'));
    }

    public static function sendAgentInvitationFromOverlay($token,$email)
    {
        $currentUser = Auth::user();
        $data['url'] = url('registration' . '/' . $token);
        $data['inviter'] = $currentUser;
        $data['emailtext'] = 'TEST';
        $data['sellerAccount'] = $currentUser->account;
        $template = 'seller-invite-new-agent-from-overlay';


        $userData = [
            'salutation' => '',
            'firstname' => '',
            'surname' => '',
            'email' =>$email,
        ];
        $toUser = new User();
        $toUser->fill($userData);

        $toUser->sendMail($template, $data);

        return redirect('/onboarding-seller/')->with('notification', array(
            'type' => 'success',
            'title' => 'Erfolg',
            'description' => 'Einladung erfolgreich versendet.'
        ));
    }

    public static function agentRecommendsSellerFromOverlay($accountSeller)
    {
        $currentUser = Auth::user();
        $data['agentAccount'] = $currentUser->account;
        $template = 'agent-recommend-seller-from-overlay';

        $accountSeller->users()->first()->sendMail($template, $data);

        return;
    }

    /**
     * Method to set alias.
     *
     * @param AliasRequest $request
     * @return mixed
     */
    public function setAlias(AliasRequest $request)
    {
        $user = Auth::user();
        $user->alias = $request->get('alias');
        $user->save();

        return Redirect::to(route('user-profile'))->with('notification', array(
            'type'        => 'success',
            'title'       => 'Erfolg',
            'description' => ''
        ));
    }

    /**
     * Method to set userpic.
     *
     * @param Request $request
     * @return mixed
     */
    public function setUserpic(Request $request)
    {
        $user = Auth::user();

        $this->setAvatar($request, $user);
        $user->save();

        activity($user)->register('userpic edited', null, ['userpic' => $user->upload_portrait]);

        $route = (new RouteService($user))->getProfileRoute();

        return redirect()
            ->to($route)
            ->with('notification', $this->_s('Benutzerportrait wurde erfolgreich hochgeladen'));
    }


    /**
     * Method to set logo.
     *
     * @param Request $request
     * @return mixed
     */
    public function setLogo(Request $request)
    {
        $user   = Auth::user();
        $seller = $this->getSeller();

        $this->setCompanyLogo($request, $seller);

        activity($user)->register('logo edited', null, ['logo' => $seller->account->upload_portrait]);

        return redirect()
            ->to(route('seller.company.index'))
            ->with('notification', $this->_s('Firmenlogo wurde erfolgreich hochgeladen'));
    }

    /**
     * Method to set company picture.
     *
     * @param CompanyPictureRequest $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function setPicture(CompanyPictureRequest $request)
    {
        $user   = Auth::user();
        $seller = $this->getSeller();

        $seller->account->picture = $request->file('picture');
        $seller->account->save();

        activity($user)->register('picture edited', null, ['picture' => $seller->account->picture]);

        return redirect()
            ->to(route('seller.company.index'))
            ->with('notification', $this->_s('Firmenbild wurde erfolgreich hochgeladen'));
    }

    /**
     * Method to show user's profile.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function userProfile()
    {
        $user = Auth::user();

        $actions = Action::whereIn('action', ['logged in', 'failed login attempt'])
            ->pluck('id', 'action');

        $loginAttempts = Activity::with('useragent', 'location')
            ->where('user_id', $user->id)
            ->where('action_id', $actions['logged in'])
            ->latest()
            ->limit(5)
            ->get();

        if ( isset($actions['failed login attempt']) ) {
            $failedAttempts = Activity::with('useragent', 'location')
                ->where('user_id', $user->id)
                ->where('action_id', $actions['failed login attempt'])
                ->latest()
                ->limit(5)
                ->get();
        } else {
            $failedAttempts = collect([]);
        }

        $sessionID = session()->getId();

        $activeSessions = $user->sessions()
            ->with('useragent', 'location')
            ->get()
            ->map(function ($item) use ($sessionID) {
                $item->current = $item->id === $sessionID;
                return $item;
            })->sortByDesc('current');

        session(['paypal_redirect' => route('user.profile') . '#payment']);

        return view('redesign.front.user.profile', [
            'user' => $user,
            'loginAttempts'  => $loginAttempts,
            'failedAttempts' => $failedAttempts,
            'activeSessions' => $activeSessions,
        ]);
    }

    /**
     * Method to show user's favorites.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function favorites()
    {
        $user = Auth::user();

        $favs = $user->favorites()
            ->with(['seller.account', 'provisionGift', 'discountGift', 'media', 'category'])
            ->get();

        return view('redesign.front.fav.index', [
            'favs' => $favs,
        ]);
    }

    /**
     * Method to set user's name.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function setName(Request $request)
    {
        $user = Auth::user();
        $user->firstname = trim($request->name);
        $user->surname = trim($request->surname);
        $user->save();

        return redirect()->back();
    }

    /**
     * Method to set user's banking data.
     *
     * @param BankingDataRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function setBankingData(BankingDataRequest $request)
    {
        $user = Auth::user();
        $user->account->iban = trim($request->iban);
        $user->account->iban_owner = trim($request->iban_owner);
        $user->account->save();

        return redirect()->back();
    }

    /**
     * Method to show user's reviews page.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function reviews()
    {
        $reviews = Rating::with(['user', 'account', 'account.users', 'deal'])
            ->where('user_id', Auth::id())
            ->latest()
            ->get();

        return view('redesign.front.user.reviews', [
            'reviews' => $reviews,
        ]);
    }

    /**
     * Method to show user's account page.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function transactions()
    {
        $user = Auth::user();

        $transactions = Transaction::with(['coupon.template.seller.account', 'coupon.agent', 'coupon.consumer', 'credit_note'])
            ->where('user_id', $user->id)
            ->where('amount', '>', 0)
            ->latest()
            ->get();

        $unratedCount = 0;
        foreach ($transactions as $transaction) {
            if ( Coupon::STATUS_PAID === $transaction->coupon->status && $user->id === $transaction->coupon->consumer->id ) {
                $unratedCount++;
            }
        }

        return view('redesign.front.user.transactions', [
            'transactions' => $transactions,
            'unratedCount' => $unratedCount,
        ]);
    }

    /**
     * Method to show user's coupons page.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function coupons()
    {
        $user = Auth::user();

        $coupons = Coupon::with('template.seller.account', 'provisionGift', 'discountGift', 'template.media')
            ->where(function ($coupons) use ($user) {
                return $coupons
                    ->where('consumer_user_id', $user->id)
                    ->orWhere('agent_user_id', $user->id);
            })
            ->latest()
            ->get();

        $coupons = $coupons->filter(function ($value) {
            return !(Coupon::STATUS_ACTIVE === $value->status && Carbon::today() >= Carbon::parse($value->validity_date));
        });

        $coupons = $coupons->groupBy(function ($item) use ($user) {
            return ( $item->consumer_user_id === $item->agent_user_id || $item->consumer_user_id === $user->id ) ? 'consumer' : 'agent';
        });

        $coupons['consumer'] = isset($coupons['consumer'])
            ? $coupons['consumer']->groupBy('status')
            : collect([]);

        $coupons['agent'] = isset($coupons['agent'])
            ? $coupons['agent']->groupBy('status')
            : collect([]);

        return view('redesign.front.user.coupons', [
            'coupons' => $coupons,
        ]);
    }

    /**
     * Method to show user's referrals page.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function referrals()
    {
        $user = Auth::user();

        $links = (new ReferralLinkService())->getAllLinks();

        $referrals = $user->referrals;

        $coupons = Coupon::where('agent_user_id', $user->id)
            ->where('consumer_user_id', '!=', $user->id)
            ->groupBy('consumer_user_id')
            ->get()
            ->keyBy('consumer_user_id');

        $attachedSeller = $user->getAttachedSeller();

        return view('redesign.front.user.referrals', [
            'links'          => $links,
            'coupons'        => $coupons,
            'referrals'      => $referrals,
            'attachedSeller' => $attachedSeller,
        ]);
    }

    /**
     * Method to show user's seller invite page.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function sellers()
    {
        $user = Auth::user();

        $links   = (new SellerInvitationLinkService())->getAllLinks();
        $sellers = $user->invitedSellers()->with('account')->get();

        return view('redesign.front.user.sellers', [
            'links'   => $links,
            'sellers' => $sellers,
        ]);
    }

    /**
     * Method to show user's payout page.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function payout()
    {
        $user = Auth::user();

        if ( !$user->email_verified ) {
            return redirect(route('user.transactions'))->with('notification', $this->_e('Bitte bestätige Deine E-Mail, damit Du Dein Cashback auszahlen kannst.'));
        }

        session(['paypal_redirect' => route('user.payout') . '?p']);

        $transactions = Transaction::with('coupon.template.ratings', 'coupon.consumer')
            ->where('user_id', $user->id)
            ->where('amount', '>', 0)
            ->where('status', Transaction::STATUS_ALLOCATED)
            ->latest()
            ->get();

        return view('redesign.front.user.payout', [
            'user'         => $user,
            'transactions' => $transactions,
        ]);
    }

    /**
     * Method to flush all sessions except current.
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function flushSessions()
    {
        $user = Auth::user();

        (new UserService($user))->flushSessions();

        $redirectRoute = (new RouteService($user))->getProfileRoute() . '#security';

        return redirect()
            ->to($redirectRoute)
            ->with('notification', $this->_s('Du hast von anderen Geräten erfolgreich abgemeldet'));
    }
}
