<?php

namespace App\Http\Controllers;

use App\Coupon;
use App\CreditNote;
use App\Http\Grids\AbstractAccountGrid;
use App\Http\Grids\AgentTransactionGrid;
use App\Http\Grids\ConsumerTransactionGrid;
use App\Http\Grids\TransactionGrid;
use App\Http\Requests;
use App\Invoice;
use App\Library\Facades\Format;
use App\Services\StatisticsService;
use App\Transaction;
use App\User;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Redirect;
use App\Rating;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Symfony\Component\HttpFoundation\File\File;

class TransactionController extends Controller
{
    public function _construct()
    {
        parent::__construct();
        $this->middleware(['auth', 'completed.profile']);
    }

    /**
     * Calculate Numbers and Build array
     */
    public function calculateTransactionNumbers()
    {
        $openTransactions = [];
        $agentProvision = 0;
        $consumerDiscount = 0;
        $operatorProvision = 0;

        $openOperatorAmount = 0;
        $openOperatorAmounts = DB::table('transactions')
            ->where('type', '=', 'operator-coupon-provision-credit')
            ->whereNotIn('status', ['pending', 'paid'])
            ->get();
        foreach ($openOperatorAmounts as $operatorAmount) {
            $openOperatorAmount += $operatorAmount->amount;
        }

        $openConsumerMembership = 0;
        $openConsumerMemberships = DB::table('transactions')
            ->whereIn('type', Transaction::TYPE_FEE)
            ->whereNotIn('status', ['paid'])
            ->get();
        foreach ($openConsumerMemberships as $consumerMembership) {
            $openConsumerMembership += $consumerMembership->amount;
        }

        if (!Auth::user()->hasRole('operator')) {
            $openTransactions = Auth::user()->transactions()->whereStatus('allocated')->get();
            $allocatedTransactions = Auth::user()->transactions()
                ->join('credit_notes', 'credit_note_id', '=', 'credit_notes.id', 'left')
                ->where('credit_notes.status', '=', CreditNote::STATUS_COMPLETE)
                ->get();
            $paidTransactions = Auth::user()->transactions()->where('status', '=', 'paid')->get();
        }

        if ($this->user->hasRole('operator')) {
            $baseTransactionsQuery = Transaction::getQuery();
        } else {
            $baseTransactionsQuery = $this->user->transactions()->getQuery();
        }

        $query = clone $baseTransactionsQuery;
        $transactionsQuery = $query->where('status', '<>', Transaction::STATUS_PAID);
        $sums = $transactionsQuery->groupBy('type')
            ->selectRaw('SUM(amount) as sum, type')
            ->pluck('sum', 'type');

        $query = clone $baseTransactionsQuery;
        $paidTransactionsQuery = $query->where('status', '=', Transaction::STATUS_PAID);
        $paidSums = $paidTransactionsQuery->groupBy('type')
            ->selectRaw('SUM(amount) as sum, type')
            ->pluck('sum', 'type');

        $numbers = [];

        if ($this->user->hasRole('seller')) {
            $numbers[] =
                [
                    'name' => 'Umsätze',
                    'number' => Format::currency($this->user->account->sellerCalculateCouponsNetto()),
                    'icon' => 'bar-chart'
                ];
            $numbers[] =
                [
                    'name' => 'Offenes Cashback',
                    'number' => Format::currency($sums[Transaction::TYPE_SELLER_CONSUMER_DISCOUNT_PAYMENT] ?? 0),
                    'icon' => 'money'
                ];
            $numbers[] =
                [
                    'name' => 'Offenes RECO.CASH',
                    'number' => Format::currency($sums[Transaction::TYPE_SELLER_AGENT_PROVISION_PAYMENT] ?? 0),
                    'icon' => 'money-1'
                ];

            $numbers[] =
                [
                    'name' => 'Offene Gebühren',
                    'number' => Format::currency(
                        ($sums[Transaction::TYPE_SELLER_OPERATOR_PROVISION_PAYMENT] ?? 0)
                        + ($sums[Transaction::TYPE_SELLER_FEE_YEARLY_PAYMENT] ?? 0)
                        + ($sums[Transaction::TYPE_SELLER_FEE_MONTHLY_PAYMENT] ?? 0)
                    ),
                    'icon' => 'dollar-symbol'
                ];
        }

        if ($this->user->hasRole('operator')) {
            $numbers[] =
                [
                    'name' => 'Offene Gebühren',
                    'number' => Format::currency(
                        $sums[Transaction::TYPE_SELLER_OPERATOR_PROVISION_PAYMENT] ?? 0
                    ),
                    'icon' => 'dollar-symbol'
                ];
            $numbers[] =
                [
                    'name' => 'Offene Mitgliedsbeträge',
                    'number' => Format::currency(
                        - ($sums[Transaction::TYPE_SELLER_FEE_YEARLY_PAYMENT] ?? 0)
                        - ($sums[Transaction::TYPE_SELLER_FEE_MONTHLY_PAYMENT] ?? 0)
                    ),
                    'icon' => 'dollar-symbol-1'
                ];
            $numbers[] =
                [
                    'name' => 'Eingezogene Gebühren',
                    'number' => Format::currency(
                        $paidSums[Transaction::TYPE_SELLER_OPERATOR_PROVISION_PAYMENT] ?? 0
                    ),
                    'icon' => 'dollar-symbol'
                ];
            $numbers[] =
                [
                    'name' => 'Eingezogen Mitgliedsbeträge',
                    'number' => Format::currency(
                        - ($paidSums[Transaction::TYPE_SELLER_FEE_YEARLY_PAYMENT] ?? 0)
                        - ($paidSums[Transaction::TYPE_SELLER_FEE_MONTHLY_PAYMENT] ?? 0)
                    ),
                    'icon' => 'dollar-symbol-1'
                ];
        }

        return $numbers;
    }


    public function index()
    {
        if (!Auth::user()->hasRole('operator')) {
            return redirect('/');
        }

        $grid = new \App\Http\Grids\TransactionGrid();
        $globalFilters = $grid->getGlobalFilters();

        return view('transaction.index', compact('grid', 'globalFilters'))
            ->with($this->getTransactionStatus())
            ->with('numbers', $this->calculateTransactionNumbers())
            ->with($this->processPaymentsRequest(false));
    }

    public function rate(Request $request)
    {
        $data = $request->all();

        $transaction = Transaction::find($data['transaction_id']);
        $coupon = Coupon::find($transaction['coupon_id']);

        if (Auth::user()->hasRole('agent')) {
            $rateableUserType = 'agent';
        }
        if (Auth::user()->hasRole('consumer')) {
            $rateableUserType = 'consumer';
        }

        $rating = [
            'rating' => $data['rating'],
            'user_id' => Auth::user()->id,
            'rateable_id' => $coupon['id'],
            'description' => $data['description'],
            'rateable_type' => Coupon::class,
            'rateable_user_type' => $rateableUserType,
            'rated_account_id' => $transaction->coupon->template->seller->account->id
        ];

        /*
        $transaction['status'] = 'allocated';

        if ($transaction->credit_note->status == CreditNote::STATUS_OPEN) {
            $transaction->credit_note->status = CreditNote::STATUS_PAYMENT_REQUESTED;
        }*/
/*
 *
        $transaction->save();*/

        $coupon->status = Coupon::STATUS_DONE;
        $coupon->save();
        Rating::create($rating);

        $transaction->releaseEligible();

        return Redirect::to('/transaction')->with('notification', array(
            'type' => 'success',
            'title' => 'Erfolg',
            'description' => 'Danke für Ihre Bewertung.'
        ));
    }

    public function sepaDebitDownload()
    {
        $fileName = Invoice::generateSepaDebitXml();
        if ($fileName === false) {
            return $this->returnBackWithError('Keine zu exportierenden Rechnungen vorhanden.');
        }
        $name = pathinfo($fileName, PATHINFO_FILENAME);

        Session::flash('download.in.the.next.request', '/transaction/' . $name);
        return Redirect::to('transaction')->with('notification', array(
            'type' => 'success',
            'title' => 'Erfolg',
            'description' => 'Die Rechnungen wurde exportiert'
        ));
    }

    public function sepaCreditDownload()
    {
        $fileName = CreditNote::generateSepaCreditXml();
        if ($fileName === false) {
            return $this->returnBackWithError('Keine zu exportierenden Gutschriften vorhanden.');
        }
        $name = pathinfo($fileName, PATHINFO_FILENAME);

        Session::flash('download.in.the.next.request', '/transaction/' . $name);
        return Redirect::to('transaction')->with('notification', array(
            'type' => 'success',
            'title' => 'Erfolg',
            'description' => 'Die Gutschriften wurde exportiert'
        ));
    }

    public function download($name){
        $path = storage_path('app/sepa/') . $name . '.xml';
        return response()->download($path);
    }


    protected function processPaymentsRequest($forReal = false)
    {
        return $this->user->processCredits($forReal);
    }

    public function requestAllPayments()
    {
        $this->processPaymentsRequest(true);

        return $this->returnBackWithSuccess('Ihre Auszahlung wird bearbeitet.');
    }

    /**
     * Method to show seller transactions page.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function sellerIndex()
    {
        $seller = $this->getSeller();

        $numbers = StatisticsService::calculateTransactionsNumbers($seller);

        $transactions = Transaction::with('coupon', 'coupon.agent', 'coupon.consumer', 'invoice')
            ->where('user_id', $seller->id)
            ->where('coupon_id', '<>', null)
            ->orderBy('created_at', 'desc')
            ->get()
            ->groupBy('coupon_id');

        return view('redesign.back.metronic.transactions.index', [
            'numbers'      => $numbers,
            'transactions' => $transactions,
        ]);
    }
}
