<?php


namespace App\Http\Controllers;

use App\Action;
use App\Activity;
use App\CouponTemplate;
use App\Http\Grids\AbstractAccountGrid;
use App\Http\Grids\Agent\MySellerGrid;
use App\Http\Grids\Agent\SellerGrid;
use App\Place;
use App\Services\ReferralLinkService;
use App\Services\SellerInvitationLinkService;
use App\User;
use App\Rating;
use Illuminate\Support\Facades\Auth;

class SellerController extends Controller {

    /**
     * Show my sellers, available sellers, with the option to recommend a seller
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index() {
        $myActiveSellers = [];
        $mySellers       = $this->user->getAvailableSellers();

        foreach ( $mySellers as $mySeller ) {
            $couponTemplates = CouponTemplate::where( 'user_id', '=', $mySeller->id )
                                             ->get();

            if ( $couponTemplates != '' ) {
                array_push( $myActiveSellers, $mySeller );
            }
        }

        $mySellerGrid = new MySellerGrid();
        $sellerGrid   = new SellerGrid();
        $globalGrid   = new AbstractAccountGrid();
        $globalGrid->exportGlobalFilters( [ $sellerGrid, $mySellerGrid ] );

        return view( 'sellers/index', [
            'globalFilters'   => $globalGrid->getGlobalFilters(),
            'myseller'        => $mySellerGrid,
            'grid'            => $sellerGrid,
            'mySellers'       => count( $mySellers ),
            'myActiveSellers' => count( $myActiveSellers )
        ] );
    }

    /**
     * Get seller main page by sellers alias.
     *
     * @param $alias
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function showSellerPage($alias)
    {
        $seller = User::where('alias', $alias)->first();

        if ( $seller && $seller->hasRole('seller') ) {
            $deals         = $seller->getAvailableCouponTemplates();
            $places        = $seller->places->where('status', Place::STATUS_ACTIVE);
            $reviews       = Rating::with('user', 'deal.seller')->where('rated_account_id', $seller->account_id)->latest()->get();
            $reviewsAvg    = $reviews->avg('rating');
            $reviewsCount  = $reviews->count();
            $reviewsCounts = [
                1 => $reviews->where('rating', '1')->count(),
                2 => $reviews->where('rating', '2')->count(),
                3 => $reviews->where('rating', '3')->count(),
                4 => $reviews->where('rating', '4')->count(),
                5 => $reviews->where('rating', '5')->count(),
            ];

            return view('redesign.front.seller.index', [
                'deals'         => $deals,
                'places'        => $places,
                'seller'        => $seller,
                'reviews'       => $reviews,
                'reviewsAvg'    => $reviewsAvg,
                'reviewsCount'  => $reviewsCount,
                'reviewsCounts' => $reviewsCounts,
            ]);
        } else {
            abort(404);
        }
    }

    /**
     * Method to show seller share page.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function share()
    {
        return view('redesign.back.metronic.share.index', [
            'deals' => $this->getSeller()->getAvailableCouponTemplates(),
            'links' => (new ReferralLinkService())->getAllLinks(),
        ]);
    }

    /**
     * Method to show seller invite sellers page.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function inviteSellers()
    {
        $seller = $this->getSeller();

        $employeesIds = User::where('boss_id', $seller->id)->pluck('id');
        $employeesIds->push($seller->id);

        $sellers = User::with('account')
            ->whereIn('invited_by', $employeesIds)
            ->get();

        return view('redesign.back.metronic.share.seller', [
            'links'   => (new SellerInvitationLinkService())->getAllLinks(),
            'sellers' => $sellers,
        ]);
    }

    /**
     * Method to show seller reviews page.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function reviews()
    {
        $reviews = Rating::with('user', 'deal')
            ->where('rated_account_id', $this->getSeller()->account_id)
            ->latest()
            ->get();
        $reviewsAvg    = round($reviews->avg('rating'), 2);
        $reviewsCount  = $reviews->count();
        $reviewsCounts = [
            1 => $reviews->where('rating', 1)->count(),
            2 => $reviews->where('rating', 2)->count(),
            3 => $reviews->where('rating', 3)->count(),
            4 => $reviews->where('rating', 4)->count(),
            5 => $reviews->where('rating', 5)->count(),
        ];

        return view('redesign.back.metronic.reviews.index', [
            'reviews'       => $reviews,
            'reviewsAvg'    => $reviewsAvg,
            'reviewsCount'  => $reviewsCount,
            'reviewsCounts' => $reviewsCounts,
        ]);
    }

    /**
     * Method to show seller/employee profile page.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function profile()
    {
        $user = Auth::user();

        $actions = Action::whereIn('action', ['logged in', 'failed login attempt'])
            ->pluck('id', 'action');

        $loginAttempts = Activity::with('useragent', 'location')
            ->where('user_id', $user->id)
            ->where('action_id', $actions['logged in'])
            ->latest()
            ->limit(5)
            ->get();

        $failedAttempts = Activity::with('useragent', 'location')
            ->where('user_id', $user->id)
            ->where('action_id', $actions['failed login attempt'])
            ->latest()
            ->limit(5)
            ->get();

        $sessionID = session()->getId();

        $activeSessions = $user->sessions()
            ->with('useragent', 'location')
            ->get()
            ->map(function ($item) use ($sessionID) {
                $item->current = $item->id === $sessionID;
                return $item;
            })->sortByDesc('current');

        return view('redesign.back.metronic.profile.index', [
            'user'           => $user,
            'loginAttempts'  => $loginAttempts,
            'failedAttempts' => $failedAttempts,
            'activeSessions' => $activeSessions,
        ]);
    }

    /**
     * Method to show company options page.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function company()
    {
        return view('redesign.back.metronic.company.index');
    }

    /**
     * Method to show agents page (users who has seller coupons).
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function agents()
    {
        $coupons = $this->getSeller()->publishedCoupons()
            ->select(['coupons.id', 'coupon_template_id', 'agent_user_id', 'coupons.status'])
            ->with('agent.affiliate')
            ->get()
            ->groupBy('agent_user_id');

        $agents = collect([]);

        foreach ($coupons as $group) {
            $agent = $group->first()->agent;
            $agent->coupon_count = $coupons[$agent->id]->count();
            $agent->redeem_count = $coupons[$agent->id]->filter(function ($coupon) {
                return 'active' !== $coupon->status;
            })->count();
            $agent->success_rate = 100 * $agent->redeem_count / $agent->coupon_count;
            $agents->push($agent);
        }

        return view('redesign.back.metronic.agents.index', [
            'agents' => $agents,
        ]);
    }
}