<?php

namespace App\Http\Controllers;

use App\Http\Requests\PlaceRequest;
use App\Place;
use App\Services\PlaceService;
use Illuminate\Support\Facades\Auth;

class PlacesController extends Controller
{
    /**
     * Method to show seller places page.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index()
    {
        $places = $this->getSeller()->places()->get();

        return view('redesign.back.metronic.places.index', [
            'places' => $places,
        ]);
    }

    /**
     * Method to show place create page.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function store()
    {
        return view('redesign.back.metronic.places.store');
    }

    /**
     * Method to create new place.
     *
     * @param PlaceRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function create(PlaceRequest $request)
    {
        (new PlaceService())->create($request->all());

        return redirect()
            ->to(route('seller.places.index'))
            ->with('notification', $this->_s('Der Ort wurde erfolgreich erstellt'));
    }

    /**
     * Method to show place edit page.
     *
     * @param Place $place
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit(Place $place)
    {
        if ( !Auth::user()->can('update', $place) ) {
            return redirect(route('seller.places.index'));
        }

        return view('redesign.back.metronic.places.edit', [
            'place' => $place,
        ]);
    }

    /**
     * Method to update place.
     *
     * @param PlaceRequest $request
     * @param Place $place
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function update(PlaceRequest $request, Place $place)
    {
        if ( !Auth::user()->can('update', $place) ) {
            return redirect(route('seller.places.index'));
        }

        (new PlaceService($place))->update($request->all());

        return redirect()
            ->to(route('seller.places.index'))
            ->with('notification', $this->_s('Der Ort wurde erfolgreich bearbeitet'));
    }

    /**
     * Method to activate place.
     *
     * @param Place $place
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function activate(Place $place)
    {
        if ( !Auth::user()->can('update', $place) ) {
            return redirect(route('seller.places.index'));
        }

        (new PlaceService($place))->activate();

        return redirect(route('seller.places.index'));
    }

    /**
     * Method to deactivate place.
     *
     * @param Place $place
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function deactivate(Place $place)
    {
        if ( !Auth::user()->can('update', $place) ) {
            return redirect(route('seller.places.index'));
        }

        (new PlaceService($place))->deactivate();

        return redirect(route('seller.places.index'));
    }

    /**
     * Method to delete place.
     *
     * @param Place $place
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     * @throws \Exception
     */
    public function delete(Place $place)
    {
        if ( !Auth::user()->can('update', $place) ) {
            return redirect(route('seller.places.index'));
        }

        (new PlaceService($place))->delete();

        return redirect()
            ->to(route('seller.places.index'))
            ->with('notification', $this->_s('Der Ort wurde erfolgreich entfernt'));
    }
}