<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Redirect;
use App\Gift;
use App\Http\Requests\GiftRequest;

class GiftController extends Controller
{
    /**
     * Method to show gifts list.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index()
    {
        $user = Auth::user();
        $canCreate = $user->hasPermission('create gifts');
        $canEdit   = $user->hasPermission('edit gifts');
        $canDelete = $user->hasPermission('delete gifts');

        return view('gifts.index', [
            'gifts'     => Gift::all(),
            'canCreate' => $canCreate,
            'canEdit'   => $canEdit,
            'canDelete' => $canDelete,
        ]);
    }

    /**
     * Method to show new gift create form.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create()
    {
        return view('gifts.create');
    }

    /**
     * Method to store new gift.
     *
     * @param GiftRequest $request
     * @return mixed
     */
    public function store(GiftRequest $request)
    {
        Gift::create([
            'name'        => $request->get('name'),
            'price'       => $request->get('price'),
            'image'       => $request->file('image'),
            'active'      => (bool)$request->get('active'),
            'description' => $request->get('description'),
        ]);

        return Redirect::to(route('gift-list'))->with('notification', array(
            'type'        => 'success',
            'title'       => 'Erfolg',
            'description' => 'Geschenk erfolgreich erstellt.'
        ));
    }

    /**
     * Method to show gift edit form.
     *
     * @param Gift $gift
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit(Gift $gift)
    {
        return view('gifts.edit', [
            'gift' => $gift,
        ]);
    }

    /**
     * Method to update existed gift.
     *
     * @param GiftRequest $request
     * @param Gift $gift
     * @return mixed
     */
    public function update(GiftRequest $request, Gift $gift)
    {
        $gift->name        = $request->get('name');
        $gift->price       = $request->get('price');
        $gift->active      = (bool)$request->get('active');
        $gift->description = $request->get('description');
        if ($request->file('image')) {
            $gift->image = $request->file('image');
        }

        $gift->save();

        return Redirect::to(route('gift-list'))->with('notification', array(
            'type'        => 'success',
            'title'       => 'Erfolg',
            'description' => 'Geschenk wurde erfolgreich bearbeitet.'
        ));
    }

    /**
     * Method to remove gift.
     *
     * @param Gift $gift
     * @return mixed
     * @throws \Exception
     */
    public function destroy(Gift $gift)
    {
        if ( $gift->hasCoupons() || $gift->hasCouponTemplates() ) {
            return Redirect::to(route('gift-list'))->with('notification', array(
                'type'        => 'error',
                'title'       => 'Fehler',
                'description' => 'Geschenk mit Coupons oder Kampagnen kann nicht entfernt werden.'
            ));
        }

        $gift->delete();

        return Redirect::to(route('gift-list'))->with('notification', array(
            'type'        => 'success',
            'title'       => 'Erfolg',
            'description' => 'Geschenk erfolgreich entfernt.'
        ));
    }

    /**
     * Method to show gift.
     *
     * @param Gift $gift
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function show(Gift $gift)
    {
        return view('gifts.show', ['gift' => $gift]);
    }

    /**
     * Method to activate gift.
     *
     * @param Gift $gift
     * @return mixed
     */
    public function activate(Gift $gift)
    {
        $gift->active = true;
        $gift->save();

        return Redirect::to(route('gift-list'))->with('notification', array(
            'type'        => 'success',
            'title'       => 'Erfolg',
            'description' => 'Geschenk erfolgreich aktiviert.'
        ));
    }

    /**
     * Method to deactivate gift.
     *
     * @param Gift $gift
     * @return mixed
     */
    public function deactivate(Gift $gift)
    {
        $gift->active = false;
        $gift->save();

        return Redirect::to(route('gift-list'))->with('notification', array(
            'type'        => 'success',
            'title'       => 'Erfolg',
            'description' => 'Geschenk erfolgreich deaktiviert.'
        ));
    }
}
