<?php

namespace App\Http\Controllers;


use App\Activity;
use App\Http\Requests\EmployeeRegisterRequest;
use App\User;
use App\Http\Requests\EmployeeRequest;
use App\Services\EmployeeService;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class EmployeesController extends Controller
{
    /**
     * Method to show employees list page.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index()
    {
        $seller = Auth::user();
        $employees = $seller->employees()
            ->where('status', '<>', User::STATUS_DELETED)
            ->withCount(['redeems', 'referrals', 'invitedSellers'])
            ->with('permissions')
            ->get();

        return view('redesign.back.metronic.employees.index', [
            'employees' => $employees,
        ]);
    }

    /**
     * Method to show employee create form.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function store()
    {
        if ( !Auth::user()->hasSellerProfileFilled() ) {
            return redirect(route('seller.company.index'))
                ->with('warning', 'Bevor Du einen neuen Mitarbeiter erstellen, musst Du das Profil ausfüllen. Mindestanforderungen sind Firmenlogo, Kontaktdaten und Unternehmensdaten');
        }

        return view('redesign.back.metronic.employees.store', [
            'permissions' => User::EMPLOYEE_PERMISSIONS,
        ]);
    }

    public function create(EmployeeRequest $request)
    {
        (new EmployeeService())->create($request->all());

        return redirect()
            ->to(route('seller.employees.index'))
            ->with('notification', $this->_s('Neuen Mitarbeiter wurde erfolgreich eingeladen'));
    }

    /**
     * Method to show employee edit form.
     *
     * @param User $employee
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit(User $employee)
    {
        if ( !Auth::user()->can('updateEmployee', $employee) ) {
            return redirect(route('seller.employees.index'));
        }

        return view('redesign.back.metronic.employees.edit', [
            'employee'    => $employee,
            'permissions' => User::EMPLOYEE_PERMISSIONS,
        ]);
    }

    /**
     * Method to update employee.
     *
     * @param EmployeeRequest $request
     * @param User $employee
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function update(EmployeeRequest $request, User $employee)
    {
        if ( !Auth::user()->can('updateEmployee', $employee) ) {
            return redirect(route('seller.employees.index'));
        }

        (new EmployeeService($employee))->update($request->all());

        return redirect()
            ->to(route('seller.employees.index'))
            ->with('notification', $this->_s('Mitarbeiter wurde erfolgreich bearbeitet'));
    }

    /**
     * Method to activate employee.
     *
     * @param User $employee
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function activate(User $employee)
    {
        if ( !Auth::user()->can('updateEmployee', $employee) ) {
            return redirect(route('seller.employees.index'));
        }

        (new EmployeeService($employee))->activate();

        return redirect()
            ->to(route('seller.employees.index'))
            ->with('notification', $this->_s('Mitarbeiter wurde erfolgreich aktiviert'));
    }

    /**
     * Method to deactivate employee.
     *
     * @param User $employee
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function deactivate(User $employee)
    {
        if ( !Auth::user()->can('updateEmployee', $employee) ) {
            return redirect(route('seller.employees.index'));
        }

        (new EmployeeService($employee))->deactivate();

        return redirect()
            ->to(route('seller.employees.index'))
            ->with('notification', $this->_s('Mitarbeiter wurde erfolgreich deaktiviert'));
    }

    /**
     * Method to delete employee.
     *
     * @param User $employee
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function delete(User $employee)
    {
        if ( !Auth::user()->can('updateEmployee', $employee) ) {
            return redirect(route('seller.employees.index'));
        }

        (new EmployeeService($employee))->delete();

        return redirect()
            ->to(route('seller.employees.index'))
            ->with('notification', $this->_s('Mitarbeiter wurde erfolgreich gelöscht'));
    }

    /**
     * Method to show employee activation form.
     *
     * @param User $employee
     * @param $token
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|\Illuminate\View\View
     */
    public function activation(User $employee, $token)
    {
        if ( User::STATUS_PENDING !== $employee->status || !$employee->boss || $token !== $employee->email_verification_token ) {
            return redirect(route('home'));
        }

        return view('redesign.auth.employee_activation', [
            'employee' => $employee,
            'token'    => $token,
        ]);
    }

    /**
     * Method to register employee.
     *
     * @param EmployeeRegisterRequest $request
     * @param User $employee
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function register(EmployeeRegisterRequest $request, User $employee)
    {
        (new EmployeeService($employee))->register($request->all());

        Auth::login($employee);

        return redirect(route('home'));
    }

    /**
     * Method to show employees activity page.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function activityIndex()
    {
        $seller = Auth::user();

        $employees = $seller->employees()
            ->where('status', '<>', User::STATUS_DELETED)
            ->get();

        $actions = DB::table('user_log')
            ->whereIn('user_log.user_id', $employees->pluck('id'))
            ->join('user_log_actions', 'user_log.action_id', '=', 'user_log_actions.id')
            ->select('user_log_actions.id', 'user_log_actions.action')
            ->distinct()
            ->get();

        $minDateRow = Activity::whereIn('user_id', $employees->pluck('id'))->oldest()->first();
        $minDate = $minDateRow ? $minDateRow->created_at : Carbon::today();

        $maxDateRow = Activity::whereIn('user_id', $employees->pluck('id'))->latest()->first();
        $maxDate = $maxDateRow ? $maxDateRow->created_at : Carbon::today();

        return view('redesign.back.metronic.employees.activity', [
            'actions'   => $actions,
            'minDate'   => $minDate->format('d.m.Y'),
            'maxDate'   => $maxDate->format('d.m.Y'),
            'employees' => $employees,
        ]);
    }
}