<?php

namespace App\Http\Controllers;

use App\Account;
use App\Coupon;
use App\CouponTemplate;
use App\Library\Dashboard\Filter;
use App\Http\Requests\DashboardFilterRequest;
use App\Library\Facades\Format;
use App\Services\SellerService;
use App\Services\StatisticsService;
use App\Transaction;
use App\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use willvincent\Rateable\Rating;

class DashboardController extends Controller
{
    /**
     * @var Filter
     */
    protected $sellerAccountFilter;

    /**
     * @var Filter
     */
    protected $agentAccountFilter;

    /**
     * @var Filter
     */
    protected $consumerAccountFilter;

    /**
     * @var Filter
     */
    protected $couponTemplateFilter;

    public function __construct()
    {
        parent::__construct();
        $this->middleware(['auth', 'completed.profile']);
    }

    /**
     * Method to show seller dashboard (main admin page).
     */
    public function seller()
    {
        $seller = $this->getSeller();

        $deals = $seller->couponTemplates
            ->whereIn('status', [CouponTemplate::STATUS_ACTIVE, CouponTemplate::STATUS_INACTIVE])
            ->groupBy('status');

        $sellerCoupons = $seller->couponsSharedFromThisSeller;

        $reviews       = Rating::where('rated_account_id', $seller->account_id)->latest()->get();
        $reviewsAvg    = round($reviews->avg('rating'), 2);
        $reviewsCount  = $reviews->count();
        $reviewsCounts = [
            5 => $reviews->where('rating', 5)->count(),
            4 => $reviews->where('rating', 4)->count(),
            3 => $reviews->where('rating', 3)->count(),
            2 => $reviews->where('rating', 2)->count(),
            1 => $reviews->where('rating', 1)->count(),
        ];

        $numbers = StatisticsService::calculateTransactionsNumbers($seller);

        $activatedCoupons = StatisticsService::getActivatedCouponsByPeriod($seller, 5, 6);

        return view('redesign.back.metronic.dashboard.index', [
            'deals'            => $deals,
            'numbers'          => $numbers,
            'coupons'          => $sellerCoupons->groupBy('status'),
            'reviews'          => $reviews,
            'reviewsAvg'       => $reviewsAvg,
            'reviewsCount'     => $reviewsCount,
            'reviewsCounts'    => $reviewsCounts,
            'activatedCoupons' => $activatedCoupons,
        ]);
    }

    /**
     * Initializes Filters
     *
     * @param $request
     * @return array
     */
    public function initializeFilters($request)
    {
        $this->sellerAccountFilter = new Filter(Account::class, User::SELLER);
        $this->agentAccountFilter = new Filter(Account::class, User::AGENT);
        $this->consumerAccountFilter = new Filter(Account::class, User::CONSUMER);
        $this->couponTemplateFilter = new Filter(CouponTemplate::class, null);

        if (!Auth::user()->hasRole('operator')) {
            $this->sellerAccountFilter->setAvailable(
                Auth::user()->account->availableSellerAccountsQuery()->get()
            );
            $this->agentAccountFilter->setAvailable(
                Auth::user()->account->relatedAgentAccounts
            );
            $this->consumerAccountFilter->setAvailable(
            // FIXME: nur die relevanten consumer listen
            // $availableConsumerAccounts = Auth::user()->account->getUsersWithCouponsFromSellersOfAgent();
                $availableConsumerAccounts = User::allConsumers()
            );
            $this->couponTemplateFilter->setAvailable(
                Auth::user()->couponTemplates
            );
        }

        $this->sellerAccountFilter->parseRequest($request);
        $this->agentAccountFilter->parseRequest($request);
        $this->consumerAccountFilter->parseRequest($request);
        $this->couponTemplateFilter->parseRequest($request);
    }

    public function getCompactedFilters()
    {
        return [
            'sellerAccountFilter' => $this->sellerAccountFilter,
            'agentAccountFilter' => $this->agentAccountFilter,
            'consumerAccountFilter' => $this->consumerAccountFilter,
            'couponTemplateFilter' => $this->couponTemplateFilter
        ];
    }

    public function operator(DashboardFilterRequest $request)
    {
        $this->initializeFilters($request);

        $openProvision = 0;
        $openDiscount = 0;
        $membershipContributions = 0;
        $couponContributions = 0;
        $openCouponCount = 0;
        $redeemedCouponCount = 0;
        $agentProvisoin = 0;
        $activeAgents = [];
        $topSellers = [];
        $topAgents = [];
        $countSellers = 0;
        $sellerOperatorProvision = 0;
        $cash = 0;
	    $nettoAllCoupons = 0;
	    $nettoMembershipContributions = 0;
        $recoCash = 0;

        $countAgents = count($this->agentAccountFilter->actual);
        foreach ($this->agentAccountFilter->actual as $agent) {
            $pendingProvisions = $this->getProvision($agent->id, 'pending');
            foreach ($pendingProvisions as $pendingProvision) {
                $openProvision = $openProvision + $pendingProvision->amount;
            }
        }

        foreach ($this->consumerAccountFilter->actual as $consumer) {
            $pendingDiscounts = $this->getProvision($consumer->id, 'allocated');
            foreach ($pendingDiscounts as $pendingDiscount) {
                $openDiscount = $openDiscount + $pendingDiscount->amount;
            }
        }

        $countSellers = count($this->sellerAccountFilter->actual);
        foreach ($this->sellerAccountFilter->actual as $seller) {
            $transactions = Transaction::where('user_id', '=', $seller->id)
                ->where('status', '=', 'paid')
                ->get();

            $transactions = Transaction::where('user_id', '=', $seller->id)
                ->where('type', '=', 'seller-operator-provision-payment')
                ->where('status', '=', 'paid')
                ->get();
            foreach ($transactions as $transaction) {
                $couponContributions += $transaction->amount;
            }
        }
        $couponContributions *= -1;

        $transactions = Transaction::where('type', '=', 'seller-fee-monthly-payment')
            ->orWhere('type', '=', 'seller-fee-yearly-payment')
            ->get();
        foreach ($transactions as $transaction) {
            $nettoMembershipContributions += $transaction->amount;
        }
        $nettoMembershipContributions *= -1;

        $transactions = Transaction::whereIn('type', [
            'agent-coupon-provision-credit',
            'consumer-coupon-discount-credit',
            'seller-fee-monthly-payment',
            'seller-fee-yearly-payment'
        ])
            ->get();
        foreach ($transactions as $transaction) {
            $cash += $transaction->amount;
        }

        $transactions = Transaction::where('type', '=', 'seller-operator-provision-payment')
            ->get();
        foreach ($transactions as $transaction) {
            $sellerOperatorProvision += $transaction->amount;
        }
        $sellerOperatorProvision *= -1;

        // selecting a seller means to choose all their coupon templates
        $couponTemplatesFromSellers = CouponTemplate::whereIn('user_id', $this->sellerAccountFilter->actual->pluck('id'))->get();
        $matchingCouponTemplates = $this->couponTemplateFilter->actual->intersect($couponTemplatesFromSellers);

        // FIXME: evtl. AND logik für den fall das tatsächlich filter definiert wurden
        $allCoupons = Coupon::query(); /* Coupon::whereIn('consumer_user_id', $this->consumerAccountFilter->actual->pluck('id'))
            ->orWhereIn('agent_user_id', $this->agentAccountFilter->actual->pluck('id'))
            ->orWhereIn('coupon_template_id', $matchingCouponTemplates->pluck('id')); */

	    $nettoAllCoupons += $allCoupons->get()->sum('netto');
	    $openCoupons = clone $allCoupons;
        $openCouponCount = $openCoupons->whereStatus(Coupon::STATUS_ACTIVE)->count();

        $redeemedCoupons = clone $allCoupons;
        $redeemedCouponCount = $redeemedCoupons->whereStatus(Coupon::STATUS_REDEEMED)->count();

        $all = 0;
        $done = 0;
        $couponsMediatedFromSeller = 0;


        $transactions = Transaction::whereIn('type', [
            'consumer-coupon-discount-credit',
        ])
            ->get();
        foreach ($transactions as $transaction) {
            $recoCash += $transaction->amount;
        }

        $transactions = Transaction::whereIn('type', [
            'agent-coupon-provision-credit',
        ])
            ->get();
        foreach ($transactions as $transaction) {
            $agentProvisoin += $transaction->amount;
        }

        /**
         * @var $seller User
         */
        foreach ($this->sellerAccountFilter->actual as $seller) {
            $couponTemplatesFromSeller = $seller->couponTemplates;

            $couponsFromSeller = 0;
            $couponsDoneFromSeller = 0;
            $couponsMediatedFromSeller = 0;
            foreach ($couponTemplatesFromSeller as $couponTemplateFromSeller) {
                $couponsFromSeller += $couponTemplateFromSeller->coupons->count();
                $couponsDoneFromSeller += $couponTemplateFromSeller->coupons()->whereIn('status', Coupon::STATUS_GROUP_FINISHED)->count();
            }

            $all += $couponsFromSeller;
            $done += $couponsDoneFromSeller;

            if ($couponsFromSeller != 0 && $couponsDoneFromSeller != 0) {
                $conversionRate = 100 / $couponsFromSeller * $couponsDoneFromSeller;
                $seller->conversionrate = round($conversionRate, 2);

                array_push($topSellers, $seller);
            }
        }

        if ($all != 0 && $done != 0) {
            $couponsMediatedFromSeller = 100 / $all * $done;
        }
        uasort($topSellers, function ($a, $b) {
            return -($a->conversionrate - $b->conversionrate);
        });

        /**
         * @var User $agent
         */
        foreach ($this->agentAccountFilter->actual as $agent) {
            $couponsFromAgent = $agent->couponsByAgent()->count();

            $couponsDoneFromAgent = $agent->couponsByAgent()->whereIn('status', Coupon::STATUS_GROUP_FINISHED)->count();

            if ($couponsFromAgent != 0 || $couponsDoneFromAgent != 0) {
                $conversionRate = 100 / $couponsFromAgent * $couponsDoneFromAgent;
                $agent->conversionrate = round($conversionRate, 2);

                array_push($topAgents, $agent);
            }
        }
        uasort($topAgents, function ($a, $b) {
            return -($a->conversionrate - $b->conversionrate);
        });

        $coupons = Coupon::all();
        $couponsFinishedCount = $coupons->whereIn('status', Coupon::STATUS_GROUP_FINISHED)->count();
        if (Coupon::countMultiBonOnlyOnce(Coupon::query()) != 0 && $couponsFinishedCount != 0) {
            $averageConversionRate = 100 / Coupon::countMultiBonOnlyOnce(Coupon::query()) * $couponsFinishedCount;
        } else {
            $averageConversionRate = 0;
        }


//dd($countSellers);
        $numbers = [
            [
                'name' => 'Anzahl Unternehmen',
                'number' => $countSellers,
                'link' => route('user'),
                'icon' => 'group-1'
            ],
            [
                'name' => 'Anzahl Empfehler',
                'number' => $countAgents,
                'link' => route('user'),
                'icon' => 'user'
            ],
            [
                'name' => 'Anzahl Verbraucher',
                'number' => $this->consumerAccountFilter->actual->count(),
                'link' => route('user'),
                'icon' => 'people'
            ],
            [
                'name' => 'Netto Umsätze',
                'number' => Format::currency($nettoAllCoupons),
                'link' => route('transaction'),
                'icon' => 'people'
            ],
            [
                'name' => 'Mitgliedsbeiträge',
                'number' => Format::currency($nettoMembershipContributions),
                'link' => route('transaction'),
                'icon' => 'dollar-symbol-1'
            ],
            [
                'name' => 'Portal Gebühren',
                'number' => Format::currency($sellerOperatorProvision),
                'link' => route('transaction'),
                'icon' => 'dollar-symbol'
            ],
            [
                'name' => 'Summe Cashback',
                'number' => Format::currency($recoCash),
                'link' => route('transaction'),
                'icon' => 'money'
            ],
            [
                'name' => 'Summe RECO.CASH',
                'number' => Format::currency($agentProvisoin),
                'link' => route('transaction'),
                'icon' => 'money-1'
            ],
            [
                'name' => 'Ø Erfolgsrate',
                'number' => Format::percent($averageConversionRate),
                'link' => route('transaction'),
                'icon' => 'startup'
            ],
            [
                'name' => 'Zahlungsanfragen',
                'number' => Coupon::whereStatus(Coupon::STATUS_PAYMENT_ASSERTED)->count(),
                'link' => route('transaction'),
                'icon' => 'settings'
            ],
            [
                'name' => 'Empfohlene BONS',
                'number' => Coupon::countMultiBonOnlyOnce(Coupon::query()),
                'link' => route('transaction'),
                'icon' => 'settings'
            ],
            [
                'name' => 'Eingelöste BONS',
                'number' => Coupon::countMultiBonOnlyOnce(Coupon::whereIn('status', Coupon::STATUS_GROUP_FINISHED)),
                'link' => route('transaction'),
                'icon' => 'handshake'
            ],
        ];

        return view('dashboard.operator',
            compact(
                'numbers'
            ))
            ->with('activeAgentCount', count($activeAgents))
            ->with('topSellers', array_splice($topSellers, 0, 5))
            ->with('topAgents', array_splice($topAgents, 0, 5))
            ->with($this->getCompactedFilters())
            ->with($this->getTransactionStatus());
    }

    public function agent(DashboardFilterRequest $request)
    {
        $guidanceShown = $this->user->guidanceShown();
        $this->initializeFilters($request);

        $paidOutProvision = 0;
        $openProvision = 0;
        $couponsFinishedCount = 0;
        $averageConversionRate = 0;
        $averageProvision = 0;
        $sellersForThisAgent = [];
        $topSellers = [];
        $provision = 0;
        $templatesWithConversionRate = [];
        $result = 0;
        $agent = Auth::user();

        // If CouponTemplates are explicitly filtered, then we use those
        if ($this->couponTemplateFilter->selected) {
            $couponTemplates = $this->couponTemplateFilter->selected;
        } else { // otherwise we use all of the selected seller accounts
            $couponTemplates = [];
            foreach ($this->sellerAccountFilter->actual as $selectedSellerAccount) {
                $selectedCouponTemplates = $selectedSellerAccount->couponTemplates()->get()->keyBy('id');
                $couponTemplates = array_merge($couponTemplates, $selectedCouponTemplates->toArray());
            }
        }

        foreach ($this->sellerAccountFilter->actual as $account) {
            $templates = DB::table('coupon_templates')
                ->whereIn('user_id', $account->users()->pluck('id'))
                ->where('status', '=', 'active')
                ->get();
            if ($templates != null) {
                foreach ($templates as $template) {
                    array_push($couponTemplates, $template);
                }
            }
        }

        $disbursedProvisions = $this->getProvision(Auth::id(), 'paid');
        foreach ($disbursedProvisions as $disbursedProvision) {
            $paidOutProvision = $paidOutProvision + $disbursedProvision->amount;
        }

        $pendingProvisions = $this->getProvision(Auth::id(), 'pending');
        foreach ($pendingProvisions as $pendingProvision) {
            $openProvision = $openProvision + $pendingProvision->amount;
        }

        $couponsFinished = Coupon::whereAgentUserId(Auth::id())
            ->where('selection', '', 0)
            ->whereIn('status', Coupon::STATUS_GROUP_FINISHED)->get();
        $couponsFinishedCount = count($couponsFinished);
        if (Coupon::countMultiBonOnlyOnce(Auth::user()->couponsByAgent()) != 0 && $couponsFinishedCount != 0) {
            $averageConversionRate = round(100 / Coupon::countMultiBonOnlyOnce(Auth::user()->couponsByAgent())* $couponsFinishedCount, 2);
        } else {
            $averageConversionRate = 0;
        }

        foreach($couponsFinished as $finished){
            $result += $finished->calculateAgentProvision();
        }

        if (count($couponsFinished) > 0) {
	        $averageProvision = $result / Count( $couponsFinished );
        } else {
        	$averageProvision = null;
        }

        $sellersForThisAgent = Auth::user()->account->availableSellerAccountsQuery()->get();
        $couponCount = 0;
        foreach ($sellersForThisAgent as $seller) {
            $templates = $seller->couponTemplates;
            $provision = 0;

            foreach ($templates as $template) {
                $couponsFinished =
	                Coupon::where('coupon_template_id', '=', $template->id)
                    ->where('agent_user_id', '=', Auth::id())
                    ->where('selection', '=', 0)
                    ->get();

                foreach ($couponsFinished as $coupon) {
                    if(in_array($coupon->status, Coupon::STATUS_GROUP_FINISHED)){
                        $provision = $provision + $coupon->calculateAgentProvision();
                    }
                }

	            $allCouponsOfThisTemplateAndAgentCount = $template->coupons()->whereSelection(0)->whereAgentUserId($agent->id)->count();
                $finishedCouponCount = $template->coupons()->whereAgentUserId($agent->id)->whereIn('status', Coupon::STATUS_GROUP_FINISHED)->count();

	            $couponCount += $allCouponsOfThisTemplateAndAgentCount;

                if($allCouponsOfThisTemplateAndAgentCount > 0) {
	                $template->conversionRate = $finishedCouponCount / $allCouponsOfThisTemplateAndAgentCount * 100;
	                array_push($templatesWithConversionRate, $template);
                }
            }
            $seller->provision = $provision;

            if($provision > 0) {
                array_push($topSellers, $seller);
            }
        }

        $sortedSellers = $topSellers;
        uasort($sortedSellers, function ($a, $b) {
            return -($a->provision - $b->provision);
        });
        uasort($templatesWithConversionRate, function ($a, $b) {
            return -($a->conversionRate - $b->conversionRate);
        });

        $agentNumbers = $this->calculateAgentNumbers(true);

        $numbers = [
            [
                'name' => 'Erfolgsrate',
                'number' => Format::percent($averageConversionRate),
                'link' => route('coupon-redeem'),
                'icon' => 'startup'
            ],
            [
                'name' => 'Ø RECO.CASH',
                'number' => Format::currency($averageProvision),
                'link' => route('transaction'),
                'icon' => 'money'
            ],
        ];

        $numbers = array_merge($numbers, $agentNumbers);


        $filters = $this->getCompactedFilters();
        unset($filters['agentAccountFilter']);

        return view('dashboard.agent', [
            'numbers' => $numbers,
            'couponTemplates' => $couponTemplates,
            'paidOutProvision' => $paidOutProvision,
            'openProvision' => $openProvision,
            'topFiveSellers' => array_splice($sortedSellers, -5),
            'templatesWithConversionRate' => $templatesWithConversionRate,
            'guidance' => $guidanceShown,
        ])
            ->with($filters)
            ->with($this->getTransactionStatus());
    }

    public function getProvision($id, $val)
    {
        return Transaction::where('user_id', '=', $id)
            ->where('status', '=', $val)
            ->get();
    }

    public function consumer(DashboardFilterRequest $request)
    {
        $guidanceShown = $this->user->guidanceShown();
        $coupons = Coupon::where('consumer_user_id', '=', Auth::id())
            ->whereStatus(Coupon::STATUS_ACTIVE)
            ->where(function($query) {
                $query->where('validity_date', '>=', Carbon::today());
                $query->orWhereNull('validity_date');
            })
            ->get();

        $couponsFinished = Coupon::where('consumer_user_id', '=', Auth::id())
            ->whereIn('status', Coupon::STATUS_GROUP_FINISHED)
            ->get();

        $selectionCouponGroups = Coupon::where('consumer_user_id', '=', Auth::id())
            ->whereStatus(Coupon::STATUS_PENDING)
            ->whereSelection(1)
            ->groupBy('selection_group')
            ->get();

        $this->user->last_visited_consumer_dashboard = Carbon::now();
        $this->user->save();

        $this->refreshNavigation();

        return view('coupon.consumer', [
            'coupons' => $coupons,
            'selectionCouponGroups' => $selectionCouponGroups,
            'couponsFinished' => $couponsFinished,
            'guidance' => $guidanceShown,
        ])
            ->with($this->getTransactionStatus());
    }

    public function loadUsers($role)
    {
        return DB::table('users')
            ->join('user_has_roles', 'users.id', '=', 'user_has_roles.user_id')
            ->join('roles', 'roles.id', '=', 'user_has_roles.role_id')
            ->where('roles.name', '=', $role)
            ->get();
    }

    /**
     * Method to show user dashboard.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function user()
    {
        $user    = Auth::user();
        $coupons = $user->coupons->count();
        $sellers = (new SellerService())->getAttachedSellers($user);

        return view('redesign.front.user.dashboard', [
            'user'    => $user,
            'sellers' => $sellers,
            'coupons' => $coupons,
        ]);
    }
}
