<?php

namespace App\Http\Controllers;

use App\Account;
use App\Place;
use App\Http\Grids\Agent\CouponTemplateGrid as AgentCouponTemplateGrid;
use App\Http\Grids\Seller\CouponTemplateGrid as SellerCouponTemplateGrid;
use App\LinkInvitations;
use App\Services\DealService;
use App\UserConnection;
use Carbon\Carbon;
use Illuminate\Http\Request;
use App\Http\Requests\CouponTemplateRequest;
use App\CouponTemplate;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Redirect;
use File;
use Spatie\MediaLibrary\Media;
use View;
use App\Presenters\CouponTemplatePresenter;
use App\Category;
use App\Gift;

class CouponTemplateController extends Controller
{
    public function __construct()
    {
        parent::__construct();
    }

    /*
    |--------------------------------------------------------------------------
    | Coupon Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the creation of new coupons.
    |
    */

    public function edit(CouponTemplate $couponTemplate)
    {
        return view('coupon/edit', [
            'couponTemplate' => $couponTemplate,
            'typeIcons'      => CouponTemplatePresenter::$typeIcons,
            'typeNames'      => CouponTemplatePresenter::$typeNames,
            'categories'     => Category::all(),
            'gifts'          => Gift::available(),
            'images'         => $couponTemplate->getMedia('images'),
            'videos'         => $couponTemplate->getMedia('videos'),
        ]);
    }

    /**
     * Create a new coupon template instance after a valid request.
     *
     * @param  CouponTemplateRequest $request
     * @param CouponTemplate $couponTemplate
     * @return CouponTemplate
     * @throws \Spatie\MediaLibrary\Exceptions\FileCannotBeAdded
     */
    protected function store(CouponTemplateRequest $request, CouponTemplate $couponTemplate)
    {
        // TODO refactor that piece of garbage

        if (CouponTemplate::TYPE_GIFT === $request->get('type_discount')) {
            $request['discount_consumer'] = 0;
        } else {
            unset($request['discount_gift_id']);
        }

        if (CouponTemplate::TYPE_GIFT === $request->get('type_provision')) {
            $request['provision_agent'] = 0;
        } else {
            unset($request['provision_gift_id']);
        }

        if (CouponTemplate::TYPE_NONE === $request->get('type_provision')) {
            $request['provision_agent'] = 0;
        }

        $oldData = $couponTemplate->getOriginal();
        if(isset( $oldData['expiration_date'])){
            $oldData['expiration_date'] = Carbon::parse($oldData['expiration_date'])->format('Y-m-d');
        }
        $sellerAccount = Auth::user()->account;
        $data = $request->except(['picture', 'images', 'images_delete', 'video', 'video_delete']);
        $data['expiration_date'] = Carbon::parse($data['expiration_date'])->format('Y-m-d');

        // TODO remove that shit
        if (isset($data['fromOverlay'])) {
            array_pop($data);//delete fromOverlay from array to avoid whoops
            try {
                CouponTemplate::create(array_merge(
                    $data,
                    array(
                        'user_id' => Auth::id(),
                        'status' => 'inactive',
                    )
                ));
            } catch (\Exception $e) {
                return redirect()->route('onboarding-seller')->with('notification', $this->notificationError($e->getMessage()));
            }
            $sellerAccount->onboarding_process_level ++;
            $sellerAccount->save();
            return redirect()->route('onboarding-seller')->with('notification', $this->notificationSuccess('RECO.BON erfolgreich erstellt.'));
        }

        if ($couponTemplate->id > 0) {
            $couponTemplate->update($data);
        } else {
            $couponTemplate = CouponTemplate::create(array_merge(
                $data,
                array(
                    'user_id' => Auth::id(),
                    'status' => 'active',
                )
            ));
        }

        if ($imagesToDelete = $request->get('images_delete')) {
            Media::destroy($imagesToDelete);
        }

        foreach ($request->file('images') as $image) {
            if (null === $image) {
                continue;
            }
            $couponTemplate->addMedia($image)
                ->preservingOriginal()
                ->toMediaLibrary('images');
        }

        if ($request->get('video_delete')) {
            $couponTemplate->clearMediaCollection('videos');
        }

        if ($video = $request->file('video')) {
            $couponTemplate->clearMediaCollection('videos');

            $couponTemplate->addMedia($video)
                ->preservingOriginal()
                ->toMediaLibrary('videos');
        }

        return Redirect::to('coupon')->with('notification', array(
            'type' => 'success',
            'title' => 'Datensatz gespeichert',
            'description' => 'Coupon Template erfolgreich gespeichert.'
        ));
    }

    protected function saveCouponImage(CouponTemplate $couponTemplate, Request $request)
    {
        if ($request->file('picture')) {
            $fileName = preg_replace('/[^[:alnum:].]/', '', $request
                ->file('picture')
                ->getClientOriginalName());

            if ($couponTemplate->img) {
                File::delete(storage_path('app/coupon_logos/') . $couponTemplate->img);
            }

            $request->file('picture')->move(storage_path('app/coupon_logos'), $fileName);
            $logoImage = \Image::make(storage_path('app/coupon_logos/') . $fileName);

            $logoImage->save();

            return $fileName;
        }

        return $couponTemplate->img;
    }

    public function getImage($coupon_logo)
    {
        $path = storage_path('app/coupon_logos/') . $coupon_logo;

        return response()->download($path, null, [], 'inline');
    }

    public function index()
    {
        if (Auth::user()->hasRole('agent')) {
            $couponTemplateGrid = new AgentCouponTemplateGrid();

            return view('coupon/index', [
                'globalFilters' => $couponTemplateGrid->getGlobalFilters(),
                'grid' => $couponTemplateGrid,
            ]);
        }

        if (Auth::user()->hasRole('seller')) {
            $sellerCouponTemplateGrid = new SellerCouponTemplateGrid();
            return view('coupon/index', [
                'globalFilters' => $sellerCouponTemplateGrid->getGlobalFilters(),
                'grid' => $sellerCouponTemplateGrid,
            ]);
        }
    }

    public function listBySeller()
    {
        if (Auth::user()->hasRole('agent')) {
            return view('coupon/previews')->with('templates', Auth::user()->availableCouponsQuery(false, false)->get());
        }
    }

    protected function getPersonalEmailTextTemplate($couponTemplateOrAccount)
    {
        //Der Übergabeparameter ist vom Typ CouponTemplate oder Account!!!
        if($couponTemplateOrAccount instanceof Account){
            return View::make('emails/agent-sends-coupon-personal-text')->with('account', $couponTemplateOrAccount)->render();
        }else{
            $account = $couponTemplateOrAccount->user->account;
            return View::make('emails/agent-sends-coupon-personal-text')->with(['account'=> $account,'couponTemplate'=>$couponTemplateOrAccount])->render();
        }
    }

    protected function getPersonalSMSTextTemplate($seller)
    {
        return View::make('sms/agent-sends-coupon-personal-text')->with('seller', $seller)->render();
    }

    //TODO send und shareCouponsFromSeller in einer Function konsolidieren
    public function send(CouponTemplate $couponTemplate, Request $request, $type = null)
    {

        if ($type == 'sms') {
            $personalTextTemplate = $this->getPersonalSMSTextTemplate($couponTemplate->user->account);
            $action = route('linkinvitation', ['reason' => 'share-coupon', 'id' => $couponTemplate->id, 'type' => $type]);
        } else {
            $personalTextTemplate = $this->getPersonalEmailTextTemplate($couponTemplate->user->account);
            $action = route('share-coupons-template-post');
        }
        return view('share-coupons', [
            'recommend' => $couponTemplate,
            'personalTextTemplate' => $personalTextTemplate,
            'action' => $action,
            'type' => $type
        ]);
    }

    public function shareViaDesktop($token,$type)
    {
        $linkInvitationByToken = LinkInvitations::where('token',$token)->first();
        $reason = $linkInvitationByToken->reason;
        $headline = '';
        if($type == 'email'){
            $action = 'send-mail-invitation-to-user';
        }else{
            $action = 'send-sms-invitation-to-user';
        }

        //FIXME $action muss auf neue route/function zeigen, mail/sms versand
        if($reason == 'share-coupon'){
            $recommend = CouponTemplate::where('id',$linkInvitationByToken->coupon_template_id)->first();
        } elseif($reason == 'share-seller'){
            $recommend = Account::where('id',$linkInvitationByToken->shared_account_id)->first();
        }

        if($reason == 'recommend-recoma'){
            $recommend = null;
            $personalTextTemplate = 'Text';
            $view = 'recommendation.index';
            $headline = 'RECO.MA empfehlen';
        }elseif ($reason == 'invite-new-agent'){
            $recommend = null;
            $personalTextTemplate = 'Text';
            $view = 'recommendation.index';
            $headline = 'Empfehler einladen';
        }

        elseif($type == 'email' && $reason != 'recommend-recoma'){
            $personalTextTemplate = $this->getPersonalEmailTextTemplate($recommend);
            if(Auth::user()->account->onboarding_process_level == 2){
                $view = 'onboarding.agent.recommend-seller-form';
            }else{
                $view = 'share-coupons';
            }
        } else {
            $personalTextTemplate = $this->getPersonalSMSTextTemplate($recommend);
            if(Auth::user()->account->onboarding_process_level == 2) {
                $view = 'onboarding.agent.recommend-seller-form';
            }else{
                $view = 'share-coupons';

            }
        }
            return view($view, [
                'recommend' => $recommend,
                'personalTextTemplate' => $personalTextTemplate,
                'action' => $action,
                'type' => $type,
                'tokenId' => $linkInvitationByToken->id,
                'headline' => $headline
            ]);
    }

    public function getCashback(CouponTemplate $couponTemplate)
    {
        $user = Auth::user();
        $this->storeCoupon($couponTemplate->id, $user->id, $user);

        return Redirect::to('coupon')->with('notification', array(
            'type' => 'success',
            'title' => 'Congratulations!',
            'description' => 'You will get cashback after redeeming coupon'
        ));
    }

    /**
     * Method to show deal page.
     *
     * @param CouponTemplate $couponTemplate
     * @param string $alias
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function deal(string $alias, CouponTemplate $couponTemplate)
    {
        if ( !$couponTemplate->isActive() || !$couponTemplate->isShareable() ) {
            return redirect(route('home'));
        }

        visits($couponTemplate)->register();

        $places        = $couponTemplate->places->where('status', Place::STATUS_ACTIVE);
        $seller        = $couponTemplate->seller;
        $reviews       = $couponTemplate->ratings()->with('user')->latest()->get();
        $otherDeals    = $seller->getOtherDeals($couponTemplate, 999);
        $reviewsAvg    = $reviews->avg('rating');
        $reviewsCount  = $reviews->count();

        return view('redesign.front.deal.index', [
            'deal'         => $couponTemplate,
            'places'       => $places,
            'reviews'       => $reviews,
            'otherDeals'   => $otherDeals,
            'reviewsAvg'   => $reviewsAvg,
            'reviewsCount' => $reviewsCount,
        ]);
    }

    /**
     * Method to show deal page.
     *
     * @param CouponTemplate $couponTemplate
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function dealOld(CouponTemplate $couponTemplate)
    {
        return $this->deal('', $couponTemplate);
    }

    /**
     * Method to set deal as favorite.
     *
     * @param CouponTemplate $couponTemplate
     * @param bool $redirect
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function favorite(CouponTemplate $couponTemplate, $redirect = true)
    {
        Auth::user()->favorites()->attach($couponTemplate);

        if ($redirect) {
            return redirect(route('user.fav'));
        }
    }

    /**
     * Method to remove deal from favorites.
     *
     * @param CouponTemplate $couponTemplate
     * @param bool $redirect
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function unfavorite(CouponTemplate $couponTemplate, $redirect = true)
    {
        Auth::user()->favorites()->detach($couponTemplate);

        if ($redirect) {
            return redirect(route('user.fav'));
        }
    }

    /**
     * Method to add deal to favorites or remove if it already in.
     *
     * @param CouponTemplate $couponTemplate
     */
    public function favoriteToggle(CouponTemplate $couponTemplate)
    {
        $user = Auth::user();
        $hasInFav = $user->favorites()->where('deal_id', $couponTemplate->id)->exists();

        if ( !$hasInFav ) {
            $this->favorite($couponTemplate, false);
        } else {
            $this->unfavorite($couponTemplate, false);
        }
    }

    /**
     * Method to show deal creation page.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create()
    {
        $seller = $this->getSeller();

        if ( !$seller->hasSellerProfileFilled() ) {
            return redirect(route('seller.company.index'))
                ->with('warning', 'Bevor Du eine neue Kampagne erstellen, musst Du das Profil ausfüllen. Mindestanforderungen sind Firmenlogo, Kontaktdaten und Unternehmensdaten');
        }

        $places = $seller->places;

        if ( !$seller->places->count() ) {
            return redirect(route('seller.place.store'))
                ->with('warning', 'Bevor Du eine neue Kampagne erstellen, musst Du mindestens ein Standort erstellen.');
        }

        $draftDeal = CouponTemplate::create([
            'user_id' => $seller->id,
            'status'  => CouponTemplate::STATUS_DRAFT,
        ]);

        return view('redesign.back.metronic.deals.constructor', [
            'deal'       => $draftDeal,
            'title'      => 'Kampagne erstellen',
            'gifts'      => Gift::available(),
            'places'     => $places,
            'hasPlaces'  => collect([]),
            'gallery'    => collect([]),
            'videos'     => collect([]),
            'categories' => Category::getTree(false, false, true),
        ]);
    }

    /**
     * Method to show deal edit page.
     *
     * @param CouponTemplate $couponTemplate
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function update(CouponTemplate $couponTemplate)
    {
        $seller = $this->getSeller();

        if ( !$seller->can('update', $couponTemplate) ) {
            return redirect(route('seller.deals.index'));
        }

        $places = $seller->places;

        if ( !$seller->places->count() ) {
            return redirect(route('seller.place.store'))
                ->with('warning', 'Bevor Sie eine Kampagne bearbeiten, müssen Sie mindestens ein Standort erstellen.');
        }

        return view('redesign.back.metronic.deals.constructor', [
            'deal'       => $couponTemplate,
            'title'      => 'Kampagne bearbeiten',
            'gifts'      => Gift::available(),
            'places'     => $places,
            'hasPlaces'  => $couponTemplate->places,
            'gallery'    => $couponTemplate->gallery,
            'videos'     => $couponTemplate->video,
            'categories' => Category::getTree(false, false, true),
        ]);
    }

    /**
     * Method to activate deal.
     *
     * @param CouponTemplate $couponTemplate
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function activate(CouponTemplate $couponTemplate)
    {
        if ( !Auth::user()->can('update', $couponTemplate) ) {
            return redirect(route('seller.deals.index'));
        }

        (new DealService($couponTemplate))->activate();

        return redirect(route('seller.deals.index'));
    }

    /**
     * Method to deactivate deal.
     *
     * @param CouponTemplate $couponTemplate
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function deactivate(CouponTemplate $couponTemplate)
    {
        if ( !Auth::user()->can('update', $couponTemplate) ) {
            return redirect(route('seller.deals.index'));
        }

        (new DealService($couponTemplate))->deactivate();

        return redirect(route('seller.deals.index'));
    }

    /**
     * Method to delete deal.
     *
     * @param CouponTemplate $couponTemplate
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function delete(CouponTemplate $couponTemplate)
    {
        if ( !Auth::user()->can('update', $couponTemplate) ) {
            return redirect(route('seller.deals.index'));
        }

        (new DealService($couponTemplate))->delete();

        return redirect()
            ->to(route('seller.deals.index'))
            ->with('notification', $this->_s('Die Kampagne wurde erfolgreich entfernt'));
    }
}