<?php

namespace App\Http\Controllers;

use App\Account;
use App\Coupon;
use App\CouponTemplate;
use App\Http\Grids\Agent\AgentMultiCouponGrid;
use App\Http\Grids\Consumer\CouponViewGrid;
use App\Http\Grids\Seller\CouponGrid;
use App\Http\Grids\Agent\CouponGrid as AgentCouponGrid;
use App\Http\Grids\Seller\MultiCouponGrid;
use App\Http\Requests\CouponRedeemRequest;
use App\Http\Requests\CouponSendRequest;
use App\Library\Facades\Format;
use App\Shortlink;
use App\User;
use Carbon\Carbon;
use Illuminate\Auth\Passwords\DatabaseTokenRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Redirect;
use App\Library\Grids\DefaultGrid;
use SimpleSoftwareIO\SMS\Facades\SMS;
use Input;
use App\Http\Requests;
use App\Events\CouponPaid;

class CouponController extends Controller
{

    /**
     * The password token repository.
     *
     * @var \Illuminate\Auth\Passwords\TokenRepositoryInterface
     */
    protected $tokens;

    public function __construct()
    {
        parent::__construct();
        $this->middleware(['auth', 'completed.profile']);
    }

    public function redeem()
    {
        // FIXME: Non-Multi-Role
        if (Auth::user()->hasRole('seller')) {
            return view('coupon/redeem', [
                'grid' => new CouponGrid(),
                'multiGrid' => new MultiCouponGrid()
            ]);
        }

        $grid = new AgentCouponGrid();
        $multiGrid = new AgentMultiCouponGrid();

        $globalFilters = $grid->getGlobalFilters();


        return view('coupon/redeem', [
            'grid' => $grid,
            'multiGrid' => $multiGrid,
            'globalFilters' => $globalFilters,
            'numbers' => $this->calculateAgentNumbers(),
        ]);
    }

    /**
     * @param CouponSendRequest $request
     * @return Redirect
     */
    public function store(CouponSendRequest $request)
    {
        $data = $request->all();

        if (isset($data['mobile'])) {
            $user = User::where('mobile', '=', $data['mobile'])->first();
        } else {
            $data['mobile'] = null;
        }
        if (isset($data['email'])) {
            $user = User::where('email', '=', $data['email'])->first();
        }

        $user = $this->checkIfUserExists($data, $user);
        $this->shareCouponWithUser($user, Auth::user());

        if (!isset($data['emailtext'])) {
            $data['emailtext'] = 'sms was sent';
        }
        if (!isset($data['smstext'])) {
            $data['smstext'] = 'email was sent';
        }

        $coupon = $this->storeCoupon($data['recommend_id'], $user, $data['emailtext'], $data['smstext']);

        $this->sendNewCouponNotification($coupon, $data['mobile']);

        return redirect('coupon/redeem');
    }

    public function createSelectionCoupons(CouponSendRequest $request)
    {
        $coupons = $this->selectionCoupons($request->recommend_id, Auth::user()->id, null, $request);

        $this->sendNewCouponNotification($coupons, $request->mobile);

        return redirect('/coupon/redeem');
    }

    /**
     * Send new coupon notification
     *
     * @param $coupons array (selectioncoup) or single coupon
     */
    public function sendNewCouponNotification($coupons, $toNumber = null)
    {
        /**
         * @var $coupon Coupon
         */
        if (is_array($coupons)) {
            $coupon = array_shift($coupons);
            $isSelectionCoupon = true;
        } else {
            $coupon = $coupons;
            $isSelectionCoupon = false;
        }

        $consumer = $coupon->consumer;
        $toUser = $consumer;
        $agent = $coupon->agent;
        $emailText = $coupon->emailtext;
        $smsText = $coupon->smstext;
        $seller = $coupon->template->seller;
        $sellerAccount = $seller->account;

        $isNewUser = $toUser->password == '';
        if ($isNewUser) {
            $token = $this->getTokenService()->create($coupon->consumer);
            $link = url('password/set/consumer', $token) . '?email=' . urlencode($coupon->consumer->email);
        } else {
            $link = url('/');
        }

        // per SMS versenden wenn Nummern vorliegen und verifiziert, ansonsten nur per E-Mail
        if ($toNumber && $coupon->agent->mobile && $coupon->agent->mobile_verified) {
            $url = Shortlink::shorten($link);

            $fromNumber = trim($agent->mobile);
            $toNumber = trim($toNumber);

            SMS::queue('sms.agent-sends-coupon-to-consumer',
                compact('url', 'smsText'),
                function ($message) use ($toUser, $agent, $fromNumber, $toNumber) {
                    $message->from($fromNumber);
                    $message->to($toNumber);
                });
        }

        $url = $link;
        $template = $isNewUser ? 'agent-sends-coupon-to-new-consumer' : 'agent-sends-coupon-to-consumer';
        $toUser->sendMail($template, compact('consumer', 'agent', 'seller', 'sellerAccount', 'url', 'emailText'));
    }

    /**
     * @return DatabaseTokenRepository
     */
    public function getTokenService()
    {
        $tokens = App::make('password.resets.tokens');
        return $tokens;
    }

    public function redeemed(CouponRedeemRequest $request)
    {
        // trim all input
        Input::merge(array_map(function ($value) {
            if (is_string($value)) {
                return trim($value);
            } else {
                return $value;
            }
        }, Input::all()));

        $data = $request->all();
        $replyToUser = Auth::user();

        $data['net_amount'] = str_replace(',', '.', $data['net_amount']);

        /**
         * @var $coupon Coupon
         */
        try {

            $coupon = Coupon::where('code', '=', str_replace(' ', '', $data['coupon_code']))->firstOrFail();
        } catch (\Exception $e) {
            return Redirect::to('/coupon/redeem')->with('notification', array(
                'type' => 'error',
                'title' => 'Fehler',
                'description' => 'Der RECO.BON existiert nicht.'
            ));
        }
        if(!$coupon->agent){
            return Redirect::to('/coupon/redeem')->with('notification', array(
                'type' => 'error',
                'title' => 'Fehler',
                'description' => 'Empfehler ist zur Zeit nicht aktiv. Transaktion nicht möglich'
            ));
        }
        if(!$coupon->consumer){
            return Redirect::to('/coupon/redeem')->with('notification', array(
                'type' => 'error',
                'title' => 'Fehler',
                'description' => 'Verbraucher ist zur Zeit nicht aktiv. Transaktion nicht möglich'
            ));
        }

        if ($coupon['status'] != Coupon::STATUS_DONE) {
            $coupon->redeem($data['net_amount']);
            $coupon->consumer->sendMail('remind-consumer-when-seller-redeems-coupon', compact('coupon'));
            $coupon->agent->sendMail('remind-agent-when-seller-redeems-coupon', compact('coupon'));


            if($data['selectedSubmitButton']=='pay-cash'){
                $this->paid($coupon);
            }

            return Redirect::to('/coupon/redeem')->with('notification', array(
                'type' => 'success',
                'title' => 'RECO.BON eingelöst',
                'description' => 'Der RECO.BON wurde angenommen.'
            ));
        }
        return Redirect::to('/coupon/redeem')->with('notification', array(
            'type' => 'error',
            'title' => 'RECO.BON bereits eingelöst',
            'description' => 'Der RECO.BON wurde bereits angenommen.'
        ));
    }

    public function accept(Coupon $coupon)
    {
        $consumer = $coupon->consumer;
        $seller = $coupon->template->seller;
        $url = url('/');
        $replyToUser = Auth::user();

        if ($coupon->selection_group != 0) {
            $coupons = Coupon::where('selection_group', '=', $coupon->selection_group)->get();
            foreach ($coupons as $item) {
                if ($item->id != $coupon->id) {
                    Coupon::destroy($item->id);
                }
            }
        }

        $coupon->status = Coupon::STATUS_ACTIVE;
        $coupon->selection = 0;
        $coupon->save();

        return Redirect::to('dashboard/consumer')->with('notification', array(
            'type' => 'success',
            'title' => 'Akzeptiert',
            'description' => 'Der Coupon wurde akzeptiert.'
        ));
    }

    public function reject(Coupon $coupon)
    {
        $coupon->status = 'rejected';
        $coupon->save();

        return Redirect::to('dashboard/consumer')->with('notification', array(
            'type' => 'success',
            'title' => 'Abgelehnt',
            'description' => 'Der RECO.BON wurde abgelehnt.'
        ));
    }

    public function paid(Coupon $coupon)
    {
        if (!in_array($coupon->status, Coupon::STATUS_GROUP_CAN_BE_MARKED_PAID)) {
            return $this->returnBackWithError('Aktion "bezahlt" im aktuellen Status nicht möglich.');
        }

        $coupon->status = Coupon::STATUS_PAID;
        $coupon->save();

        event(new CouponPaid($coupon));

        return $this->returnBackWithSuccess('Der RECO.BON wurde als bezahlt bestätigt.');
    }

    public function assertPayment(Coupon $coupon)
    {
        if ($coupon->status == Coupon::STATUS_REDEEMED) {
            $coupon->status = Coupon::STATUS_PAYMENT_ASSERTED;
            $coupon->save();
        } else {
            return $this->returnBackWithError('Der Coupon hat den falschen Status.');
        }

        $seller = $coupon->template->user;
        $user = Auth::user();

        $url = route('recommended-coupons');
        $seller->sendMail('remind-seller-coupon', compact('coupon', 'seller', 'url', 'user'));

        return $this->returnBackWithSuccess('Die Erinnerung wurde an den Dienstleister versand.');
    }

    public function viewCoupon(Coupon $coupon)
    {
        if ($coupon->consumer_user_id != Auth::user()->id) {
            return Redirect::to('/')->with('notification', array(
                'type' => 'warning',
                'title' => 'Fehler',
                'description' => 'Für diese Aktion nicht berechtigt'
            ));
        }

        return view('coupon.coupon', [
            'coupon' => $coupon,
        ]);
    }

    /**
     * Method to get view of standalone coupon
     *
     * @param Coupon $coupon
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function getCouponView(Coupon $coupon)
    {
        $userId = Auth::id();

        if ( !($userId === $coupon->consumer_user_id || $userId === $coupon->agent_user_id) ) {
            return '<p>Du kannst sehen das RECO.BON nicht</p>';
        }

        return view('redesign.front.listeo.pieces.coupon.ajax', [
            'coupon' => $coupon,
        ]);
    }

    /**
     * Method to get view with coupon for print.
     *
     * @param Coupon $coupon
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function print(Coupon $coupon)
    {
        $userId = Auth::id();

        if ( !($userId === $coupon->consumer_user_id || $userId === $coupon->agent_user_id) ) {
            return '<p>Du kannst sehen das RECO.BON nicht</p>';
        }

        return view('redesign.print.coupon', [
            'coupon' => $coupon,
        ]);
    }

    /**
     * Method to show redeeming page.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function redeemingIndex()
    {
        return view('redesign.back.metronic.coupons.redeem', [
            'code'   => request()->get('code', ''),
        ]);
    }

    /**
     * Method to show seller coupons page.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function sellerIndex()
    {
        $coupons = $this->getSeller()->couponsSharedFromThisSeller()
            ->with('template', 'template.media', 'agent', 'consumer', 'redeemer', 'provisionGift', 'discountGift')
            ->orderBy('id', 'desc')
            ->get();

        return view('redesign.back.metronic.coupons.index', [
            'coupons' => $coupons,
        ]);
    }
}
