<?php

namespace App\Http\Controllers;

use App\Account;
use App\Category;
use App\Coupon;
use App\CouponTemplate;
use App\Http\Requests\LinkRequest;
use App\Services\AbstractRegistrationService;
use App\Services\AgentRegistrationService;
use App\Services\ConsumerRegistrationService;
use App\Services\LinkInvitationService;
use App\Services\ReferralLinkService;
use App\Services\SellerRegistrationService;
use App\Invoice;
use App\Shortlink;
use App\Transaction;
use App\User;
use App\Gift;
use App\LinkInvitations;
use App\UsersGift;
use App\Favorite;
use Illuminate\Foundation\Bus\DispatchesJobs;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Foundation\Auth\Access\AuthorizesResources;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Request;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\View;
use hisorange\BrowserDetect\Facade\Parser as BrowserDetect;

class Controller extends BaseController
{
    use AuthorizesRequests, AuthorizesResources, DispatchesJobs, ValidatesRequests;

    /**
     * @var User
     */
    protected $user;

    public function __construct()
    {
        $bookedBack = false;
        if (!is_null(Auth::user())) {
            if (Auth::user()->hasRole('seller')) {
                $invoiceBookedBack = Invoice::whereStatus(Invoice::STATUS_BOOKED_BACK)
                    ->whereUserId(Auth::user()->id)
                    ->first();
                if (isset($invoiceBookedBack)) {
                    $bookedBack = true;
                }
            }
        }

        View::share('bookedBack', $bookedBack);
        $this->user = Auth::user();
        View::share('user', $this->user);
        View::share('seller', ( $this->user && $this->user->hasRole('employee') ) ? $this->user->boss : $this->user);
        $this->refreshNavigation();
        View::share('incompleteProfile', !is_null($this->user) ? !$this->user->completedProfile() : null);

        $categories = Category::getTop();
        View::share('categories', $categories);

        $userFavorites = Auth::user()
            ? Favorite::where('user_id', Auth::id())->pluck('deal_id')
            : collect([]);
        session(['favs' => $userFavorites]);
        View::share('favs', $userFavorites);

        if ( Auth::user() ) {
            $transactions = Transaction::where('user_id', Auth::id())
                ->where('amount', '>', 0)
                ->get();
            $totals = (object)[
                'paid_out'  => $transactions->whereIn('status', Transaction::STATUS_GROUP_PAID_OUT)->sum('amount'),
                'pending'   => $transactions->where('status', Transaction::STATUS_PENDING)->sum('amount'),
                'allocated' => $transactions->where('status', Transaction::STATUS_ALLOCATED)->sum('amount'),
            ];
            View::share('totals', $totals);

            $refLinks = (new ReferralLinkService())->getAllLinks();
            View::share('refLinks', $refLinks);
        }
    }

    public function refreshNavigation()
    {
        View::share('navigationElements', $this->buildNavigation());
    }

    public function buildNavigation()
    {
        if (!Auth::user()) {
            return [];
        }

        if (!$this->user->checkHasToFillTheProfile()) {
            $result = [
                [
                    'permission' => 'show seller dashboard',
                    'label' => 'Dashboard', // FIXME Rolle in Klammer schreiben, wenn mehrere Rollen zugewiesen sind
                    'url' => url('/dashboard/seller'),
                    'icon' => 'fa-th-large',
                    'id' => 'show_dashboard_seller',
                ],
                [
                    'permission' => 'show agent dashboard',
                    'label' => 'Dashboard', // FIXME Rolle in Klammer schreiben, wenn mehrere Rollen zugewiesen sind
                    'url' => url('/dashboard/agent'),
                    'icon' => 'fa-th-large',
                    'id' => 'show_dashboard_agent',
                ],
                [
                    'permission' => 'show operator dashboard',
                    'label' => 'Dashboard', // FIXME Rolle in Klammer schreiben, wenn mehrere Rollen zugewiesen sind
                    'url' => url('/dashboard/operator'),
                    'icon' => 'fa-th-large',
                    'id' => 'show_dashboard_operator',
                ],
                [
                    'permission' => 'show users',
                    'label' => 'Benutzerverwaltung',
                    'url' => url('/user'),
                    'icon' => 'fa-users',
                    'badge' => User::getPendingUserCount(),
                    'id' => 'show_users',
                ],
                [
                    'permission' => 'show recruited users',
                    'label' => 'Unternehmen eintragen',
                    'url' => url('/user'),
                    'icon' => 'fa-users',
                    'id' => 'show_users',
                ],
                [
                    'permission' => 'recommend recoma',
                    'label' => 'RECO.MA empfehlen',
                    'url' => url('/recommend-recoma'),
                    'icon' => 'fa-users',
                    'id' => 'recommend_recoma',
                ],
                [
                    'permission' => 'show agents',
                    'label' => 'Meine Empfehler',
                    'url' => url('/agent'),
                    'icon' => 'fa-users',
                    'data_step' => '5',
                    'id' => 'show_agents',
                ],
                [
                    'permission' => 'show consumer coupon',
                    'label' => 'RECO.BON',
                    'url' => url('/dashboard/consumer'),
                    'badge' => $this->user->countUnseenCoupons(),
                    'icon' => 'fa-gift',
                    'id' => 'show_consumer_coupons',
                ],
                [
                    'permission' => 'show seller',
                    'label' => 'Unternehmen empfehlen',
                    'url' => url('/sellers'),
                    'icon' => 'fa-users',
                    'id' => 'show_seller',
                ],
                [
                    'permission' => 'show agent coupons',
                    'label' => 'RECO.BON empfehlen',
                    'url' => url('/coupon'),
                    'icon' => 'fa-paper-plane',
                    'id' => 'show_agent_coupons',
                ],
                [
                    'permission' => 'show consumer',
                    'label' => 'Empfohlene BONS',
                    'url' => url('/coupon/redeem'),
                    'icon' => 'fa-gift',
                    'id' => 'show_consumer',
                ],
                [
                    'permission' => 'show coupon management',
                    'label' => 'Meine RECO.BONS',
                    'url' => url('/coupon'),
                    'icon' => 'fa-gift',
                    'id' => 'show_coupon_management',
                ],
                [
                    'permission' => 'redeem coupon',
                    'label' => 'Empfohlene BONS',
                    'url' => url('/coupon/redeem'),
                    'icon' => 'fa-magic',
                    'badge' => count(Auth::user()->couponsSharedFromThisSeller->whereIn('status', Coupon::STATUS_GROUP_CAN_BE_MARKED_PAID)),
                    'id' => 'redeem_coupon',
                ],
                [
                    'permission' => 'redeem coupon',
                    'label' => 'RECO.BON einlösen',
                    'url' => route('seller.coupon.redeeming.index'),
                    'icon' => 'fa-ticket',
                    'id' => 'coupon_redeem',
                ],
                [
                    'permission' => 'show seller transactions',
                    'label' => 'Konto',
                    'url' => url('/transaction'),
                    'icon' => 'fa-credit-card',
                    'id' => 'show_seller_transactions',
                ],
                [
                    'permission' => 'show operator transactions',
                    'label' => 'Transaktionen',
                    'url' => url('/transaction'),
                    'icon' => 'fa-credit-card',
                    'id' => 'show_operator_transactions',
                ],
                [
                    'permission' => 'show operator transactions', // FIXME: own permission: bank transactions
                    'label' => 'Konto-Umsätze',
                    'url' => url('/bank_transaction'),
                    'icon' => 'fa-credit-card',
                    'id' => 'show_operator_transactions',
                ],
                /*
                [
                    'permission' => 'show invitation',
                    'label' => 'Anfragen',
                    'url' => url('/invitation'),
                    'icon' => 'fa-envelope',
                    'badge' => $this->user->getNotificationCount(),
                    'id' => 'show_inviataions',
                ],
                */
                [
                    'permission' => 'show own rating',
                    'label' => 'Meine Bewertungen',
                    'url' => url('/user/rating'),
                    'icon' => 'fa-star-o',
                    'id' => 'show_own_rating',
                ],
                [
                    'permission' => 'show categories',
                    'label' => 'Kategorien',
                    'url' => route('category-list'),
                    'icon' => 'fa-sitemap',
                    'id' => 'show_categories',
                ],
                [
                    'permission' => 'show gifts',
                    'label' => 'Geschenke',
                    'url' => route('gift-list'),
                    'icon' => 'fa-gift',
                    'id' => 'show_gifts',
                ],
                [
                    'permission' => 'show users gifts',
                    'label' => 'Benutzer Geschenke',
                    'url' => route('usersgift-list'),
                    'icon' => 'fa-gift',
                    'id' => 'show_users_gifts',
                    'badge' => UsersGift::where('status', UsersGift::STATUS_READY)->count(),
                ],
                [
                    'permission' => 'show own gifts',
                    'label' => 'Meine Geschenke',
                    'url' => route('owngift-list'),
                    'icon' => 'fa-gift',
                    'id' => 'show_own_gifts',
                    'badge' => $this->user->usersGifts->where('status', UsersGift::STATUS_PAID)->count(),
                ],
                [
                    'permission' => 'show widgets',
                    'label' => 'Widgets',
                    'url' => route('seller.widgets.index'),
                    'icon' => 'fa-code',
                    'id' => 'show_widgets',
                ],
            ];
        } else { // profile not filled
            $result = [];
        }

        if (Auth::user()->can('show consumer transactions') || Auth::user()->can('show agent transactions')) {
            $result[] = [
                'permission' => null,
                'label' => 'Konto',
                'url' => url('/transaction'),
                'icon' => 'fa-credit-card',
                'id' => 'transactions',
            ];
        }

        $result[] = [
            'permission' => null,
            'label' => 'Profil',
            'url' => url('/user/profile'),
            'icon' => 'fa-user',
            'id' => 'profil',
        ];

        if (preg_match('/dashboard\/(seller|agent|consumer)/', Route::getCurrentRoute()->getPath())) {
            $result[] = [
                'permission' => null,
                'label' => 'Hilfe',
                'url' => url('#'),
                'icon' => 'fa-question-circle',
                'id' => 'help',
            ];
        }

        if (Auth::user()->can('edit account')) {
            $result[] = [
                'permission' => 'show account',
                'label' => 'Firmenprofil',
                'url' => url('/account'),
                'icon' => 'fa-user',
                'id' => 'show_account'
            ];
        }
        /*
         * Quick-Temporary Workaround
         * Mitarbeiterverwaltung ausblenden, nur für Operator sichtbar, auch für andere generell verfügbar
         * (aber halt im Menü weg)
         * https://tracker.imi.de/issues/20575#note-12
         */
        if (Auth::user()->hasRole('operator')) {
            $result[] = [
                'permission' => 'edit account data',
                'label' => 'Mitarbeiterverwaltung',
                'url' => url('/member'),
                'icon' => 'fa-user',
            ];
        }
        $result[] = [
            'permission' => 'show job-queue',
            'label' => 'Job Queue',
            'url' => url('/queue'),
            'icon' => 'fa-cogs',
            'badge' => \App\Job::count()
        ];


        // Process menu
        foreach ($result as $key => $element) {
            if ($element['permission'] && !$this->user->can($element['permission'])) {
                unset($result[$key]);
            } else {
                // set current-navigation-item flag
                $result[$key]['current'] = Request::url() == $element['url'];
            }
        }

        return $result;
    }

    protected function getRealHomeUrl()
    {
        return Auth::user() ? Auth::user()->getRealHomeUrl() : url('/login');
    }

    protected function notificationError($message)
    {
        return [
            'type'        => 'error',
            'title'       => 'Fehler',
            'description' => $message
        ];
    }

    protected function notificationSuccess($message)
    {
        return [
            'type'        => 'success',
            'title'       => 'Erfolg',
            'description' => $message
        ];
    }

    /**
     * Return back with error notice
     */
    protected function returnBackWithError($message)
    {
        return Redirect::back()->with('notification', $this->notificationError($message));
    }

    /**
     * Return back with success notice
     */
    protected function returnBackWithSuccess($message)
    {
        return Redirect::back()->with('notification', $this->notificationSuccess($message));
    }

    /**
     * Return back with success notice
     */
    protected function returnHomeWithSuccess($message = 'Daten erfolgreich gespeichert.')
    {
        return Redirect::to($this->getRealHomeUrl())->with('notification', $this->notificationSuccess($message));
    }

    /**
     * Get the numbers for the transaction bar (transaction-status.blade.php)
     *
     * FIXME: Currently gets more than needed!
     * @return array
     */
    protected function getTransactionStatus()
    {
        // General transaction status
        $paidTransactionsAmount = $this->user->transactions()->whereIn('status', Transaction::STATUS_GROUP_PAID_OUT)->sum('amount');
        $pendingTransactionsAmount = $this->user->transactions()->whereStatus(Transaction::STATUS_PENDING)->sum('amount');
        $allocatedTransactionsAmount = $this->user->transactions()->whereStatus(Transaction::STATUS_ALLOCATED)->sum('amount');

        return compact(
            'paidTransactionsAmount',
            'pendingTransactionsAmount',
            'allocatedTransactionsAmount'
        );

    }

    /**
     * @return array
     */
    public function calculateAgentNumbers($withLink = false)
    {
        $couponsMarkedAsRedeemed = Coupon::countMultiBonOnlyOnce(Auth::user()->couponsByAgent()->whereIn('status', Coupon::STATUS_GROUP_FINISHED));
        $couponsSharedFromAgent = Coupon::countMultiBonOnlyOnce(Auth::user()->couponsByAgent());

        $numbers = [
            [
                'name' => 'Empfohlene BONS',
                'number' => $couponsSharedFromAgent,
                'icon' => 'chat',
                'link' => $withLink ? route('coupon-redeem') : null,
            ],
            [
                'name' => 'eingelöste BONS',
                'number' => $couponsMarkedAsRedeemed,
                'icon' => 'handshake',
                'link' => $withLink ? route('coupon-redeem') : null,
            ],
        ];
        return $numbers;
    }

    /**
     * @param $reason
     * @param null $type
     * @param null $id
     * @param null $fromOverlay
     * @return mixed
     */
    public function link($reason, $type = null, $id = null, $fromOverlay = null)
    {
        if ( Auth::user() ) {
            $currentUser = Auth::user();
        } else if ( $token = request('api_token', '') ) {
            $currentUser = User::where('api_token', $token)->first();
        } else {
            $currentUser = null;
        }

        $linkInvitationService = new LinkInvitationService($currentUser);
        if ($reason == 'share-seller') {
            $linkInvitation = $linkInvitationService->create($reason, null, User::find($id)->account->id);
        }
        if ($reason == 'share-coupon') {
            $linkInvitation = $linkInvitationService->create($reason, $id);
        }

        $url = url('/registration/' . $linkInvitation->token);
        $shortLink = Shortlink::shorten($url);

        $text = 'Ich empfehle Dir RECO.MA, Dein persönliches Empfehlungsnetzwerk. Schaue es Dir hier an... ' . $shortLink;

        if ($reason == 'share-seller') {
            // holy crutch // TODO refactor
            if ( $currentUser->hasRole('seller') ) {
                return 'Seller can not recommend self';
            }
            $seller = User::find($id);
            $account = $seller->account;
            $text = 'Ich empfehle Dir ' . $account->getDisplayname() . ', Du bekommst auch Rabatt! Hier Dein RECO.BON ' . $shortLink;
            $whatsAppText = 'Schaue Dir das Angebot von '. $account->getDisplayname(). " auf RECO.MA an. Du erhältst sogar einen Gutschein mit Rabatt. Das Angebot ist begrenzt, also sichere Dir Deinen Gutschein noch heute. 
                \nHier ist Dein persönlicher Gutschein -> ".$shortLink."\n\nPS: Falls Du fragen hast, schreibe mir einfach.";
        }

        if ($reason == 'share-coupon') {
            $text = 'Ich empfehle Dir ' . CouponTemplate::find($id)->seller->account->getDisplayname() . ', Du bekommst auch Rabatt! Hier Dein RECO.BON ' . $shortLink;
            $whatsAppText = 'Schaue Dir das Angebot von '.CouponTemplate::find($id)->seller->account->getDisplayname()." auf RECO.MA an. Du erhältst sogar einen Gutschein mit Rabatt. Das Angebot ist begrenzt, also sichere Dir Deinen Gutschein noch heute. 
                \nHier ist Dein persönlicher Gutschein -> ".$shortLink."\n\nPS: Falls Du fragen hast, schreibe mir einfach.";
        }

        if($reason == 'recommend-recoma'){
            if( $currentUser->hasRole('agent') ){
                $whatsAppText = 'Hallo, ich möchte Ihr Unternehmen auf RECO.MA weiterempfehlen, dem ersten Empfehlungsnetzwerk, bei dem Sie alle Empfehlungen 100% nachvollziehen und messen können.'.
                    "\n\nWerden Sie Teil des innovativsten Netzwerkes -> ". $shortLink."\n\nPS: Falls Sie fragen haben, schreiben Sie mir einfach.";
            }else{
                $whatsAppText = 'RECO.MA ist mein persönlicher Geheimtipp für Dich. Das innovativste Empfehlungsnetzwerk, bei dem Du mehr Kunden für weniger Ausgaben bekommst. '.
                    "\n\nMein persönlicher Einladungslink für Dich -> ". $shortLink."\n\nPS: Falls Du Fragen hast, schreib mir einfach.";
            }
        }

        if($reason == 'invite-new-agent'){
                $whatsAppText = 'Empfehlen Sie unser Unternehmen an Freunde, finden Sie tolle Angebote in Ihrer Nähe und verdienen Sie nebenbei Geld. Das alles und mehr auf RECO.MA. '.
                    "\n\nMelden Sie sich hier an -> ". $shortLink;
        }

        if($fromOverlay!=null){
            return $linkInvitation->token;
        }
        $isMobile = BrowserDetect::isMobile();

        if ($isMobile) {
            $getOperatingSystems = BrowserDetect::detect();
            if ($type == 'sms') {
                if (preg_match('/iOS/', $getOperatingSystems)) {
                    return Redirect::away('sms:&body=' . $text);
                } else {
                    return Redirect::away('sms:?body=' . $text);
                }
            } elseif ($type == 'telegram') {
                $text = rawurlencode($whatsAppText);
                return Redirect::away('tg://msg_url?url=' . $shortLink . '&text=' . $text);
            } elseif ($type == 'whatsapp') {
                $text = rawurlencode($whatsAppText);
                return Redirect::away('whatsapp://send?text=' . $text);
            } else {
                return Redirect::to('share-via-desktop/' . $linkInvitation->token . '/' . $type);
            }
        } else {
            if ($type == 'telegram') {
                $text = rawurlencode($whatsAppText);
                return Redirect::away('https://telegram.me/share/url?url=' . $shortLink . '&text=' . $text);
            } else if ($type == 'whatsapp') {
                //TODO SMS-Function desktop
                $text=rawurlencode($whatsAppText);
                return Redirect::away('http://web.whatsapp.com/send?text=' . $text);
            } else {
                return Redirect::to('share-via-desktop/' . $linkInvitation->token . '/' . $type);
            }
        }
    }

    //notwendig für weiterleitung auf die verschiedenen LPs?
    public function landingPage($token)
    {
        $view = 'user.registration';
        $linkInvitation = LinkInvitations::with('user')->where('token', '=', $token)->first();


        switch ($linkInvitation->reason) {
            case"invite-new-agent":
                $view = 'redesign.auth.user_register_through_invite';
                break;
            case"invite-new-seller":
                $view = 'landingpages.seller.register';
                break;
            case"recommend-recoma":
                $view = 'landingpages.seller.landingpage_recommend_recoma';
                break;
            case "share-seller":
                $view = 'redesign.auth.user_register_through_invite';
                break;
            case "share-coupon":
                $view = 'redesign.auth.user_register_through_invite';
                break;

        }
        return view($view, compact('token','linkInvitation'));
    }

    public function preRegister($token)
    {
        $view = 'user.registration';
        $linkInvitation = LinkInvitations::with('user')->where('token', '=', $token)->first();

        switch ($linkInvitation->reason) {
            case"recommend-recoma":
                $view = 'landingpages.seller.register';
                break;
        }
        return view($view, compact('token','linkInvitation'));
    }

    public function linkInvitation(LinkRequest $request)
    {
        return $this->registration($request->all());
    }

    public function registration($data)
    {
        if ($user = AbstractRegistrationService::emailTaken($data['email'])) {
            return Redirect::to('/register/' . $data['token'])->with('notification', array(
                'type' => 'error',
                'title' => 'Fehler',
                'description' => 'Mail is already taken!'
            ));
        }

        //register user based on invitation, without invitation a seller is using the registrationpage
        if ($linkInvitation = AbstractRegistrationService::checkInvitation($data)) {
            $invitingUser = User::where('id', '=', $linkInvitation->inviting_user_id)->first();
            //Use token to add properties to account & user
            switch ($linkInvitation->reason) {
                case 'recommend-recoma':
                    $registrationService = new SellerRegistrationService($data);
                    $registrationService->setRecruiterFromInvitation($linkInvitation);
                    $user = $registrationService->register();
                    $registrationService->sendDoubleOptInMail($user);
                    //$registrationService->applyInvitationBonus($invitingUser->id, $user->account); //FIXME Muss hinten angestellt werden, Payment-Method not defined etc..., nach #24816
                    break;
                case 'invite-new-agent':
                    $registrationService = new AgentRegistrationService($data);
                    $registrationService->setRecruiterFromInvitation($linkInvitation);
                    $user = $registrationService->register();
                    $registrationService->sendDoubleOptInMail($user);
                    $registrationService->connectAgentWithSeller($invitingUser);
                    break;
                case 'share-seller':
                    $registrationService = new AgentRegistrationService($data);
                    $user = $registrationService->register();
                    $registrationService->sendDoubleOptInMail($user);
                    $this->selectionCoupons($linkInvitation->shared_account_id, $invitingUser->id, $user);

                    $sellerAccount = Account::where('id', '=', $linkInvitation->shared_account_id)->first();
                    $agent = User::where('id', '=', $linkInvitation->inviting_user_id)->first();
                    $registrationService->connectAgentWithSeller($sellerAccount->users->first());
                    $sellerAccount->sendMailToUsers('shared-seller-account-notice', compact('user', 'agent'), true);
                    break;
                case 'share-coupon':
                    $registrationService = new AgentRegistrationService($data);
                    $user = $registrationService->register();
                    $registrationService->sendDoubleOptInMail($user);
                    $this->storeCoupon($linkInvitation->coupon_template_id, $linkInvitation->inviting_user_id, $user);

                    $couponTemplate = CouponTemplate::where('id', '=', $linkInvitation->coupon_template_id)->first();
                    $sellerAccount = $couponTemplate->seller->account;
                    $agent = User::where('id', '=', $linkInvitation->inviting_user_id)->first();
                    $registrationService->connectAgentWithSeller($couponTemplate->seller);
                    $sellerAccount->sendMailToUsers('shared-seller-coupon-notice', compact('user', 'agent', 'couponTemplate'), true);
                    break;
            }
        } else { //switch default can not be called on null
            $registrationService = new SellerRegistrationService($data);
            $user = $registrationService->register();
            $registrationService->sendDoubleOptInMail($user);
        }

        return Redirect::to('/register/thank-you');

    }

    protected function createCoupon($user, $agentId, $couponTemplate, $selection = false, $selection_group = 0, $emailText = null, $smsText = null)
    {
        $date = \Carbon\Carbon::now()->addMonths($couponTemplate['validity']);

        $status = $selection ? Coupon::STATUS_PENDING : Coupon::STATUS_ACTIVE;

        if ($user->referred_by) {
            $referrer = User::find($user->referred_by);

            if (User::REFERRAL_STATUS_ACTIVE === $user->referral_status) {
                $user->referral_status = User::REFERRAL_STATUS_HAS_COUPONS;
                $user->save();
            }

            if (
                CouponTemplate::TYPE_NONE !== $couponTemplate['type_provision'] &&
                !$referrer->hasRole('seller') &&
                !$referrer->hasRole('employee') &&
                User::REFERRAL_STATUS_HAS_REDEEMED !== $user->referral_status &&
                User::REFERRAL_STATUS_PAID !== $user->referral_status
            ) {
                $agentId = $user->referred_by;
            }
        }

        if ($user['id'] !== $agentId) {
            if ( CouponTemplate::TYPE_NONE === $couponTemplate['type_provision'] ) {
                $provision = 0;
            } else if ( CouponTemplate::TYPE_GIFT === $couponTemplate['type_provision'] ) {
                $provision = Gift::find($couponTemplate['provision_gift_id'])->price;
            } else {
                $provision = $couponTemplate['provision_agent'];
            }
        } else {
            $provision = 0;
        }

        if ( CouponTemplate::TYPE_GIFT === $couponTemplate['type_discount'] ) {
            $discount = Gift::find($couponTemplate['discount_gift_id'])->price;
        } else {
            $discount = $couponTemplate['discount_consumer'];
        }

        $coupon = [
            'coupon_template_id' => $couponTemplate['id'],
            'consumer_user_id'   => $user['id'],
            'agent_user_id'      => $agentId,
            'title'              => $couponTemplate['title'],
            'expiration_date'    => $couponTemplate['expiration_date'],
            'description'        => $couponTemplate['description'],
            'provision_agent'    => $provision,
            'discount_consumer'  => $discount,
            'status'             => $status,
            'code'               => $this->generateCode(),
            'selection'          => $selection,
            'selection_group'    => $selection_group,
            'emailtext'          => $emailText,
            'smstext'            => $smsText,
            'validity_date'      => $date,
            'type_provision'     => ($user['id'] !== $agentId) ? $couponTemplate['type_provision'] : CouponTemplate::TYPE_NONE,
            'type_discount'      => $couponTemplate['type_discount'],
            'provision_gift_id'  => ($user['id'] !== $agentId) ? $couponTemplate['provision_gift_id'] : null,
            'discount_gift_id'   => $couponTemplate['discount_gift_id'],
        ];

        return Coupon::create($coupon);
    }

    protected function generateCode($length = 9)
    {
        $characters = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
        $charactersLength = strlen($characters);
        $randomString = '';
        for ($i = 0; $i < $length; $i++) {
            $randomString .= $characters[rand(0, $charactersLength - 1)];
        }

        //FIXME: lädt alle codes -> schlecht. außerdem unnötige recursion statt schleife
        $codes = DB::table('coupons')->select('code')->get();
        if (in_array($randomString, $codes)) {
            return $this->generateCode();
        }

        return $randomString;
    }

    public function storeCoupon($couponTemplateId, $invitingUserId, $user, $emailText = null, $smsText = null)
    {
        $couponTemplate = CouponTemplate::find($couponTemplateId);

        $coupon = $this->createCoupon($user, $invitingUserId, $couponTemplate, 0, 0, $emailText, $emailText);
        $coupon->status = Coupon::STATUS_ACTIVE;
        $coupon->save();

        return $coupon;
    }

    //request must be set if no user is given
    public function selectionCoupons($recommendedId, $invitingUserId, $user = null, $request = null)
    {
        $emailtext = null;
        $smstext = null;
        $account = Account::find($recommendedId);
        $couponTemplates = $account->couponTemplates()->where('coupon_templates.status', '=', 'active')->get();


        if ($user == null) {
            $user = User::where('email', '=', $request->email)->first();
        }

        if ($request != null) {
            $emailtext = $request->emailtext;
            $smstext = $request->smstext;
        }

        $invitingUser = User::find($invitingUserId);

        $this->shareCouponWithUser($user, $invitingUser);

        $coupons = [];
        $selection_group = Coupon::all()->max('selection_group') + 1;
        foreach ($couponTemplates as $couponTemplate) {
            if ($couponTemplate->isShareable()) {
                $coupon = $this->createCoupon($user, $invitingUserId, $couponTemplate, true, $selection_group, $emailtext, $smstext);
                array_push($coupons, $coupon);
            }
        }

        if (count($coupons) == 1) {
            $coupon = $coupons[0];
            $coupon->status = Coupon::STATUS_ACTIVE;
            $coupon->selection = false;
            $coupon->save();

            $coupons = $coupon;
        }
        return $coupons;
    }

    public function checkIfUserExists($request, $user)
    {

        if (!$user instanceof User) {


            $account = Account::create([
                'status' => 'active'
            ]);
            $mobile = '';
            if (isset($request['mobile'])) {
                $mobile = $request['mobile'];
            }
            $email = '';
            if (isset($request['email'])) {
                $email = $request['email'];
            }
            $user = User::create([
                'salutation' => $request['salutation'],
                'firstname' => $request['firstname'],
                'surname' => $request['surname'],
                'mobile' => $mobile,
                'email' => $email,
                'account_id' => $account['id'],
                'status' => 'inactive'
            ]);
            $user->assignRole('agent');
            $user->assignRole('consumer');
            $user->givePermissionTo('edit account');


        }

        return $user;
    }

    public function shareCouponWithUser($user, $agent)
    {
        if ($agent->email == $user->email) {
            return Redirect::to('coupon')->with('notification', array(
                'type' => 'error',
                'title' => 'Fehler',
                'description' => 'Sich selbst RECO.BONS zu vermitteln ist nicht möglich.'
            ));
        }

        if (!$user->hasRole('consumer')) {
            if ($user->hasRole('agent')) {
                $user->assignRole('consumer');
            } else {
                return Redirect::to('coupon')->with('notification', array(
                    'type' => 'error',
                    'title' => 'Fehler',
                    'description' => 'Der eingeladene Benutzer ist bereits registriert, kann aber aktuell keine Empfehlung entgegen nehmen.'
                ));
            }
        }
    }

    public function generatePermanentLinkInvitation(){

        //zusätzliche Sicherheit, falls jemand über die route auf den Controller zugreifen will (der Button ist bereits aus dem Frontend entfernt)
        if(LinkInvitations::where('inviting_user_id',Auth::user()->id)->where('is_permanent_link',true)->count()>0){
            return;
        }

        $linkInvitationService = new LinkInvitationService();
        $linkInvitation = $linkInvitationService->create('invite-new-agent',null,null,true);

        $url = url('/registration/' . $linkInvitation->token);
        $url = Shortlink::shorten($url,true);
        $linkInvitation->save();
        return redirect()->route('massinvitation');
    }

    /**
     * Method to wrap error message.
     *
     * @param $message
     * @return object
     */
    public function _e($message)
    {
        return (object)[
            'type'    => 'error',
            'message' => $message,
        ];
    }

    /**
     * Method to wrap success message.
     *
     * @param $message
     * @return object
     */
    public function _s($message)
    {
        return (object)[
            'type'    => 'success',
            'message' => $message,
        ];
    }

    /**
     * Crutch for getting current seller. If it employee, returns his boss.
     *
     * @return mixed
     */
    public function getSeller()
    {
        $seller = Auth::user();

        if ( $seller->hasRole('employee') ) {
            return $seller->boss;
        }

        return $seller;
    }
}
