<?php

namespace App\Http\Controllers;

use App\Category;
use App\CouponTemplate;
use App\Services\SortingService;
use Baum\MoveNotPossibleException;
use Illuminate\Support\Facades\Auth;
use App\Http\Requests\CategoryRequest;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\File;

class CategoryController extends Controller
{
    /**
     * Method to show categories tree.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index()
    {
        $user = Auth::user();
        $canCreate = $user->hasPermission('create categories');
        $canEdit   = $user->hasPermission('edit categories');
        $canDelete = $user->hasPermission('delete categories');

        return view('categories.index', [
            'tree'      => Category::getTree(true, true),
            'canCreate' => $canCreate,
            'canEdit'   => $canEdit,
            'canDelete' => $canDelete,
        ]);
    }

    /**
     * Method to show new category create form.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create()
    {
        return view('categories.create', [
            'categories' => Category::all(),
        ]);
    }

    /**
     * Method to store new category.
     *
     * @param CategoryRequest $request
     * @return mixed
     */
    public function store(CategoryRequest $request)
    {
        $category = Category::create([
            'name' => $request->get('name'),
            'slug' => $request->get('slug'),
            'icon' => $request->file('icon'),
        ]);

        // attach to some node if it set
        if ($request->get('parent_id') > 0) {
            $parent = Category::find($request->get('parent_id'));
            $category->makeChildOf($parent);
        }

        $category->updateURL();

        return Redirect::to(route('category-list'))->with('notification', array(
            'type'        => 'success',
            'title'       => 'Erfolg',
            'description' => 'Kategorie erfolgreich erstellt.'
        ));
    }

    /**
     * Method to show category edit form.
     *
     * @param Category $category
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit(Category $category)
    {
        return view('categories.edit', [
            'category'   => $category,
            'categories' => Category::all()->except($category->id),
        ]);
    }

    /**
     * Method to update existed category.
     *
     * @param CategoryRequest $request
     * @param Category $category
     * @return mixed
     */
    public function update(CategoryRequest $request, Category $category)
    {
        $category->name = $request->get('name');
        $category->slug = $request->get('slug');

        // icon
        if ( $request->file('icon') ) {
            $category->icon = $request->file('icon');
        } else if ( $category->icon && true === (bool)$request->get('icon_remove') ) {
            File::delete(public_path($category->icon));
            $category->icon = '';
        }

        // saving before moving, because otherwise icon won't be updated
        $category->save();

        // moving node if it set
        $parent_id = (int)$request->get('parent_id') ? (int)$request->get('parent_id') : null;
        if ( $category->parent_id !== $parent_id ) {
            if (null === $parent_id) {
                $category->makeRoot();
            } else {
                $parent = Category::find($request->get('parent_id'));
                $category->makeChildOf($parent);
            }

            $categories = $category->getDescendantsAndSelf();
            foreach ($categories as $category) {
                $category->updateURL();
            }
        }

        return Redirect::to(route('category-list'))->with('notification', array(
            'type'        => 'success',
            'title'       => 'Erfolg',
            'description' => 'Kategorie wurde erfolgreich bearbeitet.'
        ));
    }

    /**
     * Method to remove category.
     *
     * @param Category $category
     * @return mixed
     * @throws \Exception
     */
    public function destroy(Category $category)
    {
        if ( $category->hasCouponTemplates() ) {
            return Redirect::to(route('category-list'))->with('notification', array(
                'type' => 'error',
                'title' => 'Fehler',
                'description' => 'Kategorie mit Coupons kann nicht entfernt werden.'
            ));
        }

        $category->delete();

        return Redirect::to(route('category-list'))->with('notification', array(
            'type'        => 'success',
            'title'       => 'Erfolg',
            'description' => 'Kategorie erfolgreich entfernt.'
        ));
    }

    /**
     * Method to up node in tree.
     *
     * @param Category $category
     * @return mixed
     */
    public function up(Category $category)
    {
        try {
            $category->moveLeft();
        } catch (MoveNotPossibleException $e) {
            // do nothing
        }

        return Redirect::to(route('category-list'));
    }

    /**
     * Method to down node in tree.
     *
     * @param Category $category
     * @return mixed
     */
    public function down(Category $category)
    {
        try {
            $category->moveRight();
        } catch (MoveNotPossibleException $e) {
            // do nothing
        }

        return Redirect::to(route('category-list'));
    }

    /**
     * Method to show category page.
     *
     * @param string $hierarchy
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function category(string $hierarchy)
    {
        $category = Category::where('url', $hierarchy)->first();

        if (!$category) {
            abort(404);
        }

        return view('redesign.front.category.index', [
            'top'           => CouponTemplate::available()->top()->limit(10)->get(),
            'deals'         => $category->getAvailableDeals(),
            'category'      => $category,
            'sortings'      => (new SortingService())->getSortings(),
            'breadcrumbs'   => $category->getBreadcrumbs(),
            'subcategories' => $category->getImmediateDescendants(),
        ]);
    }
}
