<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\SetPasswordRequest;
use App\Http\Requests\PasswordRecoveryRequest;
use Newsletter;
use Illuminate\Foundation\Auth\ResetsPasswords;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Str;

class PasswordController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Password Reset Controller
    |--------------------------------------------------------------------------
    |
    | This controller is responsible for handling password reset requests
    | and uses a simple trait to include this behavior. You're free to
    | explore this trait and override any methods you wish to tweak.
    |
    */

    use ResetsPasswords {
        reset as traitReset;
    }

    protected $linkRequestView = 'redesign.auth.passwords.email';
    protected $resetView       = 'redesign.auth.passwords.reset';

    protected function getSendResetLinkEmailSuccessResponse($response)
    {
        if (auth::user() == null) {
            return view("redesign.auth.passwords.requested");
        } else {
            return redirect()->back()->with(['status' => trans($response), 'notification' => $this->notificationSuccess('Passwort Zurücksetzen Link erfolgreich versendet.')]);
        }
    }

    /**
     * Where to redirect users after password reset.
     *
     * @var string
     */
    protected $redirectTo = '/';

    /**
     * Create a new password controller instance.
     *
     * @return void
     */
    use ResetsPasswords;

    public function __construct()
    {
        $this->subject = 'Ihr Passwort zurücksetzen Link';
    }

    public function setUserPassword(SetPasswordRequest $request)
    {
        $result = $this->traitReset($request);

        $user = Auth::user();

        $data = $request->all();
        if ( $user->hasRole('seller') ) {
            $account = $user->account;
            $account->payment = $data['payment'];
            $account->save();
            $this->getAndSaveProvision($data['user_id'], $user->account);
        }

        if($result != null){
            $user->status = 'active';
            $user->email_verified = true;
            $user->save();
        }

        // TODO raus
        if ( isset($data['newsletter']) && $data['newsletter'] == true ) {
            Newsletter::subscribe($data['email']);
        }

        return $result;
    }

    public function sendResetLinkEmail(PasswordRecoveryRequest $request)
    {
        $data = [];
        $this->validateSendResetLinkEmail($request);

        $broker = $this->getBroker();
        $broker = \Password::broker($broker);

        $user = $broker->getUser($this->getSendResetLinkEmailCredentials($request));
        $token = $broker->createToken($user);
        $data['url'] = url('password/reset' . '/' . $token);
        $data['emailtext'] = 'empty';
        $data['token'] = $token;
        $data['user'] = $user;
        $template = 'reset-password';
        $user->sendMail($template, $data, true);

        activity($user)->register('password reset requested');

        return $this->getSendResetLinkEmailSuccessResponse(Password::RESET_LINK_SENT);
    }

    /**
     * Reset the given user's password (overriden).
     *
     * @param  \Illuminate\Contracts\Auth\CanResetPassword  $user
     * @param  string  $password
     * @return void
     */
    protected function resetPassword($user, $password)
    {
        $user->forceFill([
            'password' => bcrypt($password),
            'remember_token' => Str::random(60),
        ])->save();

        activity($user)->register('password reset success');

        Auth::guard($this->getGuard())->login($user);
    }
}