<?php

namespace App\Http\Controllers\Auth;

use App\Account;
use App\AccountConnection;
use App\CouponTemplate;
use App\Http\Requests\SellerRegistrationRequest;
use App\Services\AbstractRegistrationService;
use App\Services\MetaTagService;
use App\Services\SellerRegistrationService;
use App\Services\SellerService;
use App\User;
use Illuminate\Support\Facades\Session;
use Laravel\Socialite\Facades\Socialite;
use Laravel\Socialite\Two\InvalidStateException;
use Newsletter;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Redirect;
use Validator;
use App\Http\Controllers\Controller;
use Illuminate\Foundation\Auth\ThrottlesLogins;
use Illuminate\Foundation\Auth\AuthenticatesAndRegistersUsers;
use App\Http\Requests\UserRegisterRequest;
use App\Http\Requests\SellerRegisterRequest;
use App\Services\AgentRegistrationService;

class AuthController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Registration & Login Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users, as well as the
    | authentication of existing users. By default, this controller uses
    | a simple trait to add these behaviors. Why don't you explore it?
    |
    */

    use AuthenticatesAndRegistersUsers, ThrottlesLogins;

    /**
     * Where to redirect users after login / registration.
     *
     * @var string
     */
    protected $redirectTo = '/';

    /**
     * Create a new authentication controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware($this->guestMiddleware(), ['except' => ['logout', 'verifyEmail']]);
    }

    /**
     * Get a validator for an incoming registration request.
     *
     * @param  array $data
     * @return \Illuminate\Contracts\Validation\Validator
     */
    protected function validator(array $data)
    {
        return Validator::make($data, [
            'salutation' => 'required|max:255',
            'firstname' => 'required|max:255',
            'surname' => 'required|max:255',
            'company' => 'required|max:255',
            'payment' => 'required|max:255',
            'email' => 'required|email|max:255|unique:users',
            'password' => 'required|min:6|confirmed',
        ]);
    }

    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array  $data
     * @return User
     */
    protected function create(array $data)
    {
        $account = Account::create([
            'iban' => $data['iban'],
            'iban_owner' => $data['iban_owner'],
            'company' => $data['company'],
            'street' => $data['street'],
            'zip' => $data['zip'],
            'city' => $data['city'],
            'tax_id' => $data['tax_id'],
            'payment' => $data['payment'],
            'housenumber' => $data['housenumber'],
            'all_connections' => $data['all_connections'],
            'status' => 'active'

        ]);

        $user = User::create([
            'account_id' => $account['id'],
            'salutation' => $data['salutation'],
            'firstname' => $data['firstname'],
            'surname' => $data['surname'],
            'email' => $data['email'],
            'phone' => $data['phone'],
            'status' => 'active',
            'password' => bcrypt($data['password']),

        ]);

        $user->assignRole('seller');
        $user->assignRole('seller-active');
        $user->givePermissionTo('edit account');
        $user->givePermissionTo('edit account data');
        $url = url('/');

        $user->sendMail('confirm-registration-to-seller', compact('url'), true);

        return $user;
    }

    public function register(SellerRegistrationRequest $request)
    {
        $this->registration($request->all());

        return redirect('/register/thank-you');
    }

    public function showRegistrationForm()
    {
        return redirect()->to(route('auth.register'));

        if (property_exists($this, 'registerView')) {
            return view($this->registerView);
        }

        return view('landingpages.seller.register');
    }

    /**
     * Method to show user registration form.
     *
     * @param Request $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function showUserRegistrationForm(Request $request)
    {
        $deal     = $request->get('d') ? CouponTemplate::find($request->get('d')) : null;
        $seller   = $request->get('s') ? User::find($request->get('s')) : null;
        $referrer = $request->get('r') ? User::find($request->get('r')) : null;

        if ( $referrer && $referrer->hasRole('seller') ) {
            $deal = (new SellerService($seller ? $seller : $referrer))->getBestDeal('provision_agent');
        } else if ( $referrer && $referrer->hasRole('employee') ) {
            if ( !$referrer->canDo('share reflink') ) {
                return redirect(url('login'));
            }

            $seller = $referrer->boss;
            $deal = (new SellerService($seller ? $seller : $referrer))->getBestDeal('provision_agent');
        } else if ( $seller ) {
            $deal = (new SellerService($seller ? $seller : $referrer))->getBestDeal();
        }

        if ( null === $seller && null !== $deal ) {
            $seller = $deal->seller;
        }

        $meta = MetaTagService::userRegistrationForm($referrer, $seller, $deal);

        return view('redesign.auth.user_register', [
            'meta'     => $meta,
            'deal'     => $deal,
            'seller'   => $seller,
            'referrer' => $referrer,
        ]);
    }

    /**
     * Method to seller user registration form.
     *
     * @param Request $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function showSellerRegistrationForm(Request $request)
    {
        $referrer = $request->get('r') ? User::find($request->get('r')) : null;

        $meta = MetaTagService::sellerRegistrationForm($referrer);

        return view('redesign.auth.seller_register', [
            'meta'     => $meta,
            'referrer' => $referrer,
        ]);
    }

    /**
     * Method to register new user.
     *
     * @param UserRegisterRequest $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function userRegister(UserRegisterRequest $request)
    {
        $data = $request->all();

        $registrationService = new AgentRegistrationService($data);
        $user = $registrationService->register();
        $registrationService->sendDoubleOptInMail($user);

        if ( isset($data['referred_by']) ) {
            $referrer = User::where('id', (int)$data['referred_by'])->first();

            if ( isset($referrer) ) {
                if ( $referrer->hasRole('seller') ) { // if referrer is a seller, attach
                    $registrationService->connectAgentWithSeller($referrer);
                } else if ( $referrer->hasRole('employee') ) { // get referrer's boss and attach
                    $seller = $referrer->boss;
                    $registrationService->connectAgentWithSeller($seller);

                    activity($referrer)->register('agent invited', null, null, User::class, $user->id);
                } else { // get referrer's seller and attach to new user
                    $connection = AccountConnection::where('account_id', $referrer->account->id)
                        ->where('type', 'agent')
                        ->first();
                    if ($connection) {
                        $seller = User::where('account_id', $connection->related_account_id)->first();
                        $registrationService->connectAgentWithSeller($seller);
                    }
                }
            }
        }

        Auth::login($user);

        return redirect(route('home'));
    }

    /**
     * Method to register new seller.
     *
     * @param SellerRegisterRequest $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function sellerRegister(SellerRegisterRequest $request)
    {
        $data = $request->all();

        $registrationService = new SellerRegistrationService($data);
        $seller = $registrationService->register(true);
        $registrationService->sendDoubleOptInMail($seller);

        if ( isset($data['invited_by']) ) {
            $inviter = User::where('id', (int)$data['invited_by'])->first();

            if ( null !== $inviter) {
                activity($inviter)->register('seller invited', null, null, User::class, $seller->id);
            }
        }

        Auth::login($seller);

        return redirect(route('home'));
    }

    public function showSetPasswordForConsumerForm(Request $request, $token = null)
    {
        $this->middleware('guest');
        $selectionDummy = false;

        $email = $request->input('email');

        $coupon = User::findByToken($token)->coupons()->first();
        if ($coupon->selection_group != 0) {
            $selectionDummy = true;
        }
        return view('user/set-consumer')->with(compact('token', 'email', 'coupon', 'selectionDummy'));
    }

    public function showSetPasswordForAgentForm(Request $request, $token = null)
    {
        $this->middleware('guest');

        User::findByToken($token);
        $email = $request->input('email');

        return view('user/set-agent')->with(compact('token', 'email'));
    }

    /*
     * $user = inviter
     */
    public function showSetPasswordForSellerForm(Request $request, $user, $token = null)
    {
        $this->middleware('guest');

        User::findByToken($token);
        $email = $request->input('email');

        return view('user/set-seller')->with(compact('token', 'email', 'user'));
    }

    public function showLoginForm($invitationToken = -1){
        return view('redesign.auth.login', ['token' => $invitationToken]);
    }

    public function loginHandler(Request $request)
    {
        $this->validateLogin($request);

        if ($request->token != -1) {
            return $this->loginWithToken($request);
        } else {
            return $this->login($request);
        }
    }

    //FIXME Draft of login with token
    public function loginWithToken($request)
    {
        $credentials = $this->getCredentials($request);

        if (Auth::guard($this->getGuard())->attempt($credentials, $request->has('remember'))) {
            $user = Auth::user();
            $data = $request->all();
            if ($linkInvitation = AbstractRegistrationService::checkInvitation($data)) {
                $invitingUser = User::where('id', '=', $linkInvitation->inviting_user_id)->first();
                //Use token to add properties to account & user
                switch ($linkInvitation->reason) {
                    case 'share-seller':
                        $user->assignRole('agent');
                        $user->assignRole('consumer');
                        $this->selectionCoupons($linkInvitation->shared_account_id, $invitingUser->id, $user);
                        break;
                    case 'share-coupon':
                        $user->assignRole('agent');
                        $user->assignRole('consumer');
                        $this->storeCoupon($linkInvitation->coupon_template_id, $user);
                        break;
                }
            }
            Auth::login($user);
            return Redirect::to(url('/'));
        }
        return $this->sendFailedLoginResponse($request);
    }

    /**
     * Get the needed authorization credentials from the request (overriden).
     *
     * @param  \Illuminate\Http\Request  $request
     * @return array
     */
    protected function getCredentials(Request $request)
    {
        // only active user can be logged in
        return [
            'email'    => $request->input('email'),
            'password' => $request->input('password'),
            'status'   => User::STATUS_ACTIVE,
        ];
    }

    public function verifyEmail($id, $token)
    {
        $user = User::withoutGlobalScopes()->find($id);

        $redirectRoute = route( $user->hasRole('seller') ? 'seller.dashboard' : 'user.dashboard' );

        if (!empty($token) && $user->email_verification_token == $token) {
            $user->email_verified = true;
            $user->email_verification_token = null;
            if ($user->email_temp) {
                $user->email = $user->email_temp;
                $user->email_temp = '';
            }
            $user->email_valid = $user->email; // save last valid email
            $user->save();

            if ( null === Auth::user() ) {
                Auth::login($user);
            }

            return redirect($redirectRoute)->with('notification', $this->_s('E-Mail verifiziert.'));
        }

        return redirect($redirectRoute)->with('notification', $this->_e('E-Mail Verifizierung fehlgeschlagen.'));
    }


    public function handleEasyRegistrationType(Request $request){
        $token = $request->get('token');
        $this->validate(request(), [
            'password' => ['required','min:6','confirmed']
        ]);

        $this->validate(request(), [
            'email' => ['required','email','max:255','unique:users']
        ]);
        $this->registration($this->getDataArray($request->get('email'), $token,$request->get('password')));
        return Redirect::to('/register/thank-you');
        /*
        switch($request->registration){
            case User::REGISTER_BY_EMAIL:
                $this->validate(request(), [
                    'email' => ['required','email','max:255','unique:users']
                ]);
                $this->registration($this->getDataArray($request->get('email'), $token,$request->get('password')));
                return Redirect::to('/register/thank-you');
            break;
            case User::REGISTER_BY_FACEBOOK:
                Session::put('linkInvitationToken', $token);
                Session::put('password', $request->get('password'));
                return Socialite::driver('facebook')->redirect();
            break;
        }
        */
    }


    /**
     * Obtain the user information from facebook.
     *
     * @return Response
     */
    public function handleProviderCallback(Request $request)
    {

        if ($request->error == User::ACCESS_DENIED) {
            return $this->facebookLoginFailed();
        }

        try{
            $user = Socialite::driver('facebook')->user();
        }catch(InvalidStateException $e){
            return $this->facebookLoginFailed();
        }

        if($user){
            if(User::findByEmail($user->email)){
                 return Redirect::to('/registration/' . Session::get('linkInvitationToken'))->withErrors([
                     $this->loginUsername() => 'Ihre E-Mail-Adresse ist bereits vergeben.'
                 ]);
            }else{
                $this->registration($this->getDataArray($user->email, Session::get('linkInvitationToken'),Session::get('password')));
                Session::forget('linkInvitationToken');
                Session::forget('password');
                return Redirect::to('/register/thank-you');
            }
        }
        return $this->facebookLoginFailed();
    }

//FIXME: Data-Arrays durch die Gegend schieben, ist nicht optimal. Bessere Möglichkeit finden
    public function getDataArray($email,$token,$password)
    {

        $data=[
            'token'=>$token,
            'email'=>$email,
            'firstname'=>'',
            'surname'=>'',
            'zip'=>'',
            'password'=>$password,
            'password_confirmation'=>$password
        ];
        return $data;
    }


    public function facebookLoginFailed(){
        return Redirect::to('/registration/' . Session::get('linkInvitationToken'))->withErrors([
            $this->loginUsername() => 'Login mit facebook ist fehlgeschlagen.'
        ]);
    }

    /**
     * Method to register user through a referrer.
     *
     * @param string $alias
     * @param Request $request
     * @return mixed
     */
    public function registerWithReferrer(string $alias, Request $request)
    {
        $referrer = User::where('alias', $alias)->first();

        if ( !$referrer ) {
            return redirect(route('home'));
        }

        $add = '';

        if ( $request->get('s') ) {
            $add = '&s=' . $request->get('s');
        }

        if ( $request->get('d') ) {
            $add = '&d=' . $request->get('d');
        }

        return redirect()->to('/registration?r=' . $referrer->id . $add);
    }

    /**
     * Method to register seller through a referrer.
     *
     * @param string $alias
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function registerSellerWithReferrer(string $alias)
    {
        $referrer = User::where('alias', $alias)->first();

        if ( !$referrer ) {
            return redirect(route('home'));
        }

        return redirect()->to('/registration-seller?r=' . $referrer->id);
    }
}
