<?php

namespace App\Http\Controllers;

use App\Account;
use App\AccountConnection;
use App\Coupon;
use App\Http\Grids\Seller\AbstractAgentGrid;
use App\Http\Grids\Seller\AgentGrid;
use App\Http\Grids\Seller\MyAgentsGrid;
use App\Invitation;
use App\Library\Facades\Format;
use App\User;
use App\UserConnection;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Auth\Passwords\DatabaseTokenRepository;
use Input;
use Mockery\CountValidator\Exception;


class AgentController extends Controller
{
    /**
     * The password token repository.
     *
     * @var \Illuminate\Auth\Passwords\TokenRepositoryInterface
     */
    protected $tokens;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware(['auth', 'completed.profile']);
    }


    public function index()
    {
        $couponsCount = 0;
        $couponsDoneCount = 0;
        $conversionRate = 0;
        $activeAgents = 0;

        $myAgents = $this->user->getAgentsQuery()->get();
        foreach ($myAgents as $myAgent) {
            $coupons = $myAgent->couponsByAgent()->count();
            $couponsCount += $coupons;

            $couponsDone = $myAgent->couponsByAgent()->whereStatus(Coupon::STATUS_DONE)->count();
            $couponsDoneCount += $couponsDone;

            if ($coupons != 0) {
                $activeAgents += 1;
            }
        }

        if ($couponsCount != 0 && $couponsDoneCount != 0) {
            $conversionRate = 100 / $couponsCount * $couponsDoneCount;
        }


        $numbers = [
            [
                'name' => 'Erfolgsrate',
                'number' => Format::percent($conversionRate),
                'link' => null,
                'icon' => 'startup'
            ],
            [
                'name' => 'Aktive Empfehler',
                'number' => $activeAgents,
                'link' => url('/agent/create'),
                'icon' => 'user'
            ],
        ];


        $myAgents = new MyAgentsGrid();
        $agents = new AgentGrid();
        $globalGrid = new AbstractAgentGrid();
        $globalGrid->exportGlobalFilters([$myAgents, $agents]);


        return view('agent/index', [
            'globalFilters' => $globalGrid->getGlobalFilters(),
            'my_agents' => $myAgents,
            'agents' => $agents,
            'conversionRate' => $conversionRate,
            'couponsCount' => $couponsCount,
            'couponsDoneCount' => $couponsDoneCount,
            'activeAgents' => $activeAgents,
            'countAllAgents' => count($myAgents),
            'numbers' => $numbers,
        ]);
    }

    public function create()
    {
        return view('agent/create');
    }

    public function getTokenService()
    {
        $tokens = App::make('password.resets.tokens');
        return $tokens;
    }

    public function store(Request $request)
    {
        Input::merge(array_map(function ($value) {
            if (is_string($value)) {
                return trim($value);
            } else {
                return $value;
            }
        }, Input::all()));

        $data = $request->all();

        $user = User::findByEmail($data['email']);
        $seller = Auth::user();

        /**
         * @var $account Account
         */
        $account = Auth::user()->account;


        if ($user instanceof User) {
            if(!$user->hasRole('agent')){
                $user->assignRole('agent');
                $user->assignRole('consumer');
            }

            return $this->add($user->account);
        } else {
            $newAccount = Account::create([
                'status' => 'active'
            ]);
            $data['account_id'] = $newAccount['id'];
            $data['status'] = 'inactive';
            $user = User::create($data);

            $user->assignRole('agent');
            $user->assignRole('consumer');
            $user->givePermissionTo('edit account');

            AccountConnection::create([
                'account_id' => $newAccount->id,
                'related_account_id' => Auth::user()->account->id,
                'type' => 'agent',
                'status' => 'active'
            ]);

            $tokens = $this->getTokenService();
            $token = $tokens->create($user);
            $url = url('password/set/agent', $token) . '?email=' . urlencode($user->email);

            $user->sendMail('seller-invites-agent', compact('account', 'seller', 'url'));
        }

        return redirect('agent');
    }

    public function registerAgent()
    {
        return view('agent/register');
    }

    public function remove(Account $account)
    {
        Auth::user()->account->removeSellerAccount()->detach($account);

        return Redirect::to('agent')->with('notification', array(
            'type' => 'success',
            'title' => 'Erfolg',
            'description' => 'Der Empfehler wurde aus ihrer Liste entfernt.'
        ));
    }

    public function add(Account $agentAccount)
    {
        $sellerAccount = Auth::user()->account;
        Invitation::invite($sellerAccount, Invitation::TO_AGENT, $agentAccount);

        $url = url('/');
        $agentAccount->sendMailToUsers('agent-invite', compact('sellerAccount', 'url'));

        return Redirect::to('agent')->with('notification', array(
            'type' => 'success',
            'title' => 'Erfolg',
            'description' => 'Eine Anfrage wurde an den Empfehler versendet.'
        ));
    }

    public function removeSellerAccount(Account $account)
    {
        Auth::user()->account->removeAgentAccount()->detach($account);

        return $this->returnBackWithSuccess('Der Dienstleister wurde aus ihrer Liste entfernt.');
    }

    public function addSellerAccount(Account $sellerAccount)
    {
        if ($sellerAccount->all_connections == 1) {
            AccountConnection::create([
                'account_id' => Auth::user()->account->id,
                'related_account_id' => $sellerAccount->id,
                'type' => 'agent',
                'status' => 'active'
            ]);

            return $this->returnBackWithSuccess('Der Dienstleister wird nun von ihnen gemerkt.');
        }


        $agentAccount = Auth::user()->account;
        Invitation::invite($agentAccount, Invitation::AS_AGENT, $sellerAccount);

        $url = url('/');
        $sellerAccount->sendMailToUsers('seller-invitation', compact('agentAccount', 'url')
        );

        return $this->returnBackWithSuccess('Eine Anfrage wurde an den Dienstleister versendet.');
    }
}
