<?php

namespace App\Http\Controllers;

use App\Account;
use App\Http\Grids\MemberGrid;
use App\Http\Requests\AccountRequest;
use App\Http\Requests\MemberRequest;
use App\User;
use Illuminate\Foundation\Auth\ResetsPasswords;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Redirect;


class AccountController extends Controller
{

    use ResetsPasswords;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware(['auth']);
        if (!in_array(
            $this->getRouter()->getCurrentRoute()->getAction()['controller'],
            [
                'App\Http\Controllers\AccountController@index',
                'App\Http\Controllers\AccountController@store'
            ])) {

            $this->middleware('completed.profile');
        }
    }

    public function index()
    {
        return view('account/index');
    }

    public function store(AccountRequest $request)
    {
        $data = $request->all();
        $account = Auth::user()->account;

        if (isset($data['portrait'])) {
            $this->validate($request, ['profile' => 'mimes:jpeg,png,gif|max:256']);

            if ($account['upload_portrait'] != '') {
                unlink(storage_path('app/portraits/') . $account['upload_portrait']);
            }

            $file = $request->file('portrait');

            $name = preg_replace('/[^[:alnum:].]/', '', $file->getClientOriginalName());
            $fileName = $data['id'] . '_' . $name;

            $account->upload_portrait = $fileName;

            $file->move(storage_path('app/portraits'), $fileName);
        }

        $account->fill($data);
        $account->save();

        if (Auth::user()->hasRole('consumer') || Auth::user()->hasRole('agent')) {
            return Redirect::to('/user/profile')->with('notification', array(
                'type' => 'success',
                'title' => 'Erfolg',
                'description' => 'Daten erfolgreich gespeichert.'
            ));
        }
        return Redirect::to('/account')->with('notification', array(
            'type' => 'success',
            'title' => 'Erfolg',
            'description' => 'Daten erfolgreich gespeichert.'
        ));

    }

    public function getImage($upload_portrait)
    {

        parent::__construct();
        $path = storage_path('app/portraits/') . $upload_portrait;

        return response()->download($path, null, [], 'inline');
    }

    public function showMembers()
    {
        return view('account.member', [
            'grid' => new MemberGrid
        ]);
    }

    public function getTokenService()
    {
        $tokens = App::make('password.resets.tokens');
        return $tokens;
    }

    public function inviteMember(MemberRequest $request)
    {
        $data = $request->all();

        $account = Auth::user()->account;
        $member = [
            'email' => $data['email'],
            'account_id' => $account['id'],
            'password' => bcrypt($this->generateRandomString()),
            'firstname' => ' ',
            'surname' => ' ',
            'status' => 'pending'
        ];
        $member = User::create($member);

        if (Auth::user()->hasRole('operator')) {
            $member->assignRole('operator');
        }
        if (Auth::user()->hasRole('seller')) {
            $member->assignRole('seller');
        }
        if (Auth::user()->hasRole('agent')) {
            $member->assignRole('agent');
            $member->assignRole('consumer');
        }

        $tokens = $this->getTokenService();
        $token = $tokens->create($member);
        $url = url('password/set/agent', $token) . '?email=' . urlencode($member->email);
        $inviteingUser = Auth::user();

        $member->sendMail('anybody-invites-user',
            compact('member', 'account', 'url', 'inviteingUser')
        );

        return Redirect::to('/member')->with('notification', array(
            'type' => 'success',
            'title' => 'Erfolg',
            'description' => 'Der User wurde eingeladen.'
        ));
    }

    function generateRandomString($length = 15)
    {
        $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $charactersLength = strlen($characters);
        $randomString = '';
        for ($i = 0; $i < $length; $i++) {
            $randomString .= $characters[rand(0, $charactersLength - 1)];
        }
        return $randomString;
    }
}
